<?php

/* -------------------------------------------------------------------------
	ClientDetect class
	a part of PC2M Web Content Converter for Mobile Clients
	Copyright (C) 2005-2006 ucb.rcdtokyo
	http://www.rcdtokyo.com/pc2m/note/
------------------------------------------------------------------------- */

class ClientDetect {

	/**
	 * @var		string
	 * @access	public
	 */
	var $dev_type = 'mozilla';

	/**
	 * @var		string
	 * @access	public
	 */
	var $dev_ver = null;

	/**
	 * @var		string
	 * @access	public
	 */
	var $dev_name = null;

	/**
	 * @var		array
	 * @access	private
	 */
	var $dev_info = array();

	/**
	 * @var		string
	 * @access	public
	 */
	var $serial_id = null;

	/**
	 * @var		string
	 * @access	public
	 */
	var $hostname = null;

	/**
	 * @var		bool
	 * @access	public
	 */
	var $is_mobile = false;

	/**
	 * @var		int
	 * @access	public
	 */
	var $cache_size;

	/**
	 * @var		array
	 * @access	public
	 */
	var $screen;

	/**
	 * @var		bool
	 * @access	public
	 */
	var $is_postcapable = true;

	/**
	 * Constructor
	 *
	 * @access public
	 * @param  string User-Agent
	 */
	function ClientDetect($_ua = null) {

		if ($_ua === null and isset($_SERVER['HTTP_USER_AGENT'])) {
			$_ua = $_SERVER['HTTP_USER_AGENT'];
		}

		// DoCoMo
		if (preg_match('/^(DoCoMo\/[\d\.]+)[\s\/](.+)/', $_ua, $matches)) {
			// DoCoMo/1.0/[model]/[cache]/[mode]/[chars]
			$this->dev_type = 'docomo';
			$this->dev_ver = $matches[1];
			$this->dev_info = explode('/', $matches[2]);
			// DoCoMo/2.0 [model]([cache];[mode];[chars])
			if (preg_match('/^([^\(]+)\((.+)\)/', $this->dev_info[0], $matches)) {
				$this->dev_name = $matches[1];
				$this->dev_info = explode(';', $matches[2]);
			} else {
				$this->dev_name = array_shift($this->dev_info);
			}
			// Returns the minimum spec if the UA value does not contain cache size
			$this->cache_size = (isset($this->dev_info[0])
				and preg_match('/^c(\d{1,})$/', $this->dev_info[0], $matches))?
				$matches[1] * 1000:
				5000;
			// mova: ser[serial number (11 digit alphanumeric)]
			// FOMA: icc[FOMA card serial number (20 digits alphanumeric)]
			foreach ($this->dev_info as $value) {
				if (preg_match('/^(?:ser\w{11}|icc\w{20})$/', $value)) {
					$this->serial_id = $value;
				}
			}

		// au WAP 2.0
		// KDDI-[model] UP.Browser/[version] (GUI) [server]
		} elseif (preg_match('/^KDDI-([^\s]+)\s(UP\.Browser\/[\d\.]+)/', $_ua, $matches)) {
			$this->dev_type = 'au';
			$this->dev_ver = $matches[2];
			$this->dev_name = $matches[1];
			// [vendor id (4 digit numeric][terminal id (10 digit numeric)]_[UP.Link server]
			if (isset($_SERVER['HTTP_X_UP_SUBNO'])) {
				$this->serial_id = $_SERVER['HTTP_X_UP_SUBNO'];
			}

		// au HDML
		// UP.Browser/[version]-[model] [server]
		} elseif (preg_match('/^(UP\.Browser\/[\d\.]+)[^-]*-([^\s]+)/', $_ua, $matches)) {
			$this->dev_type = 'au';
			$this->dev_ver = $matches[1];
			$this->dev_name = $matches[2];
			// [vendor id (4 digit numeric][terminal id (10 digit numeric)]_[UP.Link server]
			if (isset($_SERVER['HTTP_X_UP_SUBNO'])) {
				$this->serial_id = $_SERVER['HTTP_X_UP_SUBNO'];
			}

		// Vodafone/J-PHONE
		// Note: some Vodafone devices have 'UP.Browser' string in the middle of UA
		} elseif (preg_match('/^((?:J-PHONE|Vodafone)\/[\d\.]+)\/([^\s\/]+)(.+)$/', $_ua, $matches)) {
			$this->dev_type = 'vodafone';
			$this->dev_ver = $matches[1];
			$this->dev_name = $matches[2];
			$this->dev_info = explode(' ', $matches[3]);
			// Serial ID format is supposed to be SN[4 digit alphabet][7 digit numeric]
			// But there seems no official statement about the number of digits.
			if (isset($this->dev_info[0])) {
				$array = explode('/', $this->dev_info[0]);
				foreach ($array as $value) {
					if (preg_match('/^SN\w+$/', $value)) {
						$this->serial_id = $value;
						break;
					}
				}
			}
			if ($this->dev_ver == 'J-PHONE/2.0') {
				$this->is_postcapable = false;
			}

		// Some Vodafone devices do not have 'Vodafone/' or 'J-PHONE/' in UA,
		// and UA is blank for some earlier devices
		// (where is the serial number for these devices?)
		} elseif (isset($_SERVER['HTTP_X_JPHONE_MSNAME'])) {
			$this->dev_type = 'vodafone';
			$this->dev_ver = $_ua? 'Vodafone/1.0': 'J-PHONE/2.0';
			$this->dev_name = $_SERVER['HTTP_X_JPHONE_MSNAME'];

		// Mozilla
		} elseif (preg_match('/^(Mozilla\/[\d\.]+)\s?\((.+)\)\s?(.*)/', $_ua, $matches)) {
			$this->dev_type = 'mozilla';
			$this->dev_ver = $matches[1];
			$this->dev_info = explode(';', $matches[2]);

			// DoCoMo PCSV
			// Mozilla/x.x ([model];FOMA;[cache];[mode];[chars])
			if (isset($this->dev_info[1]) and $this->dev_info[1] == 'FOMA') {
				$this->dev_type = 'docomo_pcsv';
				$this->dev_name = $this->dev_info[0];
				$this->cache_size = (isset($this->dev_info[2])
					and preg_match('/^c(\d{1,})$/', $this->dev_info[2], $matches))?
					$matches[1] * 1000:
					5000;

			// au PCSV
			// Mozilla/4.0 (compatible; MSIE 6.0; KDDI-[model]) [browser]
			} elseif (isset($this->dev_info[2])
				and preg_match('/^KDDI-(.+)$/', trim($this->dev_info[2]), $matches)) {
				$this->dev_type = 'au_pcsv';
				$this->dev_name = $matches[1];

			// Willcom PCSV - Browser's native mode is disregarded and detected as "Mozilla"
			// Mozilla/3.0([WILLCOM|DDIPOCKET];[vendor]/[model]/[not fixed]/[cache]) [browser]
			} elseif (isset($this->dev_info[0])
				and ($this->dev_info[0] == 'WILLCOM' or $this->dev_info[0] == 'DDIPOCKET')) {
				$this->dev_type = 'willcom_pcsv';
				if (isset($this->dev_info[1])) {
					// Number of the section in the UA's comment field is not fixed.
					// So we merge each section in a new array at first.
					$array = explode('/', $this->dev_info[1]);
					if (count($this->dev_info) > 2) {
						for ($i = 2; $i < count($this->dev_info); $i++) {
							$array = array_merge($array, explode('/', $this->dev_info[$i]));
						}
					}
					if (isset($array[1])) {
						$this->dev_name = $array[1];
						$array = array_reverse($array);
						foreach ($array as $value) {
							if (preg_match('/^C(\d{1,})$/', $value, $matches)) {
								$this->cache_size = $matches[1] * 1000;
								break;
							}
						}
					}
				}
			}
		}
		$this->getScreenInfo();
		$this->getHostname();
	}

	/**
	 * This method returns client's serial number.
	 *
	 * DoCoMo mova:      ser[serial number (11 digit alphanumeric)]
	 * DoCoMo FOMA:      icc[FOMA card serial number (20 digits alphanumeric)]
	 * EZweb:            [vendor id (4 digit numeric][terminal id (10 digit numeric)]_[UP.Link server]
	 * Vodafone/J-Phone: SN[4 digit alphabet][7 digit numeric]
	 *
	 * @access public
	 * @return string
	 */
	function getSerialId() {

		return $this->serial_id;
	}

	/**
	 * This method returns client's available memory.
	 *
	 * @access public
	 * @return int
	 */
	function getCacheSize() {

		if ($this->cache_size === null) {
			switch ($this->dev_type) {
				case 'docomo':
				case 'docomo_pcsv':
					// Returns the minimum spec if the UA value does not contain cache size
					$this->cache_size = (isset($this->dev_info[0])
						and preg_match('/^c(\d{1,})$/', $this->dev_info[0], $matches))?
						$matches[1] * 1000:
						5000;
					break;
				case 'au':
				case 'au_pcsv':
					// Returns the minimum spec if the server variable is unavailable
					$this->cache_size = isset($_SERVER['HTTP_X_UP_DEVCAP_MAX_PDU'])?
						$_SERVER['HTTP_X_UP_DEVCAP_MAX_PDU']:
						7500;
					break;
				case 'vodafone':
					// Returns 6KB if packet incompatible ('J-PHONE/2.0' and 'J-PHONE/3.0'), otherwise 12KB
					$this->cache_size = ($this->dev_ver != 'J-PHONE/2.0' and $this->dev_ver != 'J-PHONE/3.0')?
						12000:
						6000;
					break;
				case 'mozilla':
					$this->cache_size = 100000;
					break;
				default:
					$this->cache_size = 5000;
			}
		}
		return $this->cache_size;
	}

	/**
	 * This method returns client's screen information.
	 *
	 * @access public
	 * @return array (width|color|gif_support|jpeg_support|png_support)
	 */
	function getScreenInfo() {

		if ($this->screen === null) {
			switch ($this->dev_type) {
				case 'docomo':
				case 'docomo_pcsv':
					include 'DevInfo_DoCoMo.inc.php';
					if (isset($_dev_info[$this->dev_name])) {
						$this->screen = array(
							"width" => $_dev_info[$this->dev_name][0],
							"color" => $_dev_info[$this->dev_name][1],
							"gif_support" => $_dev_info[$this->dev_name][2],
							"jpg_support" => $_dev_info[$this->dev_name][3],
							"png_support" => $_dev_info[$this->dev_name][4],
						);
					// If the model name is not in the list, it is supposed to be a new one.
					// So we return average spec of 'recent' devices.
					} else {
						$this->screen = array(
							"width" => 240,
							"color" => 65536,
							"gif_support" => true,
							"jpg_support" => true,
							"png_support" => false,
						);
					}
					break;
				case 'au':
				case 'au_pcsv':
					// If the server variable is unavailable,
					// the minimum spec of known color device will be applied.
					$_screen_size = isset($_SERVER['HTTP_X_UP_DEVCAP_SCREENPIXELS'])?
						explode(',', $_SERVER['HTTP_X_UP_DEVCAP_SCREENPIXELS']):
						array(96, null);
					$_screen_depth = isset($_SERVER['HTTP_X_UP_DEVCAP_SCREENDEPTH'])?
						explode(',', $_SERVER['HTTP_X_UP_DEVCAP_SCREENDEPTH']):
						array(8, null);
					$_http_accept = isset($_SERVER['HTTP_ACCEPT'])?
						$_SERVER['HTTP_ACCEPT']:
						null;
					$this->screen = array(
						"width" => (int) $_screen_size[0],
						"color" => pow(2, (int) $_screen_depth[0]),
						"gif_support" => preg_match('/image\/gif/', $_http_accept) ? true : false,
						"jpg_support" => preg_match('/image\/jpeg/', $_http_accept) ? true : false,
						"png_support" => true,
					);
					break;
				case 'vodafone':
					// If the server variable is unavailable,
					// the minimum spec of known color device will be applied.
					$_screen_size = isset($_SERVER['X-JPHONE-DISPLAY'])?
						explode('*', $_SERVER['X-JPHONE-DISPLAY']):
						array(96, null);
					$_screen_depth = isset($_SERVER['X-JPHONE-COLOR'])?
						(int)preg_replace('/^[C|G](\d+)$/',"$1",$_SERVER['X-JPHONE-COLOR']):
						256;
					$_http_accept = isset($_SERVER['HTTP_ACCEPT'])?
						$_SERVER['HTTP_ACCEPT']:
						null;
					$this->screen = array(
						"width" => $_screen_size[0],
						"color" => $_screen_depth,
						// Devices of 'J-PHONE/' are incompatible with GIF.
						"gif_support" => preg_match('/^J-PHONE\//', $this->dev_ver)? false: true,
						// Devices of 'J-PHONE/2.0' are incompatible with JPEG.
						"jpg_support" => ($this->dev_ver == 'J-PHONE/2.0')? false: true,
						"png_support" => true,
					);
					break;
				case 'willcom_pcsv':
				case 'mozilla':
					$this->screen = array(
						"width" => 240,
						"color" => 65536,
						"gif_support" => true,
						"jpg_support" => true,
						"png_support" => true,
					);
					break;
				default:
					$this->screen = array(
						"width" => 96,
						"color" => 256,
						"gif_support" => true,
						"jpg_support" => true,
						"png_support" => true,
					);
			}
		}
		return $this->screen;
	}

	/**
	 * This method checks if the client is from mobile operators' network
	 * and returns domain suffix for valid mobile clients.
	 * REMOTE_HOST or REMOTE_ADDR is returned as it is for the others.
	 * There are two ways to detect if the client is from valid network.
	 * One is to check if the hostname contains valid domain suffix.
	 * Anotehr is to check if the IP address in the valid network range.
	 * This method takes hostname prior to IP address, and calls getHostByAddr
	 * at first to resolve the client's hostname when REMOTE_HOST is not available.
	 * If your DNS resolver does not work for the function
	 * or if you do not like the DNS resolution overhead,
	 * you may comment out the line including getHostByAddr.
	 *
	 * @access public
	 * @return string
	 */
	function getHostname() {

		if ($this->hostname === null) {
			$_remote_host = null;
			if (isset($_SERVER['REMOTE_HOST']) and $_SERVER['REMOTE_HOST']) {
				$_remote_host = $_SERVER['REMOTE_HOST'];
			} elseif (isset($_SERVER['REMOTE_ADDR']) and $_SERVER['REMOTE_ADDR']) {
				$_remote_host = getHostByAddr($_SERVER['REMOTE_ADDR']);
			}
			if (preg_match('/^.+(\.(?:docomo|ezweb|brew|jp-[ckqt]|prin|mopera)\.ne\.jp)$/', $_remote_host, $matches)) {
				$this->hostname = $matches[1];
				$this->is_mobile = true;

			// Check if the IP address is in the network address range
			// if DNS resolver does not work for getHostByAddr.
			// Mopera is not detected in this case.
			} elseif (preg_match('/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/', $_remote_host)
				and false !== $x = $this->checkIpRange($_remote_host)) {
				$this->is_mobile = true;
				switch ($x) {
					case 'docomo':
						$this->hostname = '.docomo.ne.jp';
						break;
					case 'au':
						$this->hostname = '.ezweb.ne.jp';
						break;
					case 'vodafone':
						$this->hostname = '.jp-t.ne.jp';
						break;
					case 'willcom':
						$this->hostname = '.prin.ne.jp';
						break;
				}
			} else {
				$this->hostname = $_remote_host;
			}
		}
		return $this->hostname;
	}

	/**
	 * This method returns an abbreviated name of the mobile operator
	 * when the IP address is in their network address range.
	 * Otherwise returns false.
	 * This method can be called statically,
	 * but is usually called from getHostname method
	 * only when DNS resolver does not work for getHostByAddr.
	 *
	 * @access public
	 * @param string $_address An IP address
	 * @return string (docomo|au|vodafone|willcom)
	 */
	function checkIpRange($_address) {
		require 'IPRange.inc.php';
		$_address = $this->_dumpAddress($_address);
		if ($this->_compareIp($_address, $_iprange_docomo)) {
			return 'docomo';
		} elseif ($this->_compareIp($_address, $_iprange_au)) {
			return 'au';
		} elseif ($this->_compareIp($_address, $_iprange_vodafone)) {
			return 'vodafone';
		} elseif ($this->_compareIp($_address, $_iprange_willcom)) {
			return 'willcom';
		}
		return false;
	}

	/**
	 * Private method used by checkIpRange method
	 *
	 * @access private
	 * @param string $_address An IP address
	 * @param array $_iprange An array of network addresses
	 * @return bool
	 */
	function _compareIp($_address, $_iprange) {
		foreach ($_iprange as $value) {
			list($_network, $_mask) = explode('/', $value);
			$_network = $this->_dumpAddress($_network);
			$_mask = $this->_dumpNetmask($_mask);
			if (($_address & $_mask) == ($_network & $_mask)) {
				return true;
				break;
			}
		}
		return false;
	}

	/**
	 * Private method used by checkIpRange method
	 *
	 * @access private
	 * @param string $_mask A netmask by bit unit
	 * @return binary digit
	 */
	function _dumpNetmask($_mask) {
		$i = 0;
		$x = '';
		while ($i < $_mask) {
			$x .= '1';
			$i++;
		}
		while ($i < 32) {
			$x .= '0';
			$i++;
		}
		$array = array();
		$array[] = bindec(substr($x, 0, 8));
		$array[] = bindec(substr($x, 8, 8));
		$array[] = bindec(substr($x, 16, 8));
		$array[] = bindec(substr($x, 24, 8));
		return ($array[0] << 24) | ($array[1] << 16) | ($array[2] << 8) | $array[3];
	}

	/**
	 * Private method used by checkIpRange method
	 *
	 * @access private
	 * @param string $_address An IP address
	 * @return binary digit
	 */
	function _dumpAddress($_address) {
		$array = explode('.', $_address);
		return ($array[0] << 24) | ($array[1] << 16) | ($array[2] << 8) | $array[3];
	}
}

?>
