// ------------------------------------------------
// File : proxy.cpp
// Date: 16-may-2003
// Author: dreamer12345
// Desc: 
//
// (c) 2002 dreamer12345
// ------------------------------------------------
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// ------------------------------------------------

#include "proxy.h"
#include "servmgr.h"

ProxySocket::ProxySocket(ClientSocket *socket) {
	this->socket = socket;
}
// ---------------------------------
void ProxySocket::open(Host &rh) {
	host = rh;

	if (servMgr->proxyType && servMgr->proxyHost[0]) {
		if (servMgr->proxyPort <= -1) {
			switch (servMgr->proxyType) {
				case PROXY_HTTP11:
					servMgr->proxyPort = 8080;
					break;
				case PROXY_SOCKS4:
				case PROXY_SOCKS4A:
				case PROXY_SOCKS5:
					servMgr->proxyPort = 1080;
					break;
				default:
					throw SockException("Invalid proxy settings");
			}
		}

		Host proxy;
		
		// First try to see if we have the IP before resolving the enventual domain name...
		proxy.fromStrIP(servMgr->proxyHost, servMgr->proxyPort);
		if (proxy.ip == 0)
			proxy.fromStrName(servMgr->proxyHost, servMgr->proxyPort);

		socket->open(proxy);
	}
	else
		socket->open(host);

}
// ---------------------------------
void ProxySocket::bind(Host &rg) {
	socket->bind(rg);
	host = rg;
}
// ---------------------------------
void ProxySocket::connect() {
	switch (servMgr->proxyType) {
		case PROXY_HTTP11:
			initHttp11();
			break;

		case PROXY_SOCKS4:
		case PROXY_SOCKS4A:
			initSocks4();
			break;

		case PROXY_SOCKS5:
			initSocks5();
			break;

		case PROXY_NOPROXY:
		default:
			socket->connect();
			break;

	}
}
// ---------------------------------
bool ProxySocket::active() {
	return socket->active();
}
// ---------------------------------
ClientSocket * ProxySocket::accept() {
	return socket->accept();
}
// ---------------------------------
int ProxySocket::readPending() {
	return socket->readPending();
}
// ---------------------------------
Host ProxySocket::getLocalHost() {
	return socket->getLocalHost();
}
// ---------------------------------
void ProxySocket::write(const void *p, int l) {
	socket->write(p, l);
}
// ---------------------------------
int ProxySocket::read(void *p, int l) {
	return socket->read(p, l);
}
// ---------------------------------
void ProxySocket::close() {
	socket->close();
}
// ---------------------------------
bool ProxySocket::isValid() {
	if (servMgr->proxyType > 0 && servMgr->proxyType < 5 && servMgr->proxyHost[0])
		return true;

	return false;
}
// ---------------------------------
bool ProxySocket::canTranslateDN() {
	return (servMgr->proxyType != PROXY_SOCKS4);
}


void ProxySocket::initHttp11() {
	LOG_DEBUG("Initiating HTTP/1.1 Connection (%s:%d). Authentication: %s", servMgr->proxyHost, servMgr->proxyPort, servMgr->proxyAuthenticate?"Yes":"No");

	socket->connect();

	char connectHost[256];
	connectHost[0] = 0;
	connectHost[255] = 0;

	if (host.ip != 0)
		host.IPtoStr(connectHost);
	else
		strncpy(connectHost, host.name, 256);

	socket->writeString("CONNECT %s:%d HTTP/1.1\r\n", connectHost, host.port);
	socket->writeString("Host: %s:%d\r\n", connectHost, host.port);

	if (servMgr->proxyAuthenticate) {
		char loginAndPass[512];
		memset(loginAndPass, 0, 512);
		strncpy(loginAndPass, servMgr->proxyLogin, 255);
		strcat(loginAndPass, ":");
		strncat(loginAndPass, servMgr->proxyPass, 255);

		char *base64 = util_base64_encode(loginAndPass);

		socket->writeString("Authorization: Basic %s\r\n", base64);
		socket->writeString("Proxy-Authorization: Basic %s\r\n", base64);
	}

	socket->writeString("\r\n");

	char rcvLine[256], buffer[256], *rcvTok;
	socket->readLine(rcvLine, 256);

	// We should have something like that:
	//  HTTP/1.1 200 Connection established
	//  We'll just verify there's an HTTP and a 200 :)
	strcpy(buffer, rcvLine);

	rcvTok = strtok(buffer, "/");
	if (!rcvTok || strcmp(rcvTok, "HTTP") != 0)
		throw SockException("HTTP/1.1 proxy error: Error initiating connection");

	strcpy(buffer, rcvLine);

	rcvTok = strtok(buffer, " ");
	rcvTok = strtok(0, " ");

	if (!rcvTok || strcmp(rcvTok, "200") != 0)
		throw SockException("HTTP/1.1 proxy error: Error initiating connection");

	// Let's extract the header...
	while (rcvLine[0])
		socket->readLine(rcvLine, 256);

	// That's all, connection is ready :)
}

// ---------------------------------
void ProxySocket::initSocks4() {
	LOG_DEBUG("Initiating SOCKS4/4a Connection (%s:%d).", servMgr->proxyHost, servMgr->proxyPort);

	char ver, status;

	socket->connect();

	// SOCKSv4/4a Init
	//
	// +----+----+----+----+----+----+----+----+----+----+....+----+
	// | VN | CD | DSTPORT |      DSTIP        | USERID       |NULL|
	// +----+----+----+----+----+----+----+----+----+----+....+----+
	//    1    1      2              4           variable       1
	socket->writeChar(4);						// Version
	socket->writeChar(1);						// 1=Connect

	socket->writeChar(host.port >> 8);			// Port
	socket->writeChar(host.port & 0xff);

	if (host.ip != 0) {
		socket->writeChar(host.ip >> 24);		// IPv4 Address
		socket->writeChar((host.ip >> 16) & 0xff);
		socket->writeChar((host.ip >> 8) & 0xff);
		socket->writeChar(host.ip & 0xff);

		socket->writeChar(0);					// No USERID
	}
	else if (servMgr->proxyType == PROXY_SOCKS4A) {
		// We have to supply the domain name...
		// We must send this IP: 0.0.0.x (where x is a non-zero number) and attach the
		// domain name just after the USERID...
		socket->writeChar(0);					// 0.0.0.x
		socket->writeChar(0);
		socket->writeChar(0);
		socket->writeChar(27);

		socket->writeChar(0);					// No USERID

		socket->writeString(host.name);			// Null-terminated domain name
		socket->writeChar(0);
	}
	else
		throw SockException("SOCKS4 proxy error: Can't resolve hostname trough SOCKS4 proxy");

	// SOCKSv4/4a Response
	//
	// +----+----+----+----+----+----+----+----+
	// | VN | CD | DSTPORT |      DSTIP        |
	// +----+----+----+----+----+----+----+----+
	//    1    1      2              4
	ver = socket->readChar();
	status = socket->readChar();

	socket->readChar();	// Port
	socket->readChar();
	socket->readChar();	// IPv4 Address
	socket->readChar();
	socket->readChar();
	socket->readChar();

	if (ver != 0)
		throw SockException("SOCKS4/a proxy error: Error initiating connection");

	switch (status) {
		case 90:
			break;
		case 91:
			throw SockException("SOCKS4/4a proxy error: Request rejected");
		case 92:
			throw SockException("SOCKS4/4a proxy error: SOCKS server cannot connect to identd on the client");
		case 93:
			throw SockException("SOCKS4/4a proxy error: Sent user-id and identd user-id are different!");
		default:
			throw SockException("SOCKS4/4a proxy error: Unknown error");
	}
}

// ---------------------------------
void ProxySocket::initSocks5() {
	LOG_DEBUG("Initiating SOCKS5 Connection (%s:%d). Authentication: %s", servMgr->proxyHost, servMgr->proxyPort, servMgr->proxyAuthenticate?"Yes":"No");

	char ver, method, status;

	socket->connect();

	// SOCKSv5 Init
	//
	// +----+----------+----------+
    // |VER | NMETHODS | METHODS  |
    // +----+----------+----------+
    // | 1  |    1     | 1 to 255 |
    // +----+----------+----------+
	socket->writeChar(5); // Version
	socket->writeChar(servMgr->proxyAuthenticate?2:1); // Number of logon types

	if (servMgr->proxyAuthenticate)
		socket->writeChar(2); // 2=user/pass

	socket->writeChar(0); // 0=no logon

	// SOCKSv5 Init Response
	//
	// +----+--------+
    // |VER | METHOD |
    // +----+--------+
    // | 1  |   1    |
    // +----+--------+
	ver = socket->readChar();
	method = socket->readChar();

	if (ver != 5)
		throw SockException("SOCKS5 proxy error: Error initiating connection");

	switch (method) {
		case 0:
			break;
		case 2:
			{
				// SOCKSv5 : subnegotiation protocol v1
				socket->writeChar(1); // v1
				socket->writeChar(strlen(servMgr->proxyLogin)); // length of username
				socket->writeString(servMgr->proxyLogin);
				socket->writeChar(strlen(servMgr->proxyPass)); // length of password
				socket->writeString(servMgr->proxyPass);

				ver = socket->readChar();
				status = socket->readChar();

				if (ver != 1)
					throw SockException("SOCKS5 proxy error: Error requesting authentification");
				
				if (status != 0)
					throw SockException("SOCKS5 proxy error: Bad login/password");

				break;
			}
		default:
			throw SockException("SOCKS5 proxy error: Authentification method refused");
	}

	// SOCKSv5 Request
	//
	// +----+-----+-------+------+----------+----------+
    // |VER | CMD |  RSV  | ATYP | DST.ADDR | DST.PORT |
    // +----+-----+-------+------+----------+----------+
    // | 1  |  1  | X'00' |  1   | Variable |    2     |
    // +----+-----+-------+------+----------+----------+
	SOCKS5Request socks;
	socks.ver = 5;
	socks.cmd = 1;
	socks.rsv = 0;	// Reserved
	socks.atyp = host.ip==0?3:1; // 1=IPv4 - 3=Domain name - 4=IPv6

	socket->write(&socks, 4);

	if (host.ip == 0) {
		socket->writeChar(strlen(host.name));
		socket->writeString(host.name);
	}
	else {
		// Send IP Adress
		socket->writeChar(host.ip >> 24);
		socket->writeChar((host.ip >> 16) & 0xff);
		socket->writeChar((host.ip >> 8) & 0xff);
		socket->writeChar(host.ip & 0xff);
	}

	// Send port number
	socket->writeChar(host.port >> 8);
	socket->writeChar(host.port & 0xff);

	// SOCKSv5 Response
	//
	// +----+-----+-------+------+----------+----------+
    // |VER | REP |  RSV  | ATYP | BND.ADDR | BND.PORT |
    // +----+-----+-------+------+----------+----------+
    // | 1  |  1  | X'00' |  1   | Variable |    2     |
    // +----+-----+-------+------+----------+----------+
	socket->read(&socks, 4);

	if (socks.ver != 5)
		throw SockException("SOCKS5 proxy error: Bad response from SOCKS server");

	switch (socks.cmd) {
		case 0:
			break;
		case 1:
			throw SockException("SOCKS5 proxy error: General SOCKS server failure");
			break;
		case 2:
			throw SockException("SOCKS5 proxy error: Connection not allowed by ruleset");
			break;
		case 3:
			throw SockException("SOCKS5 proxy error: Network unreachable");
			break;
		case 4:
			throw SockException("SOCKS5 proxy error: Host unreachable");
			break;
		case 5:
			throw SockException("SOCKS5 proxy error: Connection refused");
			break;
		case 6:
			throw SockException("SOCKS5 proxy error: TTL expired");
			break;
		case 7:
			throw SockException("SOCKS5 proxy error: Command not supported");
			break;
		case 8:
			throw SockException("SOCKS5 proxy error: Address type not supported");
			break;
		default:
			throw SockException("SOCKS5 proxy error: Unknown error");
	}

	switch (socks.atyp) {
		int i;
		case 1:
			socket->readLong();		// BND.ADDR (IPv4)
			socket->readShort();	// BND.PORT
			break;

		case 4:
			// BND.ADDR (IPv6)
			for (i = 0 ; i < 4 ; i++)
				socket->readLong();

			socket->readShort();	// BND.PORT
			break;

		case 3:
			// BND.ADDR (Domain Name)
			//  1st byte: len of domain name
			for (i = 0 ; i < socket->readChar() ; i++)
				socket->readChar();

			socket->readShort();	// BND.PORT
			break;
	}
}
