#
# Fuse compatible interface for Win32::Dokan
#

package Win32::Dokan::Fuse;

use strict;
use warnings;

use base qw(Win32::Dokan);
use Win32::Dokan::Const qw(-compile :attribute :iomode :error);

use Encode;
use POSIX;
use Carp;

sub new {
    my $class = shift;
    my %opts = @_;

    my $self = $class->SUPER::new({fs_encoding => 'utf-8',
				   convert_path => 1});

    $self->{_fuse_opts} = \%opts;
    $self->{_fuse_open_count} = {};

    # $self->fs_encoding('utf-8');

    return $self;
}

BEGIN {
    my @callback = qw (getattr readlink getdir mknod mkdir unlink
		       rmdir symlink rename link chmod chown truncate
		       utime open read write statfs flush release fsync
		       setxattr getxattr listxattr removexattr);
    no strict "refs";
    for my $m (@callback) {
	my $method = __PACKAGE__ . "::fuse_$m";
	*$method = sub {
	    my $self = shift;
	    my $sub = $self->{_fuse_opts}->{$m};
	    &$sub(@_);
	}
    }
}
		    
sub cb_create_file {
    my $self = shift;
    my ($fileName, $desiredAccess, $shareMode, $createDisposition, $flagsAndAttributes, $DFileInfo) = @_;

    # Fuse mknod
    # Fuse open

    my $ret;
    my $mode = 0;

    if ($DFileInfo->is_directory) {
	return 0;
    }
    if ($fileName =~ /[\\\/]\.\.?$/) {
	$DFileInfo->is_directory(1);
	return 0;
    }

    my @stat = $self->fuse_getattr($fileName);
    if ($stat[2] && POSIX::S_ISDIR($stat[2])) {
	$DFileInfo->is_directory(1);
	return 0;
    }

    if ($desiredAccess & Win32::Dokan::Const::GENERIC_READ
	&& $desiredAccess & Win32::Dokan::Const::GENERIC_WRITE) {
	$mode = O_RDWR;
    }
    elsif ($desiredAccess & Win32::Dokan::Const::GENERIC_READ) {
	$mode = O_RDONLY;
    }
    elsif ($desiredAccess & Win32::Dokan::Const::GENERIC_WRITE) {
	$mode = O_WRONLY;
    }

    if ($createDisposition == Win32::Dokan::Const::CREATE_NEW) {
	if (defined($stat[2])) {
	    $ret = -1*Win32::Dokan::Const::ERROR_ALREADY_EXISTS;
	}
	else {
	    my $mknod = $self->fuse_mknod($fileName, 0644, 0);
	    if ($mknod >= 0) {
		$ret = $self->fuse_open($fileName, $mode);
		if ($ret >= 0) {
		    my $key = "$mode,$fileName";
		    $DFileInfo->context($key);
		    $self->{_fuse_open_count}->{$key}++;
		    $ret = 0;
		}
	    }
	    else {
		$ret = -1;
	    }
	}
    }
    elsif ($createDisposition == Win32::Dokan::Const::CREATE_ALWAYS
	   || $createDisposition == Win32::Dokan::Const::OPEN_ALWAYS) {
	my $key = "$mode,$fileName";
	my $found;

	if ($stat[2]) {
	    $found = 1;
	    $ret = 0;

	    $ret = $self->fuse_open($fileName, $mode);
	    if ($ret >= 0
		&& $createDisposition == Win32::Dokan::Const::CREATE_ALWAYS) {
		$ret = $self->fuse_truncate($fileName, 0);
		unless ($ret >= 0) {
		    $self->fuse_release($fileName, $mode)
			unless ($self->{_fuse_open_count}->{$key});
		}
	    }

	    if ($ret >= 0) {
		$DFileInfo->context($key);
		$self->{_fuse_open_count}->{$key}++;
		$ret = 0;
	    }
	}
	else {
	    my $mknod = $self->fuse_mknod($fileName, 0644, 0);
	    if ($mknod >= 0) {
		$ret = $self->fuse_open($fileName, $mode);
		if ($ret >= 0) {
		    $DFileInfo->context($key);
		    $self->{_fuse_open_count}->{$key}++;
		    $ret = 0;
		}
	    }
	    else {
		$ret = -1;
	    }
	}
	$ret = Win32::Dokan::Const::ERROR_ALREADY_EXISTS if ($found && $ret == 0);
    }
    elsif ($createDisposition == Win32::Dokan::Const::OPEN_EXISTING) {
	if ($stat[2]) {
	    $ret = $self->fuse_open($fileName, $mode);
	    if ($ret >= 0) {
		my $key = "$mode,$fileName";
		$DFileInfo->context($key);
		$self->{_fuse_open_count}->{$key}++;
		$ret = 0;
	    }
	}
	else {
	    $ret = -2;
	}
    }
    elsif ($createDisposition == Win32::Dokan::Const::TRUNCATE_EXISTING) {
	if ($stat[2]) {
	    if ($self->fuse_open($fileName, $mode) >= 0) {
		$ret = $self->fuse_truncate($fileName, 0);
		my $key = "$mode,$fileName";

		if ($ret >= 0) {
		    $DFileInfo->context($key);
		    $self->{_fuse_open_count}->{$key}++;
		}
		else {
		    $self->fuse_release($fileName, $mode)
			unless ($self->{_fuse_open_count}->{$key});
		}
	    }
	    else {
		$ret = -2;
	    }
	}
	else {
	    $ret = -2;
	}
    }
    else {
	print STDERR "****************** open error!\n";
    }

    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub cb_open_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    my @s = $self->fuse_getattr($pathName);
    my $ret = ($s[2] && POSIX::S_ISDIR($s[2])) ? 0 : $s[0];

    return -1 unless (defined($ret));
    return $ret >= 0 ? 0 : $ret;
}

sub cb_create_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;
 
    my $ret = $self->use_mkdir($pathName, 0755);

    return $ret >= 0 ? 0 : $ret;
}

sub cb_cleanup {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    my $key = $DFileInfo->context;

    if (defined($key) && $self->{_fuse_open_count}->{$key}) {
	if (--$self->{_fuse_open_count}->{$key} == 0) {
	    delete $self->{_fuse_open_count}->{$fileName};
	    my $mode = ($key =~ /^(0-9]),/) ? $1 : 0;
	    $self->fuse_release($fileName, $mode);
	}
    }

    return 0;
}

sub cb_close_file {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    return 0;
}

sub cb_read_file {
    my $self = shift;
    my ($fileName, $dummy, $offset, $length, $DFileInfo) = @_;

    my $data = $self->fuse_read($fileName, $length, $offset);

    # numeric error value will be here
    # return -1
    $_[1] = $data;

    return 0;
}

sub cb_write_file {
    my $self = shift;
    my ($fileName, $data, $offset, $dummy, $DFileInfo) = @_;

    my $len = $self->fuse_write($fileName, $data, $offset);
    return $len if ($len < 0);
    $_[3] = $len;

    return 0;
}

sub cb_flush_file_buffers {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    return $self->fsync($fileName, 1);
}

sub cb_get_file_information {
    my $self = shift;
    my ($fileName, $fileInfo, $DFileInfo) = @_;

    # Fuse getattr
    my @s = $self->fuse_getattr($fileName);
    if (@s > 1) {
	my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
	    $atime,$mtime,$ctime,$blksize,$blocks) = @s;

	$fileInfo->file_size(int($size));
	if (POSIX::S_ISDIR($mode)) {
	    $fileInfo
		->file_attributes(Win32::Dokan::Const::FILE_ATTRIBUTE_DIRECTORY);
	}
	else {
	    $fileInfo
		->file_attributes(Win32::Dokan::Const::FILE_ATTRIBUTE_NORMAL);
	}
	$fileInfo->creation_time($ctime);
	$fileInfo->last_access_time($atime);
	$fileInfo->last_write_time($mtime);

	return 0;
    }
    else {
	if ($fileName =~ /[\\\/]\.\.?$/) {
	    $fileInfo
		->file_attributes(Win32::Dokan::Const::FILE_ATTRIBUTE_DIRECTORY);
	    
	}
	my $t = time;
	$fileInfo->creation_time($t);
	$fileInfo->last_access_time($t);
	$fileInfo->last_write_time($t);

	return 0;
    }

    return (defined($s[0]) && $s[0] < 0) ? $s[0] : -1;
}

sub cb_find_files {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    my @elements = $self->fuse_getdir($pathName);
    my $ret = pop(@elements);

    my $has_dot;
    my $has_dotdot;

    my @ret;
    for my $e (@elements) {
	my $find_data = Win32::Dokan::FindDataW->new();

	my @s = $self->fuse_getattr("$pathName/$e");
	if (@s > 1) {
	    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
		$atime,$mtime,$ctime,$blksize,$blocks) = @s;

	    $find_data->file_size(int($size));
	    if (POSIX::S_ISDIR($mode)) {
		$find_data->file_attributes(Win32::Dokan::Const::FILE_ATTRIBUTE_DIRECTORY);
	    }
	    else {
		$find_data->file_attributes(Win32::Dokan::Const::FILE_ATTRIBUTE_NORMAL);
	    }
	    $find_data->creation_time($ctime);
	    $find_data->last_access_time($atime);
	    $find_data->last_write_time($mtime);
	    $find_data->file_name($e);
	    $find_data->number_of_links($nlink);

	    push(@ret, $find_data);

	    $has_dot = 1 if ($e eq '.');
	    $has_dotdot = 1 if ($e eq '..');
	}
    }

    unless ($has_dotdot) {
	my $t = time;
	my $find_data = Win32::Dokan::FindDataW->new();
	$find_data->file_attributes(Win32::Dokan::Const::FILE_ATTRIBUTE_DIRECTORY);
	$find_data->creation_time($t);
	$find_data->last_access_time($t);
	$find_data->last_write_time($t);
	$find_data->file_name('..');
	$find_data->number_of_links(2);

	unshift(@ret, $find_data);
    }
    unless ($has_dot) {
	my $t = time;
	my $find_data = Win32::Dokan::FindDataW->new();
	$find_data->file_attributes(Win32::Dokan::Const::FILE_ATTRIBUTE_DIRECTORY);
	$find_data->creation_time($t);
	$find_data->last_access_time($t);
	$find_data->last_write_time($t);
	$find_data->file_name('.');
	$find_data->number_of_links(2);

	unshift(@ret, $find_data);
    }

    return (@ret, 0);
}

sub cb_delete_file {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    return $self->fuse_unlink($pathName);
}

sub cb_delete_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    return $self->fuse_unlink($pathName);
}

sub cb_set_file_attributes {
    my $self = shift;
    my ($pathName, $attributes, $DFileInfo) = @_;

    return 0;
}

sub cb_set_file_time {
    my $self = shift;
    my ($pathName, $ctime, $atime, $mtime, $DFileInfo) = @_;

    return $self->fuse_utime($pathName, $atime, $mtime);
}

sub cb_move_file {
    my $self = shift;
    my ($existingName, $newFileName, $repaceExisting, $DFileInfo) = @_;

    return $self->fuse_rename($existingName, $newFileName);
}

sub cb_set_end_of_file {
    my $self = shift;
    my ($fileName, $length, $DFileInfo) = @_;

    return $self->fuse_truncate($fileName, $length);
}

sub cb_lock_file {
    my $self = shift;
    my ($fileName, $offset, $length, $DFileInfo) = @_;

    return 0;
}

sub cb_unlock_file {
    my $self = shift;
    my ($fileName, $offset, $length, $DFileInfo) = @_;

    return 0;
}

sub cb_get_disk_free_space {
    my $self = shift;
    my ($DFileInfo) = @_;

    # Fues statfs
    # enoano
    # namelen, files, files_free, blocks, blocks_avail, blocksize
    # enoano, namelen, files, files_free, blocks, blocks_avail

    # avail (for current user), total (used bytes), free (of disk)
    return (123.0*1E15, 456.0*1E15, 123.0*1E15);
}

sub cb_get_volume_information {
    my $self = shift;
    my ($DFileInfo) = @_;
    my ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);

    $volume_name = "test";
    $serial = 0x123;
    $component_length = 255;
    $file_system_flags = 0;

    $volume_name = encode('UTF16-LE', $volume_name) if (defined($volume_name));
    $file_system_name = encode('UTF16-LE', $file_system_name) if (defined($file_system_name));

    return ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);
}

sub cb_unmount {
    my $self = shift;
    my ($DFileInfo) = @_;

    return 0;
}

sub main_loop {
    my $self = shift;

    my $drive = $self->{_fuse_opts}->{mountpoint};
    if ($drive =~ /^([a-z]):?$/i) {
	$drive = $1;
    }
    else {
	croak "bad drive: $drive";
    }
    my $opt = Win32::Dokan::DokanOptions->new();

    $opt->drive_letter(uc $drive);
    $opt->debug_mode(1);
    $opt->use_std_err(1);

    $self->SUPER::main_loop($drive, $opt);
}

BEGIN {
    no strict "refs";
    *Fuse::main = sub {
	my $fuse = Win32::Dokan::Fuse->new(@_);
	$fuse->main_loop;
    };

    # Don't load Fuse.pm
    no strict 'vars';
    use vars qw($::INC);
    $::INC{'Fuse.pm'} = 1;
}

1;
