#
# Win32::Dokan Dokan library interface
#

=head1 NAME

Win32::Dokan - Inteface for Dokan library (user mode file system for windows)

=head1 SYNOPSIS

  ###############################################
  # in your filesystem
  #
  package Your::File::System;
  use Win32::Dokan::FS;

  use base qw(Win32::Dokan::FS);
  # override methods below...


  ###############################################
  # in your main script
  #
  use Win32::Dokan::Mounter;
  use Your::File::System;

  my $fs = Your::File::System->new();
  my $mounter = Win32::Dokan::Mounter->new({debug_mode => 1,
					   use_std_err => 1});
  $mounter->mount('W', $fs);


=head1 DESCRIPTION

Win32::Dokan itself is a very low level inteface for Dokan.dll.
See Win32::Dokan::FS to implement your own filesystem.

*** Very low level interface is described below ***

Callbacks defined in Dokan library are mapped into perl functions.

Generic mapping rules are:

=over 2

  Numeric value is passed as is. If you needs Win32 constants,
  use Win32::Dokan::Const or some good module.

  Strin value is encoded with specified encoding using
  "fs_encoding" method.

=back

=over 2

  Pointer to DOKAN_FILE_INFO is mapped to reference
  to Win32::Dokan::DokanFileInfo object.

  DOKAN_FILE_INFO's member "Context" is preserved for library user.
  Context value can be set or get through "context" method
  in Win32::Dokan::DokanFileInfo.

=back

=over 2

  Callbacks are called in OO style. (eg. $obj->method).
  You can override methods in perl OO manner.

=back

=head1 METHODS

=cut

package Win32::Dokan;

use strict;
use warnings;

# use threads;

use Carp;

use Win32::Dokan::DokanFileInfo;
use Win32::Dokan::DokanOptions;
use Win32::Dokan::FileInfo;
use Win32::Dokan::FindDataW;

use Encode;
use Fcntl qw(:DEFAULT);
use Errno;

our $VERSION = '0.03_1';

sub AUTOLOAD {
    # This AUTOLOAD is used to 'autoload' constants from the constant()
    # XS function.

    my $constname;
    our $AUTOLOAD;
    ($constname = $AUTOLOAD) =~ s/.*:://;
    croak "&Win32::Dokan::constant not defined" if $constname eq 'constant';
    my ($error, $val) = constant($constname);
    if ($error) { croak $error; }
    {
	no strict 'refs';
	# Fixed between 5.005_53 and 5.005_61
#XXX	if ($] >= 5.00561) {
#XXX	    *$AUTOLOAD = sub () { $val };
#XXX	}
#XXX	else {
	    *$AUTOLOAD = sub { $val };
#XXX	}
    }
    goto &$AUTOLOAD;
}

require XSLoader;
XSLoader::load('Win32::Dokan', $VERSION);

sub _fsname_to_unicode {
    my $self = shift;
    my $name = shift;

    if ($self->{_use_mbcs}) {
	return $self->convert_native_to_unicode($name);
    }

    my $ret = $self->{_fs_enc} ? $self->{_fs_enc}->decode($name) : $name;

    return $self->{_uni_enc}->encode($ret);
}

sub _unicode_to_fsname {
    my $self = shift;
    my $name = shift;

    if ($self->{_use_mbcs}) {
	my $ret;
	if ($self->{_convert_path}) {
	    $ret = $self->{_uni_enc}->decode($name);
	    $ret =~ s/\\/\//g if ($self->{_convert_path});
	    $ret = $self->{_uni_enc}->encode($ret);
	}
	else {
	    $ret = $name;
	}
	return $self->convert_unicode_to_native($ret);
    }

    my $ret = $self->{_uni_enc}->decode($name);
    $ret =~ s/\\/\//g if ($self->{_convert_path});
    
    if ($self->{_fs_enc}) {
	return $self->{_fs_enc}->encode($ret);
    }

    return $ret;
}

sub _enflat_context {
    my $self = shift;
    my $dfi = shift;
    my $old_context_value = shift;

    my $context = $dfi->context;
    my $new_context_value;

    if (defined($context)) {
	if (ref($context)) {
	    $new_context_value = int(0+$context);
	    $self->{_files}->{$new_context_value} = [$context];
	}
	else {
	    my $new_ref = [$context];
	    $new_context_value = int(0+$new_ref);
	    $self->{_files}->{$new_context_value} = $new_ref;
	}
	$dfi->context($new_context_value);
    }

    if (defined($old_context_value)) {
	if (defined($new_context_value)) {
	    if ($new_context_value != $old_context_value) {
		delete $self->{_files}->{$old_context_value};
	    }
	}
	else {
	    delete $self->{_files}->{$old_context_value};
	}
    }
}

sub _populate_context {
    my $self = shift;
    my $dfi = shift;

    my $context = $self->{_files}->{$dfi->context};

    if (ref($context) eq 'ARRAY') {
	$dfi->context($context->[0])
    }
    else {
	$dfi->context(undef);
    }
}

sub _call_cb_simple {
    my $self = shift;
    my $method = shift;
    my $fileName = shift;
    # rest is in @_

    my $dfi;
    my $org_context_value;

    if (ref($_[$#_]) eq 'Win32::Dokan::DokanFileInfo'
	&& defined($_[$#_]->context)) {

	$dfi = $_[$#_];
	$org_context_value = $dfi->context;
	$self->_populate_context($dfi);
    }

    $fileName = $self->_unicode_to_fsname($fileName);

    if (wantarray()) {
	my @ret;
	eval {
	    @ret = $self->$method($fileName, @_);
	};
	my $err = $@;
	carp "************** exception: $@" if ($@);
	if ($dfi) {
	    $self->_enflat_context($dfi, $org_context_value);
	};
	return $err ? -1 : @ret;
    }
    else {
	my $ret;
	eval {
	    $ret = $self->$method($fileName, @_);
	};
	my $err = $@;
	carp "************** exception: $@" if ($@);
	if ($dfi) {
	    $self->_enflat_context($dfi, $org_context_value);
	};
	return $err ? -1 : $ret;
    }
}

#
# CreateFile
#

=head2 cb_create_file (filename, desiredAccess, shareMode, createDisposition, flagsAndAttributes, DFileInfo)

Callback for CreateFile.

 - desiredAccess, shareMode, createDisposition, flagsAndAttributes has same value in C. (see Win32 CreateFile document for details)

 - DFileInfo is reference to Win32::Dokan::DokanFileInfo object.
   You can set value using $DFileInfo->context("value"). This value is
   available until cb_cleanup is called. if any value is not set, internal
   value is set to trace file.

=cut

sub _cb_create_file {
    shift->_call_cb_simple("cb_create_file", @_);
}

sub cb_create_file {
    my $self = shift;
    my ($fileName, $desiredAccess, $shareMode, $createDisposition, $flagsAndAttributes, $DFileInfo) = @_;

    # carp "cb_create_file";
    return -2;
}

#
# OpenDirectory
#

=head2 cb_open_directory (pathName, DFileInfo)

Callback for OpenDirectory.

=cut

sub _cb_open_directory {
    shift->_call_cb_simple("cb_open_directory", @_);
}

sub cb_open_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    carp "cb_open_directory";
    return -2;
}

#
# CreateDirectory
#

=head2 cb_create_directory (pathName, DFileInfo)

Callback for CreateDirectory

=cut

sub _cb_create_directory {
    shift->_call_cb_simple("cb_create_directory", @_);
}

sub cb_create_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;
 
    carp "cb_create_directory";
    return -2;
}

#
# Cleanup
#

=head2 cb_cleanup (fileName, DFileInfo)

Callback for Cleanup.

=cut

sub _cb_cleanup {
    shift->_call_cb_simple("_cb_cleanup2", @_);
}

sub _cb_cleanup2 {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;
    my $ret = $self->cb_cleanup($fileName, $DFileInfo);

    # free stored context
    $DFileInfo->context(undef);

    return $ret;
}

sub cb_cleanup {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    carp "cb_cleanup";
    return 0;
}

#
# CloseFile
#

=head2 cb_close_file (fileName, DFileInfo)

Callback for CloseFile.

=cut

sub _cb_close_file {
    shift->_call_cb_simple("cb_close_file", @_);
}

sub cb_close_file {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    carp "cb_close_file";
    return 0;
}

#
# ReadFile
#

=head2 cb_read_file (fileName, buffer, offset, length, DFileInfo)

Callback for ReadFile.
Read data must be stored in $buffer passwd in @_.

=cut

sub _cb_read_file {
    shift->_call_cb_simple("cb_read_file", @_);
}

sub cb_read_file {
    my $self = shift;
    my ($fileName, $dummy, $offset, $length, $DFileInfo) = @_;

    # read data in $_[1] !!!

    carp "cb_read_file";
    return;
}

#
# WriteFile
#

=head2 cb_write_file (fileName, data, offset, DFileInfo)

Callback for WriteFile.
Data length is shown as length of $data.

=cut

sub _cb_write_file {
    shift->_call_cb_simple("cb_write_file", @_);
}

sub cb_write_file {
    my $self = shift;
    my ($fileName, $data, $offset, $dummy, $DFileInfo) = @_;
 
    # store len in $_[3] !!!

    carp "cb_write_file";
    return 0;
}

#
# FlushFileBuffers
#

=head2 cb_flush_file_buffers (fileName, DFileInfo)

Callback for FlushFileBuffers.

=cut

sub _cb_flush_file_buffers {
    shift->_call_cb_simple("cb_flush_file_buffers", @_);
}

sub cb_flush_file_buffers {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    carp "cb_flush_file_buffers";
    return 0;
}

#
# GetFileInformation
#

=head2 cb_get_file_information (fileName, fileInfo, DFileInfo)

Callback for GetFileInformation.

$fileInfo is reference to Win32::Dokan::FileInfo object.

Note:

=over 2

  If fileName is normal file, set
  Win32::Dokan::Const::FILE_ATTRIBUTE_NORMAL to $fileInfo->file_attributes.

  If fileName is directory, set
  Win32::Dokan::Const::FILE_ATTRIBUTE_DIRECTORY to $fileInfo->file_attributes.

  In Win32::Dokan::FileInfo, timestamps are seconds from epoch (1970-01-01).
  Don't use Windows FILETIME value.

=back

=cut

sub _cb_get_file_information {
    shift->_call_cb_simple("cb_get_file_information", @_);
}

sub cb_get_file_information {
    my $self = shift;
    my ($fileName, $fileInfo, $DFileInfo) = @_;

    # store info into $fileInfo
    carp "cb_get_file_information";
    return -1;
}

#
# FindFiles
#

=head2 cb_find_files (fileName, DFileInfo)

Callback for GetFindFiles.

Return array contains Win32::Dokan::FindDataW objects and errno.
($data1, $data2, ..., 0)

Note:

=over 2

  If fileName is normal file, set
  Win32::Dokan::Const::FILE_ATTRIBUTE_NORMAL to FindDataW->file_attributes.

  If fileName is directory, set
  Win32::Dokan::Const::FILE_ATTRIBUTE_DIRECTORY to FindDataW->file_attributes.

  In Win32::Dokan::FindDataW, timestamps are seconds from epoch (1970-01-01).
  Don't use Windows FILETIME value.

=back

=cut

sub _cb_find_files {
    my $self = shift;
    $self->_call_cb_simple("cb_find_files", @_);
}

sub cb_find_files {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    # return array of Win32::Dokan::FindDataW, and errno
    carp "cb_find_files";
    return (0);
}

#
# DeleteFile
#

=head2 cb_delete_file (fileName, DFileInfo)

Callback for DeleteFile.

In this callback, only check is done if the file can be deleted.
Execute deletion when cb_cleanup is called with
$DFileInfo->delete_on_close is true.

=cut

sub _cb_delete_file {
    shift->_call_cb_simple("cb_delete_file", @_);
}

sub cb_delete_file {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    carp "cb_delete_file";
    return -2;
}

#
# DeleteDirectory
#

=head2 cb_delete_directory (pathName, DFileInfo)

Callback for DeleteDirectory.

In this callback, only check is done if the directory can be deleted.
Execute deletion when cb_cleanup is called with
$DFileInfo->delete_on_close is true.

=cut

sub _cb_delete_directory {
    shift->_call_cb_simple("cb_delete_directory", @_);
}

sub cb_delete_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    carp "cb_delete_directory";
    return -2;
}


#
# SetFileAttributes
#

=head2 cb_set_file_attributes (pathName, attributes, DFileInfo)

Callback for SetFileAttributes.

=cut

sub _cb_set_file_attributes {
    shift->_call_cb_simple("cb_set_file_attributes", @_);
}

sub cb_set_file_attributes {
    my $self = shift;
    my ($pathName, $attributes, $DFileInfo) = @_;

    carp "cb_set_file_attributes";
    return -2;
}

#
# SetFileTime
#

=head2 cb_set_file_time (pathName, ctime, atime, mtime DFileInfo)

Callback for SetFileTime.

ctime, atime and mtime has seconds from epoch (1970-01-01).

=cut

sub _cb_set_file_time {
    shift->_call_cb_simple("cb_set_file_time", @_);
}

sub cb_set_file_time {
    my $self = shift;
    my ($pathName, $ctime, $atime, $mtime, $DFileInfo) = @_;

    carp "cb_set_file_time";
    return -2;
}

#
# MoveFile
#

=head2 cb_move_file (existingName, newFileName, repaceExisting, DFileInfo)

Callback for MoveFile

=cut

sub _cb_move_file {
    my $self = shift;
    my $name1 = shift;
    my $name2 = shift;
    # rest is in @_

    $name1 = $self->_unicode_to_fsname($name1);
    $name2 = $self->_unicode_to_fsname($name2);

    my $ret;
    eval {
	$ret = $self->cb_move_file($name1, $name2, @_);
    };
    my $err = $@;
    carp "************** exception: $@" if ($@);
    return $err ? -1 : $ret;
}

sub cb_move_file {
    my $self = shift;
    my ($existingName, $newFileName, $repaceExisting, $DFileInfo) = @_;

    carp "cb_move_file";
    return -2;
}

#
# SetEndOfFile
#

=head2 cb_set_end_of_file (fileName, length, DFileInfo)

Callback for SetEndOfFile

=cut

sub _cb_set_end_of_file {
    shift->_call_cb_simple("cb_set_end_of_file", @_);
}

sub cb_set_end_of_file {
    my $self = shift;
    my ($fileName, $length, $DFileInfo) = @_;

    carp "cb_set_end_of_file";
    return -2;
}

#
# LockFile
#

=head2 cb_lock_file (fileName, offset, length, DFileInfo)

Callback for LockFile

=cut

sub _cb_lock_file {
    shift->_call_cb_simple("cb_lock_file", @_);
}

sub cb_lock_file {
    my $self = shift;
    my ($fileName, $offset, $length, $DFileInfo) = @_;

    carp "cb_lock_file";
    return -2;
}

#
# UnlockFile
#

=head2 cb_unlock_file (fileName, offset, length, DFileInfo)

Callback for UnlockFile

=cut

sub _cb_unlock_file {
    shift->_call_cb_simple("cb_unlock_file", @_);
}

sub cb_unlock_file {
    my $self = shift;
    my ($fileName, $offset, $length, $DFileInfo) = @_;

    carp "cb_unlock_file";
    return -2;
}

sub _call_cb_noname {
    my $self = shift;
    my $method = shift;

    # rest is in @_
    if (wantarray()) {
	my @ret;
	eval {
	    @ret = $self->$method(@_);
	};
	my $err = $@;
	carp "************** exception: $@" if ($@);
	return $err ? -1 : @ret;
    }
    else {
	my $ret;
	eval {
	    $ret = $self->$method(@_);
	};
	my $err = $@;
	carp "************** exception: $@" if ($@);
	return $err ? -1 : $ret;
    }
}

#
# GetDiskFreeSpace
#

=head2 cb_get_disk_free_space (DFileInfo)

Callback for GetDiskFreeSpace.
Return (avaiable_bytes_for_uset, used_bytes, free_bytes_for_disk).

All values are treated floating point value for over 32bit value.

=cut

sub _cb_get_disk_free_space {
    my $self = shift;
    my ($avail, $total, $free)
	= $self->_call_cb_noname("cb_get_disk_free_space", @_);

    return (0.0+$avail, 0.0+$total, 0.0+$free);
}

sub cb_get_disk_free_space {
    my $self = shift;
    my ($DFileInfo) = @_;

    carp "cb_get_disk_free_space";

    # avail (for current user), total (bytes), free (of disk)
    return (0, 0, 0);
}

#
# GetVolumeInformation
#

=head2 cb_get_volume_information (DFileInfo)

Callback for GetVolumeInformation
Return (volume_name, serial, component_length, file_system_flags, file_system_name)

=cut

sub _cb_get_volume_information {
    my $self = shift;
  
    my ($volume_name,
	$serial,
	$component_length,
	$file_system_flags,
	$file_system_name);

    ($volume_name,
     $serial,
     $component_length,
     $file_system_flags,
     $file_system_name)
	= $self->_call_cb_noname("cb_get_volume_information", @_);

    if (defined($volume_name)) {
	$volume_name = $self->_fsname_to_unicode($volume_name);
    }
    else {
	$volume_name = "Win32-Dokan";
    }

    if (defined($file_system_name)) {
	$file_system_name = $self->_fsname_to_unicode($file_system_name);
    }
    else {
	$file_system_name = "Win32-Dokan";
    }

    $serial = 0 unless (defined($serial));
    $component_length = 255 unless (defined($component_length));
    $file_system_flags = 0 unless (defined($file_system_flags));


    return ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);
}

sub cb_get_volume_information {
    my $self = shift;
    my ($DFileInfo) = @_;

    my ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);

    carp "cb_get_volume_information";
    return ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);
}

sub _cb_unmount {
    shift->_call_cb_noname("cb_unmount", @_);
}

sub cb_unmount {
    my $self = shift;
    my ($DFileInfo) = @_;

    carp "cb_unmount";

    return 0;
}

our $_DLL = Win32::Dokan::_DLL->new;

#
# prepare_main
#

=head2 prepare_main

To serialize events, Win32::Dokan module uses other Win32 thread.
This control thread is stared by this method.

Normaly, it is not needed to call this method explicitly.

=cut

sub prepare_main {
    my $self = shift;
    my $opt = shift;

    $self->start_main_thread($_DLL->{handle}, $opt)
	|| croak("cannot start Dokan main thread");

    # refcnt++
    $self->{_dll} = $_DLL;

    # need unmount
    $self->{_running} = 1;
}

#
# main_loop
#

=head2 main_loop (drive, opt)


Start filesystem. Control is not returned until filesystem is unmounted.

$drive is drive letter 'A'-'Z' to mount filesystem.

$opt is options in hashref for Dokan library.

=head3 hash keys:

=over 2

=item thread_count - Dokan thread count.

=item debug_mode - Debug output is On(true) or Off(false).

=item use_std_err - Debug output is STDERR (true) or Debug Console (false).

=back

=cut

sub main_loop {
    my $self = shift;
    my ($drive, $opt) = @_;

    if (defined($opt)) {
	if (ref($opt) eq 'HASH') {
	    $opt = $self->{options} = Win32::Dokan::DokanOptions->new($opt);
	}
        elsif (ref($opt) eq 'Win32::Dokan::DokanOptions') {
	    $opt = $self->{options} = $opt;
	}
	else {
	    croak "unknown type option: " . ref($opt);
	}
    }
    unless ($opt) {
	$opt = $self->{options};
    }

    croak("Bad drive letter: $drive") unless ($drive =~ /^[A-Z]$/i);

    $opt->drive_letter(uc $drive);
    $self->prepare_main($opt);

    local $Win32::Dokan::FindDataW::Unicode_func = sub {
	my $name = shift;
	return $self->_fsname_to_unicode($name);
    };

    local $SIG{INT} = sub {
	$self->_unmount($_DLL->{handle}, $self->{options}->drive_letter);
	$self->{_running} = 0;
    };
    local $SIG{TERM} = $SIG{INT};
    local $SIG{HUP} = $SIG{INT};

    $self->main;

    $self->{_running} = 0;
} 

#
# fs_encoding
#

=head2 fs_encoding

Set or get encoding used in filesystem.

If undef is passed to set, utf8 flagged name is used for filesystem.

If 'x-mbcs' is passed, Windows MBCS is used.

Otherwise encoding name is passed to Encode module.

=cut

sub fs_encoding {
    my $self = shift;
    return $self->{_fs_encoding} unless (@_);

    my $encoding = shift;
    if (defined($encoding)) {
	if ($encoding =~ /^X-MBCS$/i) {
	    $self->{_use_mbcs} = 1;
	}
	else {
	    my $enc = Encode::find_encoding($encoding)
		or croak "unknown encoding: $encoding";
	    $self->{_fs_enc} = $enc;
	}
    }
    else {
	$self->{_fs_enc} = undef;
    }

    return $encoding;
}

#
# new
#

=head2 new (opt)

$opt is options in hashref for Win32::Dokan

=head3 hash keys:

=over 2

=item fs_encoding - encoding name used in this filesystem.

=item convert_path - convert '\' to '/' (true) or not (false).

=back

Other keys are recognized as options for Dokan library (see main_loop method).

=cut

sub new {
    my $class = shift;
    my $opt = shift;

    my $dokan_options;
    my $enc;

    my $self = bless {
    }, $class;

    if (defined($opt) && ref($opt) eq 'HASH') {
	$self->fs_encoding($opt->{fs_encoding}) if ($opt->{fs_encoding});
	$self->{_convert_path} = $opt->{convert_path};

	$dokan_options = Win32::Dokan::DokanOptions->new($opt);
    }
    else {
	$dokan_options = Win32::Dokan::DokanOptions->new();
    }

    $self->{options} = $dokan_options;
    $self->{_files} = {};
    $self->{_uni_enc} = Encode::find_encoding('UTF-16LE');

    return $self;
}

#
# unmount
#

=head2 unmount (drive)

unmount drive.

=cut

sub unmount {
    my $class_or_instance = shift;
    my $drive = shift;

    unless ($drive) {
	if (ref($class_or_instance)) {
	    $drive = $class_or_instance->{options}->drive_letter;
	}
    }
    croak "drive letter must be specified to unmount" unless ($drive);
    croak("Bad drive letter: $drive") unless ($drive =~ /^[A-Z]$/i);

    $class_or_instance->_unmount($_DLL->{handle}, uc $drive);
}

sub DESTROY {
    my $self = shift;
    if ($self->{_running}
	&& defined($self->{options})
	&& defined($self->{options}->drive_letter)) {

	$self->_unmount($_DLL->{handle}, $self->{options}->drive_letter);
    }
}


package Win32::Dokan::_DLL;

use Carp;
use Win32;

sub new {
    my $class = shift;

    my @path = split(';', $ENV{PATH});
    unshift(@path, "$ENV{windir}\\System32");

    my $handle;

    for my $p (@path) {
	my $dll_path = "$p\\Dokan.dll";
	if (-f $dll_path) {
	    last if ($handle = Win32::LoadLibrary($dll_path));
	}
    }
    croak "Cannot load Dokan.dll" unless ($handle);

    # print STDERR "dll loaded\n";

    bless {
	handle => $handle
    }, $class;
}

sub DESTROY {
    my $self = shift;

    # print STDERR "dll unloaded\n";

    Win32::FreeLibrary($self->{handle});
}

1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 SEE ALSO

Win32::Dokan::FS, Win32::Dokan::Mounter, http://dokan-dev.net/en/

=head1 AUTHOR

Toshimitsu FUJIWARA, C<< <tttfjw at gmail.com> >>

=head1 BUGS

Threading is not supported.


=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Win32::Dokan

Win32::Dokan project is maintained at
  http://sourceforge.jp/projects/perl-fuse-fv/

=head1 ACKNOWLEDGEMENTS

 Dokan library is developed by Hiroki Asakawa.
 See below for details.
   http://dokan-dev.net/en/

=head1 COPYRIGHT & LICENSE

Copyright 2009 Toshimitsu FUJIWARA, all rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.
