#
# Base class for Dokan root
#
package Win32::Dokan::FS;

=head1 NAME

Win32::Dokan::FS - Win32::Dokan filesystem base class.

=head1 SYNOPSIS

  ###############################################
  # in your filesystem
  #
  package Your::File::System;
  use Win32::Dokan::FS;

  use base qw(Win32::Dokan::FS);
  # override methods below...


  ###############################################
  # in your main script
  #
  use Win32::Dokan::Mounter;
  use Your::File::System;

  my $fs = Your::File::System->new();
  my $mounter = Win32::Dokan::Mounter->new({debug_mode => 1,
					   use_std_err => 1});
  $mounter->mount('W', $fs);


=head1 DESCRIPTION

This is a base class of filesystem for Win32::Dokan::Mounter.
Each methods are called from Win32::Dokan::Mounter main loop.

=head2 EXPORT

None by default.

=cut

use Carp;
use Win32::Dokan::Const qw(-compile :error);

=head1 METHODS

=head2 new

constructor

=cut

sub new {
    my $class = shift;
    bless {}, $class;
}

=head2 encoding

Define character encoding used for filename.
If not defined (default), utf-8 flagged string is used for filename.

Special encoding 'X-MBCS' is defined for Windows MBCS character encoding.

=cut

sub encoding {
    # undef, X-MBCS, UTF-8, or something defined in Encode module.
    return undef;
}

=head2 open(filename, mode, dokan_file_info)

Called to open a existing file. 'mode' has flags defined in Fcntl module
like O_RDWR, O_RDONLY, O_WRONLY, etc.

dokan_file_info is reference to Win32::Dokan::DokanFileInfo object.
If you set dokan_file_info->context, that value is preserved
until close method is called.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error.)

=cut

sub open {
    my ($self, $path, $mode, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef
    carp("open $path, $mode\n");

    return undef;
}

=head2 create(filename, mode, dokan_file_info)

Called to create a new file or truncate existing file.
'mode' has flags defined in Fcntl module like O_RDWR, O_RDONLY, O_WRONLY, etc.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error).

=cut

sub create {
    my ($self, $path, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef
    return undef;
}

=head2 truncate(filename, length, dokan_file_info)

Called to truncate a file. 'length' has file size truncated to.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error).

=cut

sub truncate {
    my ($self, $filename, $length, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef
    return undef;
}

=head2 opendir(filename, path, dokan_file_info)

Called to start access to directory.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error).

=cut

sub opendir {
    my ($self, $path, $fileinfo) = @_;
    return 0 if ($path eq '' || $path eq '/');

    # OK: 0 or positive value
    # NG: negative errno or undef
    return -1 * Win32::Const::ERROR_FILE_NOT_FOUND;
}

=head2 mkdir(filename, path, dokan_file_info)

Called to create new directory.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error).

=cut

sub mkdir {
    my ($self, $path, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef
    return undef;
}

=head2 close(filename, dokan_file_info)

Called to close file or directory.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error).

=cut

sub close {
    my ($self, $path, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef
    return 1;
}

=head2 read(filename, offset, length, dokan_file_info)

Called to read data from a file.

Return value is read data or undef (generic unknown error).

When error has occured, you can return array like '(undef, -errno)',
and second value is treated as errno.

=cut

sub read {
    my ($self, $path, $offset, $length, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef
    return undef;
}

=head2 write(filename, file, offset, data, dokan_file_info)

Called to write data to file.

Return value is written size.

When error has occured, you can return array like '(undef, -errno)',
and second value is treated as errno.

=cut

sub write {
    my ($self, $path, $offset, $data, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef
    return undef;
}


=head2 flush(filename, dokan_file_info)

Called to flush buffered data for file.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error).

=cut

sub flush {
    my ($self, $path, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef
    return undef;
}

=head2 stat(filename, dokan_file_info)

Called to get file/directory attributes.

Return value is reference to array having attributes.
attribute arrayref is defined as below:

  [size, attr, ctime, atime, mtime]

  attr:
    normal file - will be Win32::Dokan::Const::FILE_ATTRIBUTE_NORMAL
    directory - Win32::Dokan::Const::FILE_ATTRIBUTE_NORMAL

When error has occured, you can return errno in scalar.

=cut

sub stat {
    my ($self, $fileinfo) = @_;
    # (size, attr, ctime, atime, mtime)
    # attr = Win32::Dokan::Const::FILE_ATTRIBUTE_NORMAL or Win32::Dokan::Const::FILE_ATTRIBUTE_DIRECTORY

    # OK: (size, attr, ctime, atime, mtime)
    # NG: negative errno or undef

    return;
}

=head2 setattr(filename, attr, dokan_file_info)

Called to set file/directory attributes.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error.)

=cut

sub setattr {
    my ($self, $attr, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef

    return undef;
}

=head2 utime(filename, ctime, atime, mtime, dokan_file_info)

Called to set file/directory timestamps.

Time is seconds from 1970/1/1 00:00:00.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error.)

=cut

sub utime {
    my ($self, $path, $ctime, $atime, $mtime, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef

    return undef;
}

=head2 remove(filename, dokan_file_info)

Called to remove a file.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error.)

=cut

sub remove {
    my ($self, $path, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef

    return undef;
}

=head2 remove(filename, newfilename, dokan_file_info)

Called to rename a file.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error.)

=cut

sub rename {
    my ($self, $path, $newpath, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef

    return undef;
}

=head2 rmdir(path, dokan_file_info)

Called to remove a directory.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error.)

=cut

sub rmdir {
    my ($self, $path, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef
    return undef;
}

=head2 lock(filename, offset, length dokan_file_info)

Called to lock a region of file.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error.)

=cut

sub lock {
    my ($self, $path, $offset, $length, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef
    return 0;
}

=head2 unlock(filename, offset, length dokan_file_info)

Called to unlock a region of file.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error.)

=cut

sub unlock {
    my ($self, $path, $offset, $length, $fileinfo) = @_;

    # OK: 0 or positive value
    # NG: negative errno or undef
    return 0;
}

=head2 unmount(dokan_file_info)

Called when unmounting a filesystem.

Return value can be 0(OK), negative value (-errno),
or undef (generic unknown error.)

=cut

sub unmount {
    my ($self, $fileinfo) = @_;
    return undef;
}

=head2 disk_free_space(dokan_file_info)

Called to get disk free space

Return value is array consists
(available_bytes_for_current_user, bytes_in_disk, free_space).

Each value can be double (floating point) value.

=cut

sub disk_free_space {
    my $self = shift;
    my ($DFileInfo) = @_;

    return (0, 0, 0);
}

=head2 volume_information(dokan_file_info)

Called to get volume information

Return value is array consists
(volume_name, serial_number, component_length, file_system_flags, file_system_name)

If undef is used for a value, default value is used.

=cut

sub volume_information {
    my $self = shift;
    my ($DFileInfo) = @_;
    my ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);

    $volume_name = "dokan";
    $serial = 0;
    $component_length = 255;
    $file_system_flags = 0;
    $file_system_name = "dokan";

    return ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);
}

1;
__END__


=head1 SEE ALSO

Win32::Dokan::Mounter, Win32::Dokan

=head1 AUTHOR

Toshimitsu FUJIWARA, C<< <tttfjw at gmail.com> >>

=head1 BUGS

Threading is not supported.


=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Win32::Dokan::FS

Win32::Dokan project is maintained at
  http://sourceforge.jp/projects/perl-fuse-fv/

=head1 ACKNOWLEDGEMENTS

 Dokan library is developed by Hiroki Asakawa.
 See below for details.
   http://dokan-dev.net/en/


=head1 COPYRIGHT & LICENSE

Copyright 2009 Toshimitsu FUJIWARA, all rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.
