<?php
/*
 * This file is part of the pettiwork package.
 * (c) 2007-2008 Exbridge,inc. <info@exbridge.jp>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

/**
 * Project: pettiwork: the PHP lightweight web framework
 * File:    xbpwController.php
 *
 * @link http://exbridge.jp/
 * @author S.Tajima <tajima@exbridge.jp>
 * @version svn:$Id: xbpwController.php 613 2008-06-30 04:30:41Z tajima $
 * @copyright 2007-2008 Exbridge,Inc.
 */
class xbpwController
{
    /** Key for webaccess value */
    var $_pw_webaccess_key = PW_WEBACCESS_KEY;

    /** Key for action value */
    var $_pw_action_key = PW_ACTION_KEY;

    /** template file extension */
    var $_pw_template_ext = 'tpl';

    /** name of view engine */
    var $_pw_template_engine = PW_TEMPLATE_ENGINE;

    /** name of view engine class */
    var $_pw_template_class = PW_TEMPLATE_CLASS;

    /** name of prefix dir */
    var $_pw_prefix_dir = '';

    /** name of templates_prefix dir */
    var $_pw_template_prefix_dir = '';

    /** name of templates dir */
    var $_pw_template_templates_dir = PW_TEMPLATE_TEMPLATES_DIR;

    /** name of plugins dir */
    var $_pw_template_plugins_dir = PW_TEMPLATE_PLUGINS_DIR;

    /** name of compile dir */
    var $_pw_template_compile_dir = PW_TEMPLATE_COMPILE_DIR;

    /** name of template type */
    var $_pw_template = PW_LAYOUT_TEMPLATE;

    /** name of default action */
    var $_pw_default_action = 'default';

    /** name of layout template */
    var $_pw_layout_template = 'default';

    /** name of validator class */
    var $_pw_validator_class = null;

    /** name of form parameter callback method */
    var $_pw_form_parameter_callback = null;

    /** array of before filter method names */
    var $_pw_before_filters = array();

    /** array of after filter method names */
    var $_pw_after_filters = array();

    /** array of around filter class names */
    var $_pw_around_filters = array();

    /** session disable flag */
    var $_pw_disable_session = PW_DISABLE_SESSION;

    /** specified action name */
    var $_pw_action = '';

    /** instance of view processor */
    var $_pw_view = null;

    /** request parameters  */
    var $_pw_params = array();

    /** instance of Request */
    var $_pw_request = null;

    /** array of validation error messages */
    var $_pw_errors = '';

    var $_pw_css = array(); //CSS

    var $_pw_jscript = array();

    var $_pw_meta_header = array();

    /**
     * Creates new instance.
     * @access public
     */
    function xbpwController()
    {
        //session start
        if (!$this->_pw_disable_session) {
            $_pw_session_conf = PW_APP_CONFIG_DIR . DS . 'session.inc';
            if (PW_PREFIX != '') {
                $_pw_session_conf = PW_APP_CONFIG_DIR . DS . PW_PREFIX . DS . 'session.inc';
            }
            if (file_exists($_pw_session_conf)) {
                $session = new xbpwSession(include($_pw_session_conf));
            }
            else {
                $session = new xbpwSession();
            }
            $session->start();
        }
        // set default validator class name
        if (is_null($this->_pw_validator_class)) {
            $file = PW_APP_VALIDATORS_DIR . DS . PW_APP . 'Validator.php';
            if (PW_CTL != '') {
                $file = PW_APP_VALIDATORS_DIR . DS . PW_CTL . DS . PW_APP . 'Validator.php';
            }
            if (file_exists($file)) {
                require_once $file;
                $this->_pw_validator_class = PW_APP . 'Validator';
            }
        }
        // populate http request
        $this->_pw_request =& xbpwRequest::factory($this->_pw_action_key);
        $this->_pw_params = $this->_pw_request->getParameters();
        //set action
        $this->_estimateAction();
    }

    /**
     * adding css file
     * @access public
     */
    function addCSS($val1, $val2)
    {
        if (is_null($val2)) {
            $this->_pw_css[] = $val1;
        }
        else {
            $this->_pw_css[$val1] = $val2;
        }
    }

    /**
     * adding js file
     * @access public
     */
    function addJavaScript($val)
    {
        $this->_pw_jscript[] = $val;
    }

    /**
     * adding meta header
     * @access public
     */
    function addMetaHeader($key, $val)
    {
        $this->_pw_meta_header[$key] = $val;
    }

    /**
     * setting session value
     * @access public
     */
    function setSession($key, $val)
    {
        xbpwContext::setSession($key, $val);
    }

    /**
     * getting session value
     * @access public
     */
    function getSession($key)
    {
        return xbpwContext::getSession($key);
    }

    /**
     * getting session value
     * @access public
     */
    function clearSession($key)
    {
        xbpwContext::clearSession($key);
    }

    /**
     * setting flash value
     * @access public
     */
    function setFlash($key, $val)
    {
        $_SESSION[PW_SESSION_FLASH_KEY][$key] = $val;
    }

    /**
     * getting flash value
     * @access public
     */
    function getFlash($key)
    {
        return $_SESSION[PW_SESSION_FLASH_KEY][$key];
    }

    /**
     * getting flash value
     * @access public
     */
    function clearFlash($key)
    {
        unset($_SESSION[PW_SESSION_FLASH_KEY][$key]);
    }

    /**
     * setting flash cache value
     * @access public
     */
    function setCache($key, $val)
    {
        xbpwContext::setCache($key, $val);
    }

    /**
     * getting flash cache value
     * @access public
     */
    function getCache($key)
    {
        return xbpwContext::getCache($key);
    }

    /**
     * clear flash value
     * @access public
     */
    function clearCache($key)
    {
        xbpwContext::clearCache($key);
    }

    /**
     * clear all flash value
     * @access public
     */
    function clearAllCache()
    {
        xbpwContext::clearAllCache();
    }

    /**
     * clear all properties
     * @access public
     */
    function clearProperties()
    {
        foreach (get_object_vars($this) as $key => $value) {
            if (preg_match('/^_.*/i', $key)) {
                continue;
            }
            unset($this->$key);
        }
        return true;
    }

    /**
     * Default action method stub.
     * @access public
     */
    function executeDefault()
    {
    }

    /**
     * Processing request.
     * @access public
     */
    function dispatch()
    {
        try {
            //accessLog($_SERVER['REQUEST_URI'], __CLASS__, __FILE__, __LINE__);
            if (PW_EXCEPTION_DISABLED) {
                //user error handler
                ob_start('_fatal_error_handler');
            }

            //initialize setting view
            $this->_initView();

            //initialize controller
            $this->_initializeController();

            //before fillters
            $this->_applyBeforeFilters();

            //before fillters method
            $this->_applyFiltersBeforeMethod();

            //prepare method caller
            $this->_invokePrepareMethod();

            //execute method caller
            $result = $this->_invokeMethod();

            if (PW_DEBUG) {
                echo '<B><I>UserSessionInfomation</I></B><Br>';
                a($_SESSION[PW_SESSION_USER_KEY]);
                echo '<Br>';
                echo '<B><I>FrameworkSessionInfomation</I></B><Br>';
                a($_SESSION);
            }

            //if ($result === PW_TERMINATE_PROCESS) {
            //    return;
            //}
            //$this->render();

            //after fillters method
            $this->_applyFiltersAfterMethod();

            //after fillters
            $this->_applyAfterFilters();
        }
        catch (Exception $e) {
            throw $e;
        }
    }

    /**
     * Rendering view.
     * @access public
     * @return integer
     */
    function render($template = null, $status_code = null)
    {
        if (!is_object($this->_pw_view)
            || strtolower(get_parent_class($this->_pw_view)) != 'xbpwabstractview') {
            $message = "View engine is not initialized correctly.\n"
                . "_pw_template_engine = " . $this->_pw_template_engine . "\n"
                . "_pw_template_ext = " . $this->_pw_template_ext;
            if (PW_EXCEPTION_DISABLED) {
                trigger_error($message, E_USER_ERROR);
            }
            else {
                throw new xbpwException(E_USER_ERROR, $message);
            }
            return PW_TERMINATE_PROCESS;
        }
        $pwProperty = $this->_getPropertiesAsPw();
        $model      = $this->_getPropertiesAsModel();
        if (!is_null($status_code)) {
            $server_protocol = $_SERVER['SERVER_PROTOCOL'];
            if ($server_protocol == '') {
                $server_protocol = 'HTTP/1.0';
            }
            header($server_protocol . ' ' . $status_code);
        }
        foreach ($this->_pw_meta_header as $key=>$val) {
            header($key . ': ' . $val);
        }

        //テンプレートファイルが指定されていない場合、テンプレートファイル名生成
        if (is_null($template) || $template == '') {
            $class_name = get_class($this);
            $base_name = '';
            if (function_exists('preg_match')) {
                if (preg_match('/^([^_]+)_*Controller$/i', $class_name, $matches)) {
                    $base_name = $matches[1];
                } else {
                    $base_name = $class_name;
                }
            }
            else {
                if (eregi('^([^_]+)_*Controller$', $class_name, $matches)) {
                    $base_name = $matches[1];
                }
                else {
                    $base_name = $class_name;
                }
            }
            if ($this->_pw_action != '' && method_exists($this, 'execute' . $this->_pw_action)) {
                $template_name = $this->_pw_action;
            }
            else {
                $template_name = $this->_pw_default_action;
            }
            $template = $base_name . DS . $template_name;
        }
        $template = $template . '.' . $this->_pw_template_ext;

        if ($this->_pw_view->isTemplateExists($template)) {
            $this->_pw_view->dispatch($template, $model, $pwProperty);
            print $this->_pw_view->getResult();
        }
        else {
            $messages = "Template file '" . $this->_pw_template_templates_dir . DS . $template . "' is not exists.";
            if (PW_EXCEPTION_DISABLED) {
                trigger_error($messages, E_USER_ERROR);
            }
            else {
                throw new xbpwException(E_USER_ERROR, $messages);
            }
        }

        unset($_SESSION[PW_SESSION_FLASH_KEY]);
        return PW_TERMINATE_PROCESS;
    }

    /**
     * Redirect to action.
     * @access public
     */
    function redirectTo($action, $params = array())
    {
        if (is_null($this->_pw_params[$this->_pw_webaccess_key])) {
            $url = $_SERVER['PHP_SELF'] . '?action=' . $action;
            $query = array();
            foreach ($params as $key => $val) {
                $query[] = urlencode($key) . '=' . urlencode($val);
            }
            if(count($query)>0) {
                $url .= '&' . implode('&', $query);
            }
        }
        else {
            $url = convert_uri($this->_pw_params[$this->_pw_webaccess_key] . '/' . $action);
            $query = array();
            foreach ($params as $key => $val) {
                $query[] = urlencode($val);
            }
            if(count($query)>0) {
                $url .= '/' . implode('/', $query);
            }
        }
        header('Location: ' . 'http://' . $_SERVER['HTTP_HOST'] . $url);
        return PW_CONTINUE_PROCESS;
    }

    /**
     * forward to action.
     * @access public
     */
    function forward($action)
    {
        $method = 'execute' . $action;
        if (!method_exists($this, $method)) {
            return false;
        }
        if ($method == '') {
            return false;
        }
        call_user_func(array(&$this, $method));
        return PW_CONTINUE_PROCESS;
    }

    /**
     * Redirect to relative-path.
     * @access public
     */
    function redirectToPath($path, $params = array())
    {
        //$uri = $_SERVER['REQUEST_URI'];
        //$regex = "\/([^\/]+)\/";
        //preg_match ("/".$regex."/i", $uri, $match);
        if (is_null($this->_pw_params[$this->_pw_webaccess_key])) {
            $query = array();
            foreach ($params as $key => $value) {
                $query[] = urlencode($key) . '=' . urlencode($value);
            }
            if (count($query)>0) {
                $path .= '?' . implode('&', $query);
            }
            header('Location: http://' . $_SERVER['HTTP_HOST'] . convert_uri($path));
            return PW_CONTINUE_PROCESS;
        }
        else {
            $query = array();
            foreach ($params as $key => $val) {
                $query[] = urlencode($val);
            }
            if(count($query)>0) {
                $path .= '/' . implode('/', $query);
            }
            header('Location: http://' . $_SERVER['HTTP_HOST'] . convert_uri($path));
            return PW_CONTINUE_PROCESS;
        }
    }

    /**
     * Redirect to URL.
     * @access public
     */
    function redirectToUrl($url, $target=null)
    {
        if ($target == null) {
            header('Location: ' . $url);
        }
        else {
            echo sprintf("<html><head><script langage=\"javascript\">function goto(){window.open(\"%s\", \"".$target."\");}</script></head><body onload=\"goto()\"></body></html>", $url);
        }
        return PW_CONTINUE_PROCESS;
    }

    /**
     * validate.
     * @access public
     */
    function validate()
    {
        if ($this->_pw_validator_class == '') {
            $message = 'Validator class not specified.';
            if (PW_EXCEPTION_DISABLED) {
                trigger_error($message, E_USER_ERROR);
            }
            else {
                throw new xbpwException(E_USER_ERROR, $message);
            }
            return false;
        }
        if (!class_exists($this->_pw_validator_class)) {
            $message = "Validator class '" . $this->_pw_validator_class . "' is not exists.";
            if (PW_EXCEPTION_DISABLED) {
                trigger_error($message, E_USER_ERROR);
            }
            else {
                throw new xbpwException(E_USER_ERROR, $message);
            }
            return false;
        }
        $validator = new $this->_pw_validator_class;
        $method_name = 'validate' . ucfirst($this->_pw_action);

        // check validate method availability
        if (!method_exists($validator, $method_name)) {
            $message = "Method '" . $this->_pw_validator_class . "::" . $method_name . "' is not exists.";
            if (PW_EXCEPTION_DISABLED) {
                trigger_error($message, E_USER_ERROR);
            }
            else {
                throw new xbpwException(E_USER_ERROR, $message);
            }
            return false;
        }
        $model = $this->_getPropertiesAsModel();
        call_user_func_array(array(&$validator, $method_name), array($model));
        $this->_pw_errors = $validator->getErrors();
        return (count($this->_pw_errors) == 0);
    }

    /**
     * getErrors.
     * @access public
     */
    function getErrors()
    {
        return $this->_pw_errors;
    }

    /**
     * synchronize.
     * @access public
     */
    function synchronize($params)
    {
        if (is_null($params)) {
            if (PW_EXCEPTION_DISABLED) {
                trigger_error("Controller class parameter is null.", E_USER_ERROR);
            }
            else {
                throw new xbpwException(E_USER_ERROR, "Controller class parameter is null.");
            }
            return;
        }
        if(is_array($params) || is_object($params)) {
            foreach($params as $key => $val) {
                if (preg_match('/^_.*/i', $key)) {
                    continue;
                }
                if (is_object($val)) {
                    $this->$key =& $val;
                }
                else {
                    $this->$key = $val;
                }
            }
        }
    }

    /**
     * file download method
     * @access public
     * @param binary $contents
     * @param string $file
     */
    function downloadFile($contents, $file, $enc=PW_CSV_ENCODE)
    {
        header("Content-Disposition: attachment; filename=\"" . mb_convert_encoding(basename($file), $enc) . "\"");
        header('Content-Length: ' . strlen($contents));
        header('Content-Type: application/octet-stream');
        print(mb_convert_encoding($contents, $enc, PW_ENCODE));
        return PW_TERMINATE_PROCESS;
    }

    /**
     * estimate action.
     * @access private
     */
    function _estimateAction()
    {
        $action = '';
        if ($this->_pw_request->getAction() != '') {
            $action = $this->_pw_request->getAction();
        }
        else {
            $action = $this->_pw_default_action;
        }
        $this->_pw_action = $action;
    }

    /**
     * Initialize View instance.
     * @access private
     * @return boolean
     */
    function _initView()
    {
        if (!is_string($this->_pw_template_engine) || $this->_pw_template_engine == '') {
            $message = "Template engine is not specified.\n_pw_template_engine = " . $this->_pw_template_engine;
            trigger_error($message, E_USER_NOTICE);
            return false;
        }

        //$viewClass = 'xbpw' . strtolower($this->_pw_template_engine) . 'view';
        $viewClass = 'xbpw' . $this->_pw_template_engine . 'view';
        if (!class_exists($viewClass)) {
            $message = "View class '" . $viewClass . "' is not exists.";
            trigger_error($message, E_USER_NOTICE);
            return false;
        }
        $this->_pw_view = new $viewClass;
        $config = $this->_getClassVarsAsConfig('template');
        if ($this->_pw_view->init($config) === false) {
            return false;
        }
        return true;
    }

    /**
     * Initialize controller instance.
     * @access private
     * @return boolean
     */
    function _initializeController()
    {
        // call form parameter callback if defined
        if (is_string($this->_pw_form_parameter_callback)) {
            if (method_exists($this, $this->_pw_form_parameter_callback)) {
                $this->_pw_params = call_user_func(array(&$this, $this->_pw_form_parameter_callback), $this->_pw_params);
            }
        }
        foreach ($this->_pw_params as $key => $value) {
            $this->$key = $value;
        }
        // call initialize method
        $method = 'init';
        if (method_exists($this, $method)) {
            call_user_func(array(&$this, $method));
        }
        return true;
    }

    /**
     * Invoke prepare methods.
     * @access private
     * @return boolean
     */
    function _invokePrepareMethod()
    {
        $regex = "^prepare" . $this->_pw_action . "\$";
        $methods = get_class_methods($this);
        foreach ($methods as $method_name) {
            if (preg_match("/" . $regex . "/i", $method_name)) {
                call_user_func(array(&$this, $method_name));
            }
        }
        return true;
    }

    /**
     * Invoke action method.
     * @access private
     * @return integer
     */
    function _invokeMethod()
    {
        $method = 'execute' . $this->_pw_action;
        if (!method_exists($this, $method)) {
            return false;
        }
        if ($method == '') {
            return false;
        }
        return call_user_func(array(&$this, $method));
        //return PW_TERMINATE_PROCESS;
    }

    /**
     * @access private
     * @return boolean
     */
    function _applyFiltersBeforeMethod()
    {
        if (!is_array($this->_pw_around_filters)) {
            return false;
        }
        foreach ($this->_pw_around_filters as $filter) {
            $filter = strtolower(ltrim(rtrim($filter)));
            if (!class_exists($filter)) {
                $message = "Filter class '" . $filter . "' is not exists.";
                trigger_error($message, E_USER_NOTICE);
                continue;
            }
            $obj = new $filter();
            $this->_pw_around_filter_instances[] = $obj;

            if (!method_exists($obj, 'before')) {
                $message = "Filter method '" . $class . "#before' is not exists.";
                trigger_error($message, E_USER_NOTICE);
                continue;
            }
            call_user_func(array(&$obj, 'before'), $this);
        }
        return true;
    }

    /**
     * @access private
     * @return boolean
     */
    function _applyFiltersAfterMethod()
    {
        foreach (array_reverse($this->_pw_around_filter_instances) as $filter) {
            call_user_func(array(&$filter, 'after'), $this);
        }
        return true;
    }

    /**
     * Apply before filter methods.
     * @access private
     * @return boolean
     */
    function _applyBeforeFilters()
    {
        if (!is_array($this->_pw_before_filters)) {
            return false;
        }
        foreach ($this->_pw_before_filters as $filter) {
            $tmp = $this->_extractClassNameAndMethodName($filter);
            if (count($tmp) == 1) {
                // invoke filter method
                $method = $tmp[0];
                if (!method_exists($this, $method)) {
                    $message = "Filter method '" . $method . "' is not exists.";
                    trigger_error($message, E_USER_NOTICE);
                    continue;
                }
                call_user_func(array(&$this, $method));
            }
            else {
                // invoke filter class method
                $class = $tmp[0];
                $method = $tmp[1];
                if (!class_exists($class)) {
                    $message = "Filter class '" . $class . "' is not exists.";
                    trigger_error($message, E_USER_NOTICE);
                    continue;
                }
                $filter = new $class();
                if (!method_exists($filter, $method)) {
                    $message = "Filter method '" . $class . "#" . $method . "' is not exists.";
                    trigger_error($message, E_USER_NOTICE);
                    continue;
                }
                call_user_func(array(&$filter, $method), $this);
            }
        }
        return true;
    }

    /**
     * Apply after filter methods.
     * @access private
     * @return boolean
     */
    function _applyAfterFilters()
    {
        if (!is_array($this->_pw_after_filters)) {
            return false;
        }
        foreach ($this->_pw_after_filters as $filter) {
            $tmp = $this->_extractClassNameAndMethodName($filter);
            if (count($tmp) == 1) {
                // invoke filter method
                $method = $tmp[0];
                if (!method_exists($this, $method)) {
                    $message = "Filter method '" . $method . "' is not exists.";
                    trigger_error($message, E_USER_NOTICE);
                    continue;
                }
                call_user_func(array(&$this, $method));
            }
            else {
                // invoke filter class method
                $class = $tmp[0];
                $method = $tmp[1];
                if (!class_exists($class)) {
                    $message = "Filter class '" . $class . "' is not exists.";
                    trigger_error($message, E_USER_NOTICE);
                    continue;
                }
                $filter = new $class();
                if (!method_exists($filter, $method)) {
                    $message = "Filter method '" . $class . "#" . $method . "' is not exists.";
                    trigger_error($message, E_USER_NOTICE);
                    continue;
                }
                call_user_func(array(&$filter, $method), $this);
            }
        }
        return true;
    }

    /**
     * Assign controller properties to view.
     * @access private
     * @return array
     */
    function _getPropertiesAsPw()
    {
        $pwProperty = array();
        foreach (get_object_vars($this) as $key => $value) {
            if (preg_match('/^_.*/i', $key)) {
                $pwProperty[$key] = $value;
            }
        }
        return $pwProperty;
    }

    /**
     * Assign controller properties to view.
     * @access private
     * @return array
     */
    function _getPropertiesAsModel()
    {
        $model = array();
        foreach (get_object_vars($this) as $key => $value) {
            if (preg_match('/^_.*/i', $key)) {
                continue;
            }
            $model[$key] = $value;
        }
        return $model;
    }

    /**
     * コントローラのクラス変数のうち、変数名が _$section で始まるもの
     * の値を配列に格納して返します。配列のキーは変数名となります。
     * @access private
     * @param string $section
     * @return array
     */
    function _getClassVarsAsConfig($section)
    {
        $config = array();
        foreach (get_object_vars($this) as $key => $value) {
            if (preg_match('/^(_pw_' . $section . '_.*)/i', $key, $matches)) {
                $name = strtolower($matches[1]);
                $config[$name] = $value;
            }
        }
        return $config;
    }

    /**
     * @access private
     * @param string $src Source string
     * @return string
     */
    function _extractClassNameAndMethodName($src)
    {
        $buf = strtolower(ltrim(rtrim($src)));
        if (strpos($buf, '::') === false) {
            return array($buf);
        }
        return explode('::', $buf);
    }
}
