<?php
/* vim: set tabstop=4 shiftwidth=4: */

/*
 * Log Factory Implementation
 *
 *
 * PHP version 5
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   PHP
 * @package    Commons
 * @author     Yomei Komiya
 * @copyright  2008 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    SVN: $Id: LogFactoryImpl.php 251 2008-03-07 14:11:08Z whitestar $
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      File available since Release 0.9.0
 */

//namespace Commons::Logging::Impl;

require_once 'Commons/IO/FilenameUtils.php';
require_once 'Commons/Lang/ClassLoader.php';
require_once 'Commons/Logging/LogFactory.php';
/*
use Commons::IO::FilenameUtils;
use Commons::Lang::ClassLoader;
use Commons::Logging::LogFactory;
*/

/**
 * LogFactoryImpl
 *
 *
 * @category   PHP
 * @package    Commons.Logging.Impl
 * @author     Yomei Komiya
 * @copyright  2008 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 1.0.1
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 0.9.0
 */
class Commons_Logging_Impl_LogFactoryImpl extends Commons_Logging_LogFactory {

	/**
	 * The property name used to identify the Commons_Logging_Log implementation
	 * class name.
	 */
	const LOG_PROPERTY = 'commons.logging.Log';
	
	/**
	 * The fully qualified class name of the log interface.
	 */
	const LOG_INTERFACE = 'Commons_Logging_Log';
	
	/**
	 * The property name used to identify the Log4PHP configuration file.
	 */
	const LOG4PHP_CONF_PROPERTY = 'log4php.configuration';
	
	/**
	 * The property name used to identify the PEARLog configuration file.
	 */
	const PEARLOG_CONF_PROPERTY = 'pearlog.configuration';
	
	const PEARLOG_DEFAULT_CONF = 'Commons/Logging/Impl/pearlog_default.properties';
	
	const LOG4PHP_LOGGER_CLASS = 'Commons_Logging_Impl_Log4PHPLogger';
	
	const PEARLOG_LOGGER_CLASS = 'Commons_Logging_Impl_PEARLogLogger';
	
	const SIMPLELOG_LOGGER_CLASS = 'Commons_Logging_Impl_SimpleLog';
	
	/**
	 * Attributes
	 *
	 * @var array
	 */
	protected $attributes = array();
	
	/**
	 * Already instantiated logger objects. keyed by name.
	 *
	 * @var array
	 */
	protected $instances = array();
	
	/**
	 * The fully qualified class name of Commons_Logging_Log implementation.
	 *
	 * @var string
	 */
	private $_logClassName = null;
	
	/**
	 * The implementation class of the Commons_Logging_Log
	 *
	 * @var ReflectionClass
	 */
	protected $logClass = null;
	
	/**
	 * The one-argument setLogFactory method of the selected
	 * Commons_Logging_Log method, if it exists.
	 *
	 * @var ReflectionMethod
	 */
	protected $logMethod = null;
	
	
	/**
	 * Constructor
	 */
	public function Commons_Logging_Impl_LogFactoryImpl() {
		parent::__construct();
	}
	
	
	public function __destruct() {
	}
	
	
	public function getAttribute($name) {
		if (isset($this->attributes[$name])) {
			return $this->attributes[$name];
		}
		else {
			return null;
		}
	}
	
	
	public function getAttributeNames() {
		return array_keys($this->attributes);
	}
	
	
	public function getInstance($name) {
		if (!isset($this->instances[$name])) {
			$this->instances[$name] = $this->newInstance($name);
		}
		
		return $this->instances[$name];
	}
	
	
	public function release() {
		$this->instances = array();
	}
	
	
	public function removeAttribute($name) {
		unset($this->attributes[$name]);
	}
	
	
	public function setAttribute($name, $value) {
		if (is_null($value)) {
			unset($this->attributes[$name]);
		}
		else {
			$this->attributes[$name] = $value;
		}
	}
	
	
	/**
	 * Gets Log implementation class name.
	 *
	 * @return string class name.
	 * @throws Commons_Logging_LogConfigurationException
	 */
	protected function getLogClassName() {
		if (!is_null($this->_logClassName)) {
			return $this->_logClassName;
		}
		
		// 1. Explicitly defined
		$this->_logClassName = $this->getAttribute(self::LOG_PROPERTY);

		// 2. Environmental variable
		if (is_null($this->_logClassName)) {
			$tmp = getenv(self::LOG_PROPERTY);
			if ($tmp !== false) {
				$this->_logClassName = $tmp;
			}
		}
		// 3. Log4PHP
		if (is_null($this->_logClassName) && $this->isLog4PHPAvailable()) {
			$this->_logClassName = self::LOG4PHP_LOGGER_CLASS;
		}
		// 4. PEAR Log
		if (is_null($this->_logClassName) && $this->isPEARLogAvailable()) {
			$this->_logClassName = self::PEARLOG_LOGGER_CLASS;
			
		}
		// 5. SimpleLog	
		if (is_null($this->_logClassName)) {
			$this->_logClassName = self::SIMPLELOG_LOGGER_CLASS;
		}
		
		$this->initLogConfiguration($this->_logClassName);
		
		return $this->_logClassName;
	}
	
	
	/**
	 * Initializes logging framework configuration.
	 *
	 * @param string $logClassName logger class name.
	 */
	protected function initLogConfiguration($logClassName) {
		if ($logClassName === self::LOG4PHP_LOGGER_CLASS) {
			$log4PHPConf = $this->getAttribute(self::LOG4PHP_CONF_PROPERTY);
			if (!Commons_IO_FilenameUtils::isAbsolute($log4PHPConf)) {
				$log4PHPConf = Commons_Lang_ClassLoader::getResource($log4PHPConf);
			}
			if (!is_null($log4PHPConf) && !defined('LOG4PHP_CONFIGURATION')) {
				define('LOG4PHP_CONFIGURATION', $log4PHPConf);
			}
			else {
				// do nothing. Log4PHP default initializaton.
			}
		}
		elseif ($logClassName === self::PEARLOG_LOGGER_CLASS) {
			$PEARLogConf = $this->getAttribute(self::PEARLOG_CONF_PROPERTY);
			if (is_null($PEARLogConf)) {
				$this->setAttribute(self::PEARLOG_CONF_PROPERTY,
					self::PEARLOG_DEFAULT_CONF);
			}
		}
	}
	
	
	/**
	 * Obtains Commons_Logging_Log implementation's reflection class.
	 *
	 * @return ReflectionClass
	 * @throws Commons_Logging_LogConfigurationException
	 */
	protected function getLogClass() {
		if (!is_null($this->logClass)) {
			return $this->logClass;
		}
		
		try {
			$logClassName = $this->getLogClassName();
			
			if (!class_exists($logClassName, false)) {
				Commons_Lang_ClassLoader::load($logClassName);
			}
			$this->logClass = new ReflectionClass($logClassName);
			if (is_null($this->logClass)) {
				throw new Commons_Logging_LogConfigurationException(
					'No suitable Log implementation for ' . $logClassName);
			}
			if (!$this->logClass->implementsInterface(self::LOG_INTERFACE)) {
				throw new Commons_Logging_LogConfigurationException(
					'Class ' . $logClassName . ' does not implement \''
					. self::LOG_INTERFACE . '\'.');
			}
		}
		catch (Commons_Logging_LogConfigurationException $lce) {
			throw $lce;
		}
		catch (Exception $e) {
			throw new Commons_Logging_LogConfigurationException($e->getMessage());
		}
		
		try {
			$this->logMethod = $this->logClass->getMethod('setLogFactory');
		}
		catch (Exception $e) {
			$this->logMethod = null;
		}

		return $this->logClass;
	}
	
	
	/**
	 * Tests whether Log4PHP is available or not.
	 *
	 * @return bool true if Log4PHP is available.
	 */
	protected function isLog4PHPAvailable() {
		// check the resource existence.
		// but do not load the resource not to execute the Log4PHP implicit initialization.
		if (is_null(Commons_Lang_ClassLoader::getResource(
			'log4php/LoggerManager.php'))) {
			return false;
		}
		else {
			return true;
		}
	}
	

	/**
	 * Tests whether PEARLog is available or not.
	 *
	 * @return bool true if PEARLog is available.
	 */
	protected function isPEARLogAvailable() {
		$ret = include_once 'Log.php';
		if ($ret === false) {
			return false;
		}
		else {
			return true;
		}
	}
	
	
	/**
	 * Creates and returns Commons_Logging_Log instance
	 *
	 * @param string $name new logger name.
	 * @return Commons_Logging_Log new logger.
	 * @throws Commons_Logging_LogConfigurationException
	 */
	protected function newInstance($name) {
		$instance = null;
		try {
			// Commons_Logging_Log implementation must have one-argument constructor.
			$instance = $this->getLogClass()->newInstance($name);
			if (!is_null($this->logMethod)) {
				$params = array($this);
				$this->logMethod->invokeArgs($instance, $params);
			}
			return $instance;
		}
		catch (Commons_Logging_LogConfigurationException $lce) {
			throw $lce;
		}
		catch (Exception $e) {
			throw new Commons_Logging_LogConfigurationException($e->getMessage());
		}
	}

}

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * c-hanging-comment-ender-p: nil
 * End:
 */
?>