<?php
/* vim: set tabstop=4 shiftwidth=4: */

/*
 * Stop Watch
 * (based on org.apache.commons.lang.time.StopWatch)
 *
 * PHP version 5
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   PHP
 * @package    Commons
 * @author     Yomei Komiya
 * @copyright  2007 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    SVN: $Id: StopWatch.php 97 2008-01-04 12:58:32Z whitestar $
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      File available since Release 0.8.0
 */

//namespace Commons::Lang::Time;

require_once 'Commons/Lang/Object.php';
require_once 'Commons/Lang/RuntimeException.php';
require_once 'Commons/Lang/IllegalStateException.php';
require_once 'Commons/Lang/Time/MicrosecondTime.php';
require_once 'Commons/Lang/Time/DurationFormatUtils.php';
/*
use Commons::Lang::Object;
use Commons::Lang::RuntimeException;
use Commons::Lang::IllegalStateException;
use Commons::Lang::Time::MicrosecondTime;
use Commons::Lang::Time::DurationFormatUtils;
*/

/**
 * StopWatch
 *
 *
 * @category   PHP
 * @package    Commons.Lang.Time
 * @author     Yomei Komiya
 * @copyright  2007 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 1.0.2
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 0.8.0
 */
class Commons_Lang_Time_StopWatch extends Commons_Lang_Object {
//class StopWatch extends Object {
	
	// running states
	const STATE_UNSTARTED = 0;
	const STATE_RUNNING   = 1;
	const STATE_STOPPED   = 2;
	const STATE_SUSPENDED = 3;

	// split state
	const STATE_UNSPLIT = 10;
	const STATE_SPLIT   = 11;

	/**
	 * The current running state of the StopWatch.
	 * @var int
	 */
	private $_runningState = self::STATE_UNSTARTED;

	/**
	 * Whether the stopwatch has a split time recorded.
	 * @var int
	 */
	private $_splitState   = self::STATE_UNSPLIT;
	
	/**
	 * @var Commons_Lang_Time_MicrosecondTime Start time (microsecond) object
	 */
	private $_startTime = null;
	
	/**
	 * @var Commons_Lang_Time_MicrosecondTime Stop time (microsecond) object
	 */
	private $_stopTime = null;
	
	
	/**
	 * Constructor
	 */
	public function Commons_Lang_Time_StopWatch() {
		parent::__construct();
	}
	
	
	public function __destruct() {
		// do nothing.
	}
	
	
	/**
	 * Starts stopwatch
	 * 
	 * @throws Commons_Lang_IllegalStateException
	 */
	public function start() {
		if ($this->_runningState === self::STATE_STOPPED) {
			throw new Commons_Lang_IllegalStateException(
				'Stopwatch must be reset before being restarted.');
		}
		if ($this->_runningState !== self::STATE_UNSTARTED) {
			throw new Commons_Lang_IllegalStateException('Stopwatch already started.');
		}
		$this->_stopTime = null;
		$this->_startTime = new Commons_Lang_Time_MicrosecondTime();
		$this->_runningState = self::STATE_RUNNING;
	}


	/**
	 * Stops stopwatch
	 * 
	 * @throws Commons_Lang_IllegalStateException if the StopWatch is not running.
	 */
	public function stop() {
		if ($this->_runningState !== self::STATE_RUNNING
			&& $this->_runningState !== self::STATE_SUSPENDED) {
			throw new Commons_Lang_IllegalStateException('Stopwatch is not running.');
		}
		$this->_stopTime = new Commons_Lang_Time_MicrosecondTime();
		$this->_runningState = self::STATE_STOPPED;
	}


	/**
	 * Resets stopwatch
	 */
	public function reset() {
		$this->_runningState = self::STATE_UNSTARTED;
		$this->_splitState   = self::STATE_UNSPLIT;
		$this->_startTime = null;
		$this->_stopTime  = null;
	}
	
	
	/**
	 * Splits stopwatch
	 * 
	 * @throws Commons_Lang_IllegalStateException if the StopWatch is not running.
	 */
	public function split() {
		if ($this->_runningState !== self::STATE_RUNNING) {
			throw new Commons_Lang_IllegalStateException('Stopwatch is not running.');
		}
		$this->_stopTime = new Commons_Lang_Time_MicrosecondTime();
		$this->_splitState = self::STATE_SPLIT;
	}
	
	
	/**
	 * Unsplits stopwatch
	 * 
	 * @throws Commons_Lang_IllegalStateException if the StopWatch has not been split yet.
	 */
	public function unsplit() {
		if ($this->_splitState !== self::STATE_SPLIT) {
			throw new Commons_Lang_IllegalStateException('Stopwatch has not been split.');
		}
		$this->_stopTime = null;
		$this->_splitState = self::STATE_UNSPLIT;
	}
	
	
	/**
	 * Suspends stopwatch
	 * 
	 * @throws Commons_Lang_IllegalStateException if the StopWatch is not running.
	 */
	public function suspend() {
		if ($this->_runningState !== self::STATE_RUNNING) {
			throw new Commons_Lang_IllegalStateException(
				'Stopwatch must be running to suspend.');
		}
		$this->_stopTime = new Commons_Lang_Time_MicrosecondTime();
		$this->_runningState = self::STATE_SUSPENDED;
	}


	/**
	 * Resumes stopwatch
	 * 
	 * @throws Commons_Lang_IllegalStateException if the StopWatch has not been suspended yet.
	 */
	public function resume() {
		if ($this->_runningState !== self::STATE_SUSPENDED) {
			throw new Commons_Lang_IllegalStateException(
				'Stopwatch must be suspended to resume.');
		}
		$currentTime = new Commons_Lang_Time_MicrosecondTime();
		//startTime += (System.currentTimeMillis() - stopTime);
		$this->_startTime
			= $this->_startTime->add($currentTime->subtract($this->_stopTime));
		$this->_stopTime = null;
		$this->_runningState = self::STATE_RUNNING;
	}


	/**
	 * Gets measurement time (millisecond)
	 * 
	 * @return float measurement time (millisecond)
	 * @throws Commons_Lang_RuntimeException
	 */
	public function getTime() {
		return $this->getMicrosecondTime()->toMillis();
	}
	
	
	/**
	 * Gets measurement time (microsecond)
	 *
	 * @return Commons_Lang_Time_MicrosecondTime measurement time (object)
	 * @throws Commons_Lang_RuntimeException
	 */
	public function getMicrosecondTime() {
		if ($this->_runningState === self::STATE_STOPPED
			|| $this->_runningState === self::STATE_SUSPENDED) {
			return $this->_stopTime->subtract($this->_startTime);
		}
		elseif ($this->_runningState === self::STATE_UNSTARTED) {
			return new Commons_Lang_Time_MicrosecondTime(0.0, 0.0);
		}
		elseif ($this->_runningState === self::STATE_RUNNING) {
			$currentMicrotime = new Commons_Lang_Time_MicrosecondTime();
			return $currentMicrotime->subtract($this->_startTime);
		}
		throw new Commons_Lang_RuntimeException('Illegal running state has occurred.');
	}


	/**
	 * Gets the split time (millisecond)
	 * 
	 * @return float split time (millisecond)
	 * @throws Commons_Lang_IllegalStateException
	 */
	public function getSplitTime() {
		return $this->getSplitMicrosecondTime()->toMillis();
	}

	
	/**
	 * Gets the split time (microsecond)
	 *
	 * @return Commons_Lang_Time_MicrosecondTime split time (object)
	 * @throws Commons_Lang_IllegalStateException
	 */
	public function getSplitMicrosecondTime() {
		if ($this->_splitState !== self::STATE_SPLIT) {
			throw new Commons_Lang_IllegalStateException(
				'Stopwatch must be split to get the split time.');
		}
		return $this->_stopTime->subtract($this->_startTime);
	}
	
	
	/**
	 * Returns the string representation (H:mm:ss.SSS) of elapsed time
	 *
	 * @return string
	 */
	public function __toString() {
		return Commons_Lang_Time_DurationFormatUtils::formatDurationHMS($this->getTime());
	}
	
	
	/**
	 * __toString alias method
	 *
	 * @return string
	 */
	public function toString() {
		return $this->__toString();
	}
	
	
	/**
	 * Returns the string representation (H:mm:ss.SSS) of the split time
	 *
	 * @return string
	 */
	public function toSplitString() {
		return Commons_Lang_Time_DurationFormatUtils
			::formatDurationHMS($this->getSplitTime());
	}
	
}

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * c-hanging-comment-ender-p: nil
 * End:
 */
?>