<?php
/* vim: set tabstop=4 shiftwidth=4: */

/*
 * Abstract S2Container DataSource Test Case
 * (based on S2Container_S2PHPUnit2TestCase and org.seasar.extension.unit.S2TestCase)
 *
 *
 * PHP version 5
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   PHP
 * @package    Commons
 * @author     Yomei Komiya
 * @copyright  2008 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    SVN: $Id: S2DataSourceTestCase.php 329 2008-08-03 06:11:46Z whitestar $
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      File available since Release 1.0.2
 */

//namespace Commons::Test;

require_once 'PHPUnit/Extensions/Database/TestCase.php';

require_once 'Commons/Lang/StringUtils.php';
require_once 'Commons/S2Container/ExtendedS2ContainerFactory.php';
/*
use Commons::Lang::StringUtils;
use Commons::S2Container::ExtendedS2ContainerFactory;
*/

/**
 * S2DataSourceTestCase
 *
 *
 * @category   PHP
 * @package    Commons.Test
 * @author     Yomei Komiya
 * @copyright  2008 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 1.0.2
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 1.0.2
 */
abstract class Commons_Test_S2DataSourceTestCase extends PHPUnit_Extensions_Database_TestCase {

	/**
	 * @var S2Container
	 */
	private $container = null;
	
	private $boundFields = array();
	
	/**
	 * @var S2Container_DataSource
	 */
	private $dataSource = null;
	
	/**
	 * @var PDO
	 */
	private $pdoConnection = null;
	
	/**
	 * @var PHPUnit_Extensions_Database_DB_IMetaData
	 */
	private $databaseMetaData = null;
	
	
	/**
	 * This implementation is final. Override onSetUp() method in place of this method.
	 */
	final protected function setUp() {
		$this->setUpContainer();
		$this->onSetUp();
		$this->setUpForEachTestMethod();
		$this->container->init();
		$this->setUpAfterContainerInit();
		$this->bindFields();
		$this->setUpAfterBindFields();
		
		if ($this->needTransaction()) {
			$this->pdoConnection->beginTransaction();
		}
		
		// Database test case setup depends on S2Container's setup.
		parent::setUp();
	}
	

	/**
	 * Overridden runTest.
	 * 
	 * @throws RuntimeException
	 */
	protected function runTest() {
		parent::runTest();
	}
	
	
	/**
	 * This implementation is final. Override onTearDown() method in place of this method.
	 */
	final protected function tearDown() {
		parent::tearDown();
		
		if ($this->needTransaction() && !is_null($this->pdoConnection)) {
			$this->pdoConnection->rollBack();
		}
		
		$this->tearDownBeforeUnbindFields();
		$this->unbindFields();
		$this->tearDownBeforeContainerDestroy();
		$this->tearDownContainer();
		$this->tearDownForEachTestMethod();
		$this->onTearDown();
	}
	
	
	/**
	 * @return S2Container
	 */
	public function getContainer() {
		return $this->container;
	}

	
	/**
	 * @param string $componentName
	 * @return object
	 * @see S2Container#getComponent()
	 */
	public function getComponent($componentName) {
		return $this->container->getComponent($componentName);
	}

	
	/**
	 * @param string $componentName
	 * @return S2Container_ComponentDef
	 * @see S2Container#getComponentDef()
	 */
	public function getComponentDef($componentName) {
		return $this->container->getComponentDef($componentName);
	}
	

	/**
	 * @param object $component
	 * @param string $componentName
	 * @see S2Container#register()
	 */
	public function register($component, $componentName = ''){
		$this->container->register($component, $componentName);
	}
	
	
	/**
	 * @param string $path
	 */
	protected function includeDicon($diconFullPath) {
		S2ContainerFactory::includeChild($this->container, $diconFullPath);
	}


	protected function includeDiconOnIncludePath($diconResourceName) {
		Commons_S2Container_ExtendedS2ContainerFactory
			::includeChildFromDiconOnIncludePath($this->container, $diconResourceName);
	}
	
	
	protected function setUpContainer() {
		$this->container = new S2ContainerImpl();
		S2Container_SingletonS2ContainerFactory::setContainer($this->container);
	}
	

	protected function setUpForEachTestMethod() {
		$targetName = $this->getTargetName();
		if ($targetName !== '') {
			$this->invoke('setUp' . $targetName);
		}
	}
	

	/**
	 * Subclasses can override this method.
	 * e.g. includeDicon()
	 */
	protected function onSetUp() {
	}
	
	
	protected function setUpAfterContainerInit() {
		$this->setupDataSource();
	}

	
	protected function setupDataSource() {
		try {
			$dataSourceNameOrType = null;
			
			if (Commons_Lang_StringUtils::isNotEmpty($this->getDataSourceName())) {
				$dataSourceNameOrType = $this->getDataSourceName();
			}
			elseif ($this->container->hasComponentDef('S2Container_DataSource')) {
				$dataSourceNameOrType = 'S2Container_DataSource';
			}

			$this->dataSource = $this->container->getComponent($dataSourceNameOrType);
			$this->pdoConnection = $this->dataSource->getConnection();
		}
		catch (Exception $e) {
			//
		}
	}
	
	
	protected function setUpAfterBindFields() {
	}
	
	
	protected function tearDownBeforeUnbindFields() {
	}


	protected function tearDownBeforeContainerDestroy() {
		$this->tearDownDataSource();
	}

	
	protected function tearDownDataSource() {
		if (!is_null($this->pdoConnection)) {
			$this->pdoConnection = null;
        }
		
		$this->dataSource = null;
	}
	
	
	protected function tearDownContainer() {
		$this->container->destroy();
		S2Container_SingletonS2ContainerFactory::setContainer(null);
		$this->container = null;
	}
	

	protected function tearDownForEachTestMethod() {
		$targetName = $this->getTargetName();
		if ($targetName !== '') {
			$this->invoke('tearDown' . $targetName);
		}
	}
	
	
	protected function onTearDown() {
	}
	
	
	private function bindFields() {
		if ($this->isAutoBindable()) {
			$ref = new ReflectionClass($this);
			$props = $ref->getProperties();
			foreach ($props as $prop) {
				if (!$this->isFrameworkProperty($prop)) {
					$this->bindField($prop);
				}
			}
		}
	}
	

	private function unbindFields() {
		foreach ($this->boundFields as $field) {
			$this->unbindField($field);
		}
		$this->boundFields = array();
	}
	

	private function bindField(ReflectionProperty $field) {
		$propName = $field->getName();
		
		if (!$this->getContainer()->hasComponentDef($propName)) {
			return;
		}
		
		if ($this->isAccessibleProperty($field)) {
			$field->setValue($this, $this->getComponent($propName));
			$this->boundFields[] = $field;
		}
		elseif ($this->hasCallableSetter($field)) {
			$setter = 'set' . $propName;
			$this->$setter($this->getComponent($propName));
			$this->boundFields[] = $field;
		}
	}
	

	private function unbindField(ReflectionProperty $field) {
		$propName = $field->getName();
		
		if ($this->isAccessibleProperty($field)) {
			$field->setValue($this, null);
		}
		elseif ($this->hasCallableSetter($field)) {
			$setter = 'set' . $propName;
			$this->$setter(null);
		}
	}
	

	/**
	 * Whether this test case is auto-bindable or not.
	 * Subclasses can override this method.
	 *
	 * @return bool
	 */
	protected function isAutoBindable() {
		return true;
	}
	
	
	private function isAccessibleProperty(ReflectionProperty $field) {
		return $this->isPublicAndNonStaticProperty($field);
	}
	
	
	private function isPublicAndNonStaticProperty(ReflectionProperty $field) {
		return $field->isPublic() && !$field->isStatic();
	}
	
	
	private function isFrameworkProperty(ReflectionProperty $field) {
		return preg_match(
			'/^(PHPUnit|Commons_Test)_/',
			$field->getDeclaringClass()->getName());
	}
	
	
	private function hasCallableSetter(ReflectionProperty $field) {
		return is_callable(array($this, 'set' . $field->getName()));
	}
	

	private function invoke($methodName) {
		try {
			$method = S2Container_ClassUtil::getMethod(new ReflectionClass($this), $methodName);
			S2Container_MethodUtil::invoke($method, $this, null);
		}
		catch (S2Container_NoSuchMethodRuntimeException $ignore) {
			//print "invoke ignored. [$methodName]\n";
		}
	}

	
	private function getTargetName() {
		return substr($this->getName(), 4);	// testTargetName()
	}
	
	
	protected function needTransaction() {
        return Commons_Lang_StringUtils::endsWith($this->getName(), 'Tx');
    }
    
    
    /**
     * Subclasses can override this method to specify DataSource name or type.
     * If this method is not overridden, the SINGLETON DataSource object is used automatically.
     *
     * @return string DataSource name or type.
     */
	protected function getDataSourceName() {
		return null;
	}
	
	
	/**
	 * Gets the data source object.
	 *
	 * @return S2Container_DataSource
	 */
	protected function getDataSource() {
		return $this->dataSource;
	}
	
	
	/**
	 * Subclasses can override this method to specify database schema
	 *
	 * @return PHPUnit_Extensions_Database_DB_IDatabaseConnection
	 */
	protected function getConnection() {
		return $this->createDefaultDBConnection($this->pdoConnection, $this->getSchemaName());
	}
	
	
	/**
	 * Gets the database's meta data object.
	 *
	 * @return PHPUnit_Extensions_Database_DB_IMetaData
	 */
	protected function getDatabaseMetaData() {
		if (is_null($this->databaseMetaData)) {
			$this->databaseMetaData = $this->getConnection()->getMetaData();
		}
		
		return $this->databaseMetaData; 
	}
	
	
	/**
	 * Subclasses must implement this method.
	 * Note: database name in MySQL
	 * 
	 * @return string schema name.
	 */
	abstract protected function getSchemaName();
	
}

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * c-hanging-comment-ender-p: nil
 * End:
 */
?>