/* -*- Mode: c; c-basic-offset: 8; Coding: utf-8-unix -*- ;; */
/* $Id: cmd.c 2 2008-05-27 07:44:27Z mtaneda $	 */

/*
 * Copyright 2008 The piranha Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PIRANHA PROJECT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE PIRANHA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the piranha Project.
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<unistd.h>
#include	"strtool.h"
#include	"pairlist.h"
#include	"sock.h"
#include	"cmd.h"

#define	DELIM	": "
#define	EOL	"\x0d\x0a"


struct cmd
{
	int             type;
	pairlist_t     *header;
	char           *body;
};

int             cmd_read_type(int fd, cmd_t * obj);
int             cmd_read_header(int fd, cmd_t * obj);
int             cmd_read_body(int fd, cmd_t * obj);
int             cmd_get_type_by_str(const char *str);

cmd_t          *
cmd_create(void)
{
	cmd_t          *obj = NULL;

	obj = (cmd_t *) calloc(1, sizeof(cmd_t));
	if (!obj)
	{
		goto ERROR;
	}
	obj->header = pairlist_create();
	if (!obj->header)
	{
		goto ERROR;
	}
	return (obj);
ERROR:
	cmd_destroy(obj);
	return (NULL);
}

int 
cmd_destroy(cmd_t * obj)
{
	if (!obj)
	{
		goto ERROR;
	}
	pairlist_destroy(obj->header);
	free(obj->body);
	free(obj);

	return (0);
ERROR:
	return (1);
}

int 
cmd_set_type(cmd_t * obj, const int type)
{
	if (!obj || obj->type != 0 || type == 0 || type > CMD_TYPE_SCENARIO)
	{
		goto ERROR;
	}
	obj->type = type;

	return (0);
ERROR:
	return (1);
}

int 
cmd_set_header(cmd_t * obj, const char *name, const char *value)
{
	int             err;

	if (!obj || strtool_isempty(name) || strtool_isempty(value))
	{
		goto ERROR;
	}
	if (pairlist_is_exist(obj->header, name))
	{
		goto ERROR;
	}
	err = pairlist_push(obj->header, name, value);

	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

int 
cmd_set_body(cmd_t * obj, const char *body, const int size)
{
	char            value[32];
	int             err;

	if (!obj || obj->body || !body || size == 0)
	{
		goto ERROR;
	}
	obj->body = (char *) calloc(size, sizeof(char));

	memcpy(obj->body, body, size);

	memset(value, 0, sizeof(value));
	sprintf(value, "%d", size);

	err = cmd_set_header(obj, "Content-Length", value);

	if (err)
	{
		free(obj->body);
		obj->body = NULL;
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

int 
cmd_get_type(const cmd_t * obj)
{
	if (!obj)
	{
		goto ERROR;
	}
	return (obj->type);
ERROR:
	return (0);
}

char           *
cmd_refer_header_value_by_name(const cmd_t * obj, const char *name)
{
	if (!obj)
	{
		goto ERROR;
	}
	return (pairlist_refer_value_by_name(obj->header, name));
ERROR:
	return (NULL);
}

char           *
cmd_get_header_value_by_name(const cmd_t * obj, const char *name)
{
	if (!obj)
	{
		goto ERROR;
	}
	return (pairlist_get_value_by_name(obj->header, name));
ERROR:
	return (NULL);
}

int 
cmd_refer_body(const cmd_t * obj, char **body)
{
	char           *tmp = NULL;
	int             len;

	if (!obj)
	{
		goto ERROR;
	}
	tmp = cmd_refer_header_value_by_name(obj, "Content-Length");
	if (!tmp)
	{
		goto ERROR;
	}
	len = atoi(tmp);

	*body = obj->body;

	return (len);
ERROR:
	return (0);
}

int 
cmd_get_body(const cmd_t * obj, char **body)
{
	char           *tmp = NULL;
	int             len;

	len = cmd_refer_body(obj, &tmp);

	if (len <= 0)
	{
		goto ERROR;
	}
	*body = (char *) calloc(len, sizeof(char));

	memcpy(*body, tmp, len);

	return (len);
ERROR:
	*body = NULL;
	return (0);
}

char           *
cmd_to_str_type(const cmd_t * obj)
{
	char           *dst = NULL;

	if (!obj)
	{
		goto ERROR;
	}
	switch (obj->type)
	{
	case CMD_TYPE_START:
		dst = strdup("START");
		break;
	case CMD_TYPE_STOP:
		dst = strdup("STOP");
		break;
	case CMD_TYPE_PAUSE:
		dst = strdup("PAUSE");
		break;
	case CMD_TYPE_RESUME:
		dst = strdup("RESUME");
		break;
	case CMD_TYPE_DATA:
		dst = strdup("DATA");
		break;
	case CMD_TYPE_CONFIG:
		dst = strdup("CONFIG");
		break;
	case CMD_TYPE_SCENARIO:
		dst = strdup("SCENARIO");
		break;
	default:
		goto ERROR;
	}

	return (dst);
ERROR:
	return (NULL);
}

char           *
cmd_to_str_header(const cmd_t * obj)
{
	char           *dst = NULL;
	char           *line;
	char           *name, *value;
	int             n, len;

	if (!obj)
	{
		goto ERROR;
	}
	len = pairlist_len(obj->header);

	if (len == 0)
	{
		line = strtool_sprintf("Content-Length%s0%s", DELIM, EOL);

		strtool_cat(&dst, line);

		free(line);
	}
	for (n = 0; n < len; n++)
	{
		pairlist_refer(obj->header, n, &name, &value);

		if (!name || !value)
		{
			goto ERROR;
		}
		line = strtool_sprintf("%s%s%s%s", name, DELIM, value, EOL);

		strtool_cat(&dst, line);

		free(line);
	}

	return (dst);
ERROR:
	free(dst);
	return (NULL);
}

cmd_t          *
cmd_dup(const cmd_t * obj)
{
	cmd_t          *dst = NULL;
	char           *tmp;

	if (!obj)
	{
		goto ERROR;
	}
	dst = cmd_create();
	if (!dst)
	{
		goto ERROR;
	}
	cmd_get_body(obj, &tmp);

	dst->type = obj->type;
	dst->header = pairlist_dup(obj->header);
	dst->body = tmp;


	return (dst);
ERROR:
	cmd_destroy(dst);
	return (NULL);
}

int 
cmd_write(int fd, const cmd_t * obj)
{
	char           *dst = NULL;
	char           *tmp;
	int             len, len2;

	if (!obj)
	{
		goto ERROR;
	}
	tmp = cmd_to_str_type(obj);
	strtool_cat(&dst, tmp);
	strtool_cat(&dst, EOL);
	free(tmp);

	tmp = cmd_to_str_header(obj);
	strtool_cat(&dst, tmp);
	strtool_cat(&dst, EOL);
	free(tmp);

	len = strlen(dst);

	len2 = cmd_get_body(obj, &tmp);

	if (len2)
	{
		dst = (char *) realloc(dst, len + len2);
		memcpy(dst + len, tmp, len2);
		len += len2;
		free(tmp);
	}
	if (write(fd, dst, len) < 0)
	{
		goto ERROR;
	}
	free(dst);

	return (0);
ERROR:
	free(dst);
	return (1);
}

cmd_t          *
cmd_read(int fd)
{
	cmd_t          *obj = NULL;
	int             err;

	obj = cmd_create();
	if (!obj)
	{
		goto ERROR;
	}
	err = cmd_read_type(fd, obj);
	if (err)
	{
		goto ERROR;
	}
	err = cmd_read_header(fd, obj);
	if (err)
	{
		goto ERROR;
	}
	if (obj->type > 0x04)
	{
		err = cmd_read_body(fd, obj);
		if (err)
		{
			goto ERROR;
		}
	}
	return (obj);
ERROR:
	cmd_destroy(obj);
	return (NULL);
}

int 
cmd_read_type(int fd, cmd_t * obj)
{
	char           *line = NULL;
	int             len;

	if (!obj)
	{
		goto ERROR;
	}
	len = read_block(fd, EOL, &line);

	obj->type = cmd_get_type_by_str(line);

	if (obj->type == 0)
	{
		goto ERROR;
	}
	free(line);

	return (0);
ERROR:
	free(line);
	return (1);
}

int 
cmd_read_header(int fd, cmd_t * obj)
{
	char           *data = NULL;
	strlist_t      *lines = NULL;
	strlist_t      *line = NULL;
	int             err;
	int             n, len;

	if (!obj)
	{
		goto ERROR;
	}
	read_block(fd, EOL EOL, &data);

	lines = strtool_split(data, EOL, 0);

	len = strlist_len(lines);

	if (len == 0)
	{
		goto ERROR;
	}
	for (n = 0; n < len; n++)
	{
		line = strtool_split(strlist_refer(lines, n), DELIM, 1);

		if (strlist_len(line) != 2)
		{
			strlist_destroy(line);
			goto ERROR;
		}
		err = cmd_set_header(obj, strlist_refer(line, 0), strlist_refer(line, 1));

		if (err)
		{
			strlist_destroy(line);
			goto ERROR;
		}
		strlist_destroy(line);
	}

	strlist_destroy(lines);
	free(data);

	return (0);
ERROR:
	strlist_destroy(lines);
	free(data);
	return (1);
}

int 
cmd_read_body(int fd, cmd_t * obj)
{
	char           *tmp;
	char           *body = NULL;
	int             len;

	if (!obj)
	{
		goto ERROR;
	}
	tmp = cmd_refer_header_value_by_name(obj, "Content-Length");

	if (!tmp || !strtool_isdigit(tmp) || strcmp(tmp, "0") == 0)
	{
		goto ERROR;
	}
	len = atoi(tmp);

	body = (char *) calloc(len, sizeof(char));

	if (read(fd, body, len) != len)
	{
		goto ERROR;
	}
	obj->body = body;

	return (0);
ERROR:
	free(body);
	return (1);
}

int 
cmd_get_type_by_str(const char *str)
{
	int             type = 0;
	char           *tmp = NULL;
	char           *p;

	if (strtool_isempty(str))
	{
		goto ERROR;
	}
	tmp = strdup(str);

	if ((p = strstr(tmp, EOL)))
	{
		*p = '\0';
	}
	if (strcmp(tmp, "START") == 0)
		type = CMD_TYPE_START;
	else if (strcmp(tmp, "STOP") == 0)
		type = CMD_TYPE_STOP;
	else if (strcmp(tmp, "PAUSE") == 0)
		type = CMD_TYPE_PAUSE;
	else if (strcmp(tmp, "RESUME") == 0)
		type = CMD_TYPE_RESUME;
	else if (strcmp(tmp, "DATA") == 0)
		type = CMD_TYPE_DATA;
	else if (strcmp(tmp, "CONFIG") == 0)
		type = CMD_TYPE_CONFIG;
	else if (strcmp(tmp, "SCENARIO") == 0)
		type = CMD_TYPE_SCENARIO;
	else
		type = 0;

ERROR:
	free(tmp);
	return (type);
}
