/* -*- Mode: c; c-basic-offset: 8; Coding: utf-8-unix -*- ;; */
/* $Id: dlist.c 2 2008-05-27 07:44:27Z mtaneda $	 */

/*
 * Copyright 2008 The piranha Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PIRANHA PROJECT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE PIRANHA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the piranha Project.
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	"dlist.h"

struct _dlist_
{
	void           *data;
	struct _dlist_ *prev;
	struct _dlist_ *next;
};

struct _dlist_info_
{
	struct _dlist_ *head;
	struct _dlist_ *tail;
	int             len;
};

dlist_t        *
dlist_create(void)
{
	dlist_t        *info = NULL;

	info = (dlist_t *) calloc(1, sizeof(dlist_t));
	if (!info)
	{
		goto ERROR;
	}
	info->head = NULL;
	info->tail = NULL;
	info->len = 0;

	return (info);
ERROR:
	return (NULL);
}

struct _dlist_ *
dlist_calloc(void)
{
	struct _dlist_ *obj = NULL;

	obj = (struct _dlist_ *) calloc(1, sizeof(struct _dlist_));
	if (!obj)
	{
		goto ERROR;
	}
	obj->data = NULL;
	obj->prev = NULL;
	obj->next = NULL;

	return (obj);
ERROR:
	return (NULL);
}

int 
dlist_destroy(dlist_t * info, void (*callback) (void *))
{
	void           *data;

	if (!info || !callback)
	{
		goto ERROR;
	}
	while (info->len > 0)
	{
		data = dlist_pop(info);

		callback(data);
	}

	free(info);

	return (0);
ERROR:
	return (1);
}

void           *
dlist_pop(dlist_t * info)
{
	struct _dlist_ *tail = NULL;
	void           *data = NULL;

	if (!info || !info->len)
	{
		goto ERROR;
	}
	data = info->tail->data;

	tail = info->tail;

	info->len -= 1;

	switch (info->len)
	{
	case 0:
		info->head = NULL;
		info->tail = NULL;
		break;
	case 1:
		info->tail = info->head;
		info->tail->next = NULL;
		break;
	default:
		info->tail = info->tail->prev;
		info->tail->next = NULL;
	}

	free(tail);

	return (data);
ERROR:
	return (NULL);
}

int 
dlist_push(dlist_t * info, const void *data)
{
	struct _dlist_ *obj = NULL;

	if (!info || !data)
	{
		goto ERROR;
	}
	obj = dlist_calloc();
	if (!obj)
	{
		goto ERROR;
	}
	obj->data = (void *) data;

	if (info->len == 0)
	{
		info->head = obj;
		info->tail = obj;
	} else
	{
		info->tail->next = obj;
		obj->prev = info->tail;
		info->tail = obj;
	}

	info->len += 1;

	return (0);
ERROR:
	return (1);
}

void           *
dlist_dequeue(dlist_t * info)
{
	void           *data;
	struct _dlist_ *head;

	if (!info || info->len < 1)
	{
		goto ERROR;
	}
	data = info->head->data;

	info->len -= 1;

	head = info->head;

	switch (info->len)
	{
	case 0:
		info->head = NULL;
		info->tail = NULL;
		break;
	case 1:
		info->head = info->tail;
		info->head->prev = NULL;
		break;
	default:
		info->head = info->head->next;
		info->head->prev = NULL;
	}

	free(head);

	return (data);
ERROR:
	return (NULL);
}

int 
dlist_queue(dlist_t * info, const void *data)
{
	return (dlist_push(info, data));
}
/*
{
struct _dlist_	*obj = NULL;

if(!info || !data)
{
	goto ERROR;
}

obj = dlist_calloc();
if(!obj)
{
	goto ERROR;
}

obj->data = (void *)data;

if(info->len == 0)
{
	info->head = obj;
	info->tail = obj;
	obj->prev  = NULL;
	obj->next  = NULL;
}
else
{
	obj->prev        = NULL;
	obj->next        = info->head;
	info->head->prev = obj;
	info->head       = obj;
}

info->len += 1;

return (0);
ERROR:
return (1);
}
*/

int 
dlist_add_next(dlist_t * info, const int offset, const void *data)
{
	struct _dlist_ *p, *obj = NULL;
	int             n, len;

	if (!info || !data || offset < 0)
	{
		goto ERROR;
	}
	len = dlist_len(info);

	if (offset > len - 1)
	{
		goto ERROR;
	} else if (offset == len - 1)
	{
		return (dlist_push(info, data));
	}
	p = info->head;
	for (n = 0; n < offset; n++)
	{
		p = p->next;

		if (!p)
		{
			goto ERROR;
		}
	}

	obj = dlist_calloc();
	if (!obj)
	{
		goto ERROR;
	}
	obj->data = (void *) data;
	obj->prev = p;
	obj->next = p->next;

	p->next->prev = obj;
	p->next = obj;

	info->len += 1;

	return (0);
ERROR:
	return (1);
}

int 
dlist_del_data_by_offset(
			 dlist_t * info,
			 const int offset,
			 void (*callback) (void *))
{
	struct _dlist_ *obj;
	void           *data;
	int             n;

	if (!info || offset < 0 || offset > info->len)
	{
		goto ERROR;
	}
	if (offset == 0)
	{
		data = dlist_dequeue(info);
		if (!data)
		{
			goto ERROR;
		}
	} else if (offset == info->len - 1)
	{
		data = dlist_pop(info);
		if (!data)
		{
			goto ERROR;
		}
	} else
	{
		obj = info->head;
		for (n = 0; n < offset && obj; n++)
		{
			obj = obj->next;
		}

		obj->prev->next = obj->next;
		obj->next->prev = obj->prev;

		data = obj->data;

		free(obj);
		info->len -= 1;
	}

	callback(data);


	return (0);
ERROR:
	return (1);
}

void           *
dlist_get_data_by_offset(const dlist_t * info, const int offset)
{
	struct _dlist_ *obj;
	int             n;

	if (!info || offset < 0 || offset > info->len - 1)
	{
		goto ERROR;
	}
	obj = info->head;

	for (n = 0; n < offset && obj; n++)
	{
		obj = obj->next;
	}

	return (obj->data);
ERROR:
	return (NULL);
}

int 
dlist_len(const dlist_t * info)
{
	if (!info)
	{
		return (0);
	}
	return (info->len);
}
