/* -*- Mode: c; c-basic-offset: 8; Coding: utf-8-unix -*- ;; */
/* $Id: maildata.c 2 2008-05-27 07:44:27Z mtaneda $	 */

/*
 * Copyright 2008 The piranha Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PIRANHA PROJECT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE PIRANHA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the piranha Project.
 */

#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<ctype.h>
#include	"strtool.h"
#include	"strlist.h"
#include	"pairlist.h"
#include	"mime.h"
#include	"maildata.h"

struct maildata
{
	pairlist_t     *header;
	mime_t         *body;
};

pairlist_t     *maildata_header_parse(const char *data, const char *eol);
mime_t         *maildata_body_parse(const char *data, const char *eol, const char *boundary);
int             maildata_body_parse_core(mime_t * obj, const int offset, const char *data, const char *eol);
char           *maildata_get_boundary(const maildata_t * obj);

maildata_t     *
maildata_create(void)
{
	maildata_t     *obj = NULL;

	obj = (maildata_t *) calloc(1, sizeof(maildata_t));
	if (!obj)
	{
		goto ERROR;
	}
	obj->header = pairlist_create();
	obj->body = mime_create();

	if (!obj->header || !obj->body)
	{
		goto ERROR;
	}
	return (obj);
ERROR:
	maildata_destroy(obj);
	return (NULL);
}

void 
maildata_destroy(maildata_t * obj)
{
	if (obj)
	{
		pairlist_destroy(obj->header);
		mime_destroy(obj->body);
		free(obj);
	}
}

maildata_t     *
maildata_parse(const char *data, const char *eol)
{
	maildata_t     *obj = NULL;
	strlist_t      *tmp = NULL;
	char           *boundary = NULL;
	char           *blank_line = NULL;

	if (strtool_isempty(data) || strtool_isempty(eol))
	{
		goto ERROR;
	}
	strtool_cat(&blank_line, eol);
	strtool_cat(&blank_line, eol);

	tmp = strtool_split(data, blank_line, 1);
	if (!tmp || strlist_len(tmp) != 2)
	{
		goto ERROR;
	}
	obj = (maildata_t *) calloc(1, sizeof(maildata_t));
	if (!obj)
	{
		goto ERROR;
	}
	obj->header = NULL;
	obj->body = NULL;

	obj->header = maildata_header_parse(strlist_refer(tmp, 0), eol);
	if (!obj->header)
	{
		goto ERROR;
	}
	boundary = maildata_get_boundary(obj);

	obj->body = maildata_body_parse(strlist_refer(tmp, 1), eol, boundary);

	strlist_destroy(tmp);
	free(blank_line);
	free(boundary);

	return (obj);
ERROR:
	maildata_destroy(obj);
	strlist_destroy(tmp);
	free(blank_line);
	free(boundary);
	return (NULL);
}

pairlist_t     *
maildata_header_parse(const char *data, const char *eol)
{
	strlist_t      *lines = NULL;
	strlist_t      *line = NULL;
	pairlist_t     *obj = NULL;
	char           *name, *value;
	char           *tmp;
	int             n, len;

	if (strtool_isempty(data) || strtool_isempty(eol))
	{
		goto ERROR;
	}
	lines = strtool_split(data, eol, 0);
	if (!lines)
	{
		goto ERROR;
	}
	len = strlist_len(lines);

	obj = pairlist_create();
	if (!obj)
	{
		goto ERROR;
	}
	for (n = 0; n < len; n++)
	{
		tmp = strlist_refer(lines, n);

		if (isspace(tmp[0]))
		{
			pairlist_pop(obj, &name, &value);

			strtool_cat(&value, eol);
			strtool_cat(&value, tmp);

			pairlist_push(obj, name, value);

			free(name);
			free(value);
			continue;
		}
		line = strtool_split(tmp, ":", 1);

		name = strdup(strlist_refer(line, 0));
		value = strdup(strlist_refer(line, 1));

		strtool_strip(&name);
		strtool_strip(&value);

		pairlist_push(obj, name, value);

		strlist_destroy(line);
		free(name);
		free(value);
	}

	strlist_destroy(lines);

	return (obj);
ERROR:
	strlist_destroy(lines);
	pairlist_destroy(obj);
	return (NULL);
}

mime_t         *
maildata_body_parse(
		    const char *data,
		    const char *eol,
		    const char *boundary)
{
	mime_t         *obj = NULL;
	strlist_t      *elems = NULL;
	int             n, len;

	if (strtool_isempty(data) || strtool_isempty(eol))
	{
		goto ERROR;
	}
	obj = mime_create();
	if (!obj)
	{
		goto ERROR;
	}
	if (boundary)
	{
		elems = strtool_split(data, boundary, 0);
		if (!elems)
		{
			goto ERROR;
		}
		len = strlist_len(elems);
		for (n = 0; n < len - 1; n++)
		{
			mime_new_data(obj);
			maildata_body_parse_core(obj, n, strlist_refer(elems, n), eol);
		}

		strlist_destroy(elems);
	} else
	{
		mime_new_data(obj);
		mime_set_body(obj, 0, data);
	}

	return (obj);
ERROR:
	mime_destroy(obj);
	return (NULL);
}

int 
maildata_body_parse_core(
			 mime_t * obj,
			 const int offset,
			 const char *data,
			 const char *eol)
{
	strlist_t      *tmp = NULL;
	strlist_t      *tmp2 = NULL;
	strlist_t      *lines = NULL;
	char           *blank_line = NULL;
	char           *line;
	char           *name, *value;
	int             n, len;

	if (!obj || strtool_isempty(data) || strtool_isempty(eol))
	{
		goto ERROR;
	}
	strtool_cat(&blank_line, eol);
	strtool_cat(&blank_line, eol);

	tmp = strtool_split(data, blank_line, 1);
	if (!tmp || strlist_len(tmp) != 2)
	{
		goto ERROR;
	}
	lines = strtool_split(strlist_refer(tmp, 0), eol, 0);
	if (!lines)
	{
		goto ERROR;
	}
	len = strlist_len(lines);

	for (n = 0; n < len; n++)
	{
		line = strlist_refer(lines, n);

		if (isspace(line[0]))
		{
			mime_pop_header(obj, offset, &name, &value);

			strtool_cat(&value, eol);
			strtool_cat(&value, line);

			mime_set_header(obj, offset, name, value);

			free(name);
			free(value);
			continue;
		}
		tmp2 = strtool_split(line, ":", 1);

		name = strlist_refer(tmp2, 0);
		value = strlist_refer(tmp2, 1);

		strtool_strip(&name);
		strtool_strip(&value);

		mime_set_header(obj, offset, name, value);

		strlist_destroy(tmp2);
	}

	mime_set_body(obj, offset, strlist_refer(tmp, 1));

	strlist_destroy(tmp);
	strlist_destroy(lines);
	free(blank_line);

	return (0);
ERROR:
	strlist_destroy(tmp);
	strlist_destroy(lines);
	free(blank_line);
	return (1);
}

char           *
maildata_get_boundary(const maildata_t * obj)
{
	char           *dst = NULL;
	char           *ctype = NULL;
	char           *sp, *ep;

	if (!obj)
	{
		goto ERROR;
	}
	ctype = pairlist_refer_value_by_name(obj->header, "Content-Type");
	if (!ctype)
	{
		goto ERROR;
	}
	sp = strstr(ctype, "boundary=\"");
	if (!sp)
	{
		goto ERROR;
	}
	sp += strlen("boundary=\"");

	for (ep = sp; *ep != '\0'; ep++)
	{
		if (*ep == '"')
		{
			dst = strdup("--");

			strtool_ncat(&dst, sp, ep - sp);
		}
	}

	return (dst);
ERROR:
	free(dst);
	return (NULL);
}

char           *
maildata_to_str(const maildata_t * obj, const char *eol)
{
	char           *dst = NULL;
	char           *tmp = NULL;
	char           *boundary = NULL;
	char           *name, *value;
	int             n, len;

	if (!obj || strtool_isempty(eol))
	{
		goto ERROR;
	}
	len = pairlist_len(obj->header);

	for (n = 0; n < len; n++)
	{
		pairlist_refer(obj->header, n, &name, &value);

		strtool_cat(&dst, name);
		strtool_cat(&dst, ": ");
		strtool_cat(&dst, value);

		strtool_cat(&dst, eol);
	}

	boundary = maildata_get_boundary(obj);

	tmp = mime_to_str(obj->body, boundary, eol);
	if (tmp)
	{
		strtool_cat(&dst, tmp);
		free(tmp);
	}
	free(boundary);

	return (dst);
ERROR:
	free(boundary);
	free(dst);
	return (NULL);
}

int 
maildata_add_header(maildata_t * obj, const char *name, const char *value)
{
	int             err;

	if (!obj || strtool_isempty(name) || strtool_isempty(value))
	{
		goto ERROR;
	}
	err = pairlist_add(obj->header, name, value);
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

int 
maildata_del_header_by_name(maildata_t * obj, const char *name)
{
	int             err;

	if (!obj)
	{
		goto ERROR;
	}
	err = pairlist_del_by_name(obj->header, name);
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

char           *
maildata_refer_header_by_name(const maildata_t * obj, const char *name)
{
	char           *value = NULL;

	if (!obj)
	{
		goto ERROR;
	}
	value = pairlist_refer_value_by_name(obj->header, name);
	if (!value)
	{
		goto ERROR;
	}
	return (value);
ERROR:
	return (NULL);
}

char           *
maildata_get_header_by_name(const maildata_t * obj, const char *name)
{
	char           *value = NULL;

	value = maildata_refer_header_by_name(obj, name);
	if (!value)
	{
		goto ERROR;
	}
	return (strdup(value));
ERROR:
	return (NULL);
}

int 
maildata_replace_header_by_name(
				maildata_t * obj,
				const char *name,
				const char *value)
{
	int             err;

	if (!obj)
	{
		goto ERROR;
	}
	err = pairlist_replace_value_by_name(obj->header, name, value);
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}
