/* -*- Mode: c; c-basic-offset: 8; Coding: utf-8-unix -*- ;; */
/* $Id: nettool.c 2 2008-05-27 07:44:27Z mtaneda $	 */

/*
 * Copyright 2008 The piranha Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PIRANHA PROJECT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE PIRANHA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the piranha Project.
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<unistd.h>
#include	<ctype.h>
#include	<sys/socket.h>
#include	<sys/types.h>
#include	<sys/ioctl.h>
#include	<netinet/in.h>
#include	<arpa/inet.h>
#include	<net/if.h>
#include	<netdb.h>
#include	"strtool.h"
#include	"strlist.h"

int             nettool_get_sockaddr_in(struct sockaddr_in * si, const int req, const char *interface);

int 
nettool_is_mac_addr(const char *str)
{
	strlist_t      *list = NULL;
	char           *tmp;
	int             n, len;

	list = strtool_split(str, ":", 0);

	len = strlist_len(list);

	if (!list || len != 6)
	{
		goto ERROR;
	}
	for (n = 0; n < len; n++)
	{
		tmp = strlist_refer(list, n);

		strtool_toupper(tmp);

		if (strcmp(tmp, "FF") > 0)
		{
			goto ERROR;
		}
	}

	return (1);
ERROR:
	return (0);
}

int 
nettool_is_ipv4_addr(const char *str, const int ismask, const int isnet)
{
	struct in_addr  tmp;
	strlist_t      *tmp2 = NULL;
	int             n, flg = 0;

	if (!str)
	{
		goto ERROR;
	}
	if (!inet_aton(str, &tmp))
	{
		goto ERROR;
	}
	if (!ismask)
	{
		if ((tmp.s_addr & 0xF0) >= 0xE0)
		{
			goto ERROR;
		}
		if ((tmp.s_addr & 0xFF) == 0x00)
		{
			goto ERROR;
		}
		if (!isnet)
		{
			if (((tmp.s_addr >> 24) & 0xFF) == 0x00)
			{
				goto ERROR;
			}
		}
		if (((tmp.s_addr >> 24) & 0xFF) == 0xFF)
		{
			goto ERROR;
		}
	} else
	{
		for (n = 0; n < 32; n++)
		{
			if (!((tmp.s_addr >> n) & 0x01))
			{
				flg = 1;
			} else
			{
				if (flg)
				{
					goto ERROR;
				}
			}
		}
		if (!flg)
		{
			goto ERROR;
		}
	}

	tmp2 = strtool_split(str, ".", 0);

	if (strlist_len(tmp2) != 4)
	{
		goto ERROR;
	}
	strlist_destroy(tmp2);

	return (1);
ERROR:
	strlist_destroy(tmp2);
	return (0);
}

int 
nettool_is_ipv4_addr_subnet(const char *str, const char *addr, const char *mask)
{
	struct in_addr  ia_str, ia_addr, ia_mask;

	if (!str)
	{
		goto ERROR;
	}
	if (!nettool_is_ipv4_addr(addr, 0, 1) || !nettool_is_ipv4_addr(mask, 1, 0))
	{
		return (-1);
	}
	if (!inet_aton(str, &ia_str))
	{
		goto ERROR;
	}
	if (!inet_aton(addr, &ia_addr))
	{
		goto ERROR;
	}
	if (!inet_aton(mask, &ia_mask))
	{
		goto ERROR;
	}
	if (ia_str.s_addr == ia_addr.s_addr)
	{
		goto ERROR;
	}
	if (ia_str.s_addr == (ia_addr.s_addr + (ia_mask.s_addr ^ 0xFFFFFFFF)))
	{
		goto ERROR;
	}
	if (((ia_str.s_addr ^ ia_addr.s_addr) & ia_mask.s_addr) > 0)
	{
		goto ERROR;
	}
	return (1);
ERROR:
	return (0);
}

int 
nettool_is_inside_range(const char *addr, const char *range_s, const char *range_e)
{
	struct in_addr  in_ar, in_rs, in_re;

	if (!addr || !range_s || !range_e)
	{
		goto ERROR;
	}
	if (!inet_aton(addr, &in_ar))
	{
		goto ERROR;
	}
	if (!inet_aton(range_s, &in_rs))
	{
		goto ERROR;
	}
	if (!inet_aton(range_e, &in_re))
	{
		goto ERROR;
	}
	if (in_ar.s_addr < in_rs.s_addr || in_ar.s_addr > in_re.s_addr)
	{
		goto ERROR;
	}
	return (1);
ERROR:
	return (0);
}

int 
nettool_is_host_name(const char *name)
{
	int             n;

	if (name[0] == '-' || name[strlen(name) - 1] == '-')
	{
		goto ERROR;
	}
	for (n = 0; name[n]; n++)
	{
		if (!isalnum(name[n]))
		{
			if (name[n] != '-' && name[n] != '.')
			{
				goto ERROR;
			}
		}
	}

	return (1);
ERROR:
	return (0);
}

int 
nettool_is_domain_name(const char *str)
{
	struct addrinfo hints, *res = NULL;

	if (strtool_isempty(str))
	{
		goto ERROR;
	}
	memset(&hints, 0, sizeof(hints));

	hints.ai_socktype = SOCK_STREAM;
	hints.ai_family = AF_INET;

	if (getaddrinfo(str, NULL, &hints, &res) != 0)
	{
		goto ERROR;
	}
	return (1);
ERROR:
	return (0);
}

int 
nettool_is_mail_addr(const char *str)
{
	strlist_t      *tmp = NULL;
	int             len;

	tmp = strtool_split(str, "@", 0);

	len = strlist_len(tmp);

	if (len != 2)
	{
		goto ERROR;
	}
	if (!nettool_is_domain_name(strlist_refer(tmp, 1)))
	{
		goto ERROR;
	}
	strlist_destroy(tmp);

	return (1);
ERROR:
	strlist_destroy(tmp);
	return (0);
}

int 
nettool_get_sockaddr_in(struct sockaddr_in * si, const int req, const char *interface)
{
	int             soc, err;
	struct ifreq    ifreq;

	if (!interface)
	{
		goto ERROR;
	}
	soc = socket(PF_INET, SOCK_DGRAM, 0);

	strcpy(ifreq.ifr_name, interface);

	err = ioctl(soc, req, &ifreq);

	close(soc);

	if (err)
	{
		goto ERROR;
	}
	memcpy(si, (struct sockaddr_in *) & ifreq.ifr_addr, sizeof(struct sockaddr_in));

	return (0);
ERROR:
	return (1);
}

char           *
nettool_get_interface_addr(const char *interface)
{
	struct sockaddr_in tmp;
	int             err;

	err = nettool_get_sockaddr_in(&tmp, SIOCGIFADDR, interface);

	if (err)
	{
		goto ERROR;
	}
	return (strdup(inet_ntoa(tmp.sin_addr)));
ERROR:
	return (NULL);
}

char           *
nettool_get_interface_mask(const char *interface)
{
	struct sockaddr_in tmp;
	int             err;

	err = nettool_get_sockaddr_in(&tmp, SIOCGIFNETMASK, interface);

	if (err)
	{
		goto ERROR;
	}
	return (strdup(inet_ntoa(tmp.sin_addr)));
ERROR:
	return (NULL);
}

char           *
nettool_get_interface_gateway(const char *interface)
{
	struct sockaddr_in tmp1, tmp2;
	int             err;

	err = nettool_get_sockaddr_in(&tmp1, SIOCGIFADDR, interface);
	if (err)
	{
		goto ERROR;
	}
	err = nettool_get_sockaddr_in(&tmp2, SIOCGIFNETMASK, interface);
	if (err)
	{
		goto ERROR;
	}
	tmp1.sin_addr.s_addr &= tmp2.sin_addr.s_addr;
	tmp1.sin_addr.s_addr |= 0x01000000;

	return (strdup(inet_ntoa(tmp1.sin_addr)));
ERROR:
	return (NULL);
}

char           *
nettool_get_hostname(void)
{
	char            host[256];

	memset(host, 0, sizeof(host));

	if (gethostname(host, sizeof(host) - 1) == -1)
	{
		goto ERROR;
	}
	return (strdup(host));
ERROR:
	return (NULL);
}

char           *
nettool_get_new_ipv4_addr(
			  const char *old_addr,
			  const char *old_subnet, const char *old_mask,
			  const char *new_subnet, const char *new_mask)
{
	struct in_addr  o_a, o_s, o_m, n_s, n_m;
	struct in_addr  tmp;
	struct in_addr  new_addr;
	char           *dst = NULL;

	if (strtool_isempty(old_addr) ||
	    strtool_isempty(old_subnet) || strtool_isempty(old_mask) ||
	    strtool_isempty(new_subnet) || strtool_isempty(new_mask))
	{
		goto ERROR;
	}
	if (!inet_aton(old_addr, &o_a))
	{
		goto ERROR;
	}
	if (!inet_aton(old_subnet, &o_s))
	{
		goto ERROR;
	}
	if (!inet_aton(new_subnet, &n_s))
	{
		goto ERROR;
	}
	if (!inet_aton(old_mask, &o_m))
	{
		goto ERROR;
	}
	if (!inet_aton(new_mask, &n_m))
	{
		goto ERROR;
	}
	tmp.s_addr = (o_s.s_addr & o_m.s_addr) ^ o_a.s_addr;

	if (tmp.s_addr & (n_s.s_addr & n_m.s_addr))
	{
		goto ERROR;
	}
	new_addr.s_addr = n_s.s_addr + tmp.s_addr;

	dst = inet_ntoa(new_addr);

	return (strdup(dst));
ERROR:
	return (NULL);
}

char           *
nettool_atoh(const char *src)
{
	struct in_addr  in;
	char           *dst = NULL;

	if (!inet_aton(src, &in))
	{
		goto ERROR;
	}
	dst = strtool_sprintf("%08X", in.s_addr);

	return (dst);
ERROR:
	return (NULL);
}

char           *
nettool_htoa(const char *src)
{
	struct in_addr  in;
	char           *dst = NULL;

	if (strtool_isempty(src))
	{
		goto ERROR;
	}
	in.s_addr = strtoul(src, NULL, 16);

	dst = inet_ntoa(in);

	return (strtool_dup(dst));
ERROR:
	return (NULL);
}
