/* -*- Mode: c; c-basic-offset: 8; Coding: utf-8-unix -*- ;; */
/* $Id: pairlist.c 2 2008-05-27 07:44:27Z mtaneda $	 */

/*
 * Copyright 2008 The piranha Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PIRANHA PROJECT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE PIRANHA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the piranha Project.
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	"dlist.h"
#include	"strtool.h"
#include	"pairlist.h"

struct pair
{
	char           *name;
	char           *value;
};

struct pairlist
{
	dlist_t        *dlist;
};

struct pair    *pairlist_calloc(void);
void            pairlist_free(void *data);

pairlist_t     *
pairlist_create(void)
{
	pairlist_t     *obj = NULL;

	obj = (pairlist_t *) calloc(1, sizeof(pairlist_t));
	if (!obj)
	{
		goto ERROR;
	}
	obj->dlist = dlist_create();
	if (!obj->dlist)
	{
		goto ERROR;
	}
	return (obj);
ERROR:
	pairlist_destroy(obj);
	return (NULL);
}

int 
pairlist_destroy(pairlist_t * obj)
{
	if (!obj)
	{
		goto ERROR;
	}
	dlist_destroy(obj->dlist, pairlist_free);

	free(obj);

	return (0);
ERROR:
	return (1);
}

struct pair    *
pairlist_calloc(void)
{
	struct pair    *obj = NULL;

	obj = (struct pair *) calloc(1, sizeof(struct pair));
	if (!obj)
	{
		goto ERROR;
	}
	obj->name = NULL;
	obj->value = NULL;

	return (obj);
ERROR:
	return (NULL);
}

void 
pairlist_free(void *data)
{
	struct pair    *tmp = (struct pair *) data;

	if (tmp)
	{
		free(tmp->name);
		free(tmp->value);
		free(tmp);
	}
}

int 
pairlist_pop(pairlist_t * obj, char **name, char **value)
{
	struct pair    *data = NULL;

	if (!obj || !name || !value)
	{
		goto ERROR;
	}
	data = dlist_pop(obj->dlist);
	if (!data)
	{
		goto ERROR;
	}
	*name = data->name;
	*value = data->value;

	free(data);

	return (0);
ERROR:
	return (1);
}

int 
pairlist_push(pairlist_t * obj, const char *name, const char *value)
{
	struct pair    *data = NULL;
	int             err;

	if (!obj || strtool_isempty(name) || strtool_isempty(value))
	{
		goto ERROR;
	}
	data = pairlist_calloc();
	if (!data)
	{
		goto ERROR;
	}
	data->name = strdup(name);
	data->value = strdup(value);

	err = dlist_push(obj->dlist, (void *) data);
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	pairlist_free((void *) data);
	return (1);
}

int 
pairlist_dequeue(pairlist_t * obj, char **name, char **value)
{
	struct pair    *data = NULL;

	if (!name || !value)
	{
		goto ERROR;
	}
	data = (struct pair *) dlist_dequeue(obj->dlist);
	if (!data)
	{
		goto ERROR;
	}
	*name = data->name;
	*value = data->value;

	free(data);

	return (0);
ERROR:
	return (1);
}

int 
pairlist_queue(pairlist_t * obj, const char *name, const char *value)
{
	struct pair    *data = NULL;

	if (!obj || strtool_isempty(name) || strtool_isempty(value))
	{
		goto ERROR;
	}
	data = pairlist_calloc();

	data->name = strdup(name);
	data->value = strdup(value);

	return (dlist_queue(obj->dlist, (void *) data));
ERROR:
	return (1);
}

int 
pairlist_add(pairlist_t * obj, const char *name, const char *value)
{
	return (pairlist_push(obj, name, value));
}

int 
pairlist_get(
	     const pairlist_t * obj,
	     const int offset,
	     char **name,
	     char **value)
{
	char           *tmp_name = NULL;
	char           *tmp_value = NULL;
	int             err;

	if (!name || !value)
	{
		goto ERROR;
	}
	err = pairlist_refer(obj, offset, &tmp_name, &tmp_value);
	if (err)
	{
		goto ERROR;
	}
	*name = strdup(tmp_name);
	*value = strdup(tmp_value);

	return (0);
ERROR:
	return (1);
}

char           *
pairlist_get_value_by_name(
			   const pairlist_t * obj,
			   const char *name)
{
	char           *value;

	value = pairlist_refer_value_by_name(obj, name);
	if (!value)
	{
		return (NULL);
	}
	return (strdup(value));
}

char           *
pairlist_get_name_by_value(
			   const pairlist_t * obj,
			   const char *value)
{
	char           *name;

	name = pairlist_refer_name_by_value(obj, value);
	if (!name)
	{
		return (NULL);
	}
	return (strdup(name));
}

int 
pairlist_refer(
	       const pairlist_t * obj,
	       const int offset,
	       char **name,
	       char **value)
{
	struct pair    *data = NULL;

	if (!name || !value)
	{
		goto ERROR;
	}
	data = (struct pair *) dlist_get_data_by_offset(obj->dlist, offset);
	if (!data)
	{
		goto ERROR;
	}
	*name = data->name;
	*value = data->value;

	return (0);
ERROR:
	return (1);
}

char           *
pairlist_refer_value_by_name(const pairlist_t * obj, const char *name)
{
	int             n, len;
	char           *t_name, *value = NULL;

	if (!obj || strtool_isempty(name))
	{
		return (NULL);
	}
	len = pairlist_len(obj);

	for (n = 0; n < len; n++)
	{
		pairlist_refer(obj, n, &t_name, &value);

		if (strcmp(t_name, name) == 0)
		{
			return (value);
		}
	}

	return (NULL);
}

char           *
pairlist_refer_name_by_value(const pairlist_t * obj, const char *value)
{
	int             n, len;
	char           *name, *t_value = NULL;

	if (!obj || strtool_isempty(value))
	{
		return (NULL);
	}
	len = pairlist_len(obj);

	for (n = 0; n < len; n++)
	{
		pairlist_refer(obj, n, &name, &t_value);

		if (strcmp(t_value, value) == 0)
		{
			return (name);
		}
	}

	return (NULL);
}

pairlist_t     *
pairlist_dup(const pairlist_t * obj)
{
	if (!obj)
	{
		return (NULL);
	}
	return (pairlist_ndup(obj, pairlist_len(obj)));
}

pairlist_t     *
pairlist_ndup(const pairlist_t * obj, const int len)
{
	pairlist_t     *dst = NULL;
	char           *name, *value;
	int             n;

	if (!obj || len <= 0 || len > pairlist_len(obj))
	{
		goto ERROR;
	}
	dst = pairlist_create();
	if (!dst)
	{
		goto ERROR;
	}
	for (n = 0; n < len; n++)
	{
		pairlist_refer(obj, n, &name, &value);
		pairlist_push(dst, name, value);
	}

	return (dst);
ERROR:
	return (NULL);
}

int 
pairlist_cat(pairlist_t * dst, const pairlist_t * src)
{
	int             err;

	if (!dst || !src)
	{
		goto ERROR;
	}
	err = pairlist_ncat(dst, src, pairlist_len(src));
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

int 
pairlist_ncat(pairlist_t * dst, const pairlist_t * src, const int len)
{
	int             n;
	char           *name, *value;

	if (!dst || !src || len < 0 || len > pairlist_len(src))
	{
		goto ERROR;
	}
	for (n = 0; n < len; n++)
	{
		pairlist_get(src, n, &name, &value);
		pairlist_push(dst, name, value);
		free(name);
		free(value);
	}

	return (0);
ERROR:
	return (1);
}

int 
pairlist_del_by_name(pairlist_t * obj, const char *name)
{
	int             n, len;
	int             err;
	char           *t_name;
	char           *t_value;

	if (!obj || strtool_isempty(name))
	{
		goto ERROR;
	}
	len = pairlist_len(obj);

	for (n = 0; n < len; n++)
	{
		pairlist_refer(obj, n, &t_name, &t_value);

		if (!strtool_isempty(t_name) && strcmp(name, t_name) == 0)
		{
			break;
		}
	}

	if (n == len)
	{
		goto ERROR;
	}
	err = dlist_del_data_by_offset(obj->dlist, n, pairlist_free);

	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

int 
pairlist_replace_value_by_name(
			       pairlist_t * obj,
			       const char *name,
			       const char *value)
{
	struct pair    *tmp = NULL;
	int             n, len;

	if (!obj || pairlist_len(obj) < 1 ||
	    strtool_isempty(name) || strtool_isempty(value))
	{
		goto ERROR;
	}
	len = pairlist_len(obj);

	for (n = 0; n < len; n++)
	{
		tmp = dlist_get_data_by_offset(obj->dlist, n);
		if (tmp && strcmp(tmp->name, name) == 0)
		{
			break;
		}
		tmp = NULL;
	}

	if (!tmp)
	{
		goto ERROR;
	}
	free(tmp->value);

	tmp->value = strdup(value);

	return (0);
ERROR:
	return (1);
}

int 
pairlist_is_exist(const pairlist_t * obj, const char *name)
{
	char           *tmp_name, *tmp_value;
	int             n, len;

	if (obj && !strtool_isempty(name))
	{
		len = pairlist_len(obj);

		for (n = 0; n < len; n++)
		{
			pairlist_refer(obj, n, &tmp_name, &tmp_value);

			if (tmp_name && strcmp(tmp_name, name) == 0)
			{
				return (1);
			}
		}
	}
	return (0);
}

int 
pairlist_len(const pairlist_t * obj)
{
	if (!obj)
	{
		goto ERROR;
	}
	return (dlist_len(obj->dlist));
ERROR:
	return (0);
}
