#!/bin/sh -efu
#
# Copyright (C) 2006 Securedog
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
#
# $Id$

is_yes()
{
	case ${1-} in
	[Yy][Ee][Ss]|[Yy]|[Tt][Rr][Uu][Ee])
		return 0 ;;
	*)	return 1 ;;
	esac
}

warn() {
	echo "** ${@-}" >&2
}

info() {
	echo "--->  ${@-}"
}

prompt_yesno() {
	local prompt default input

	prompt=${1-"OK?"}
	default=${2-"yes"}

	echo -n "${prompt} [${default}] " >&2
	read input

	is_yes ${input:-"${default}"}
}

prompt_yesno_if_interactive() {
	if is_yes ${opt_noexecute:-NO}; then
		return 1
	elif is_yes ${opt_interactive:-NO}; then
		prompt_yesno ${1+"$@"} || return 1
	fi
}

init_variables() {
#ifdef WITH_PKGSRC
	_SYSTEM=pkgsrc
	: ${PKGSRCDIR="/usr/pkgsrc"}
	: ${PACKAGES="${PKGSRCDIR}/packages"}
	: ${PACKAGEROOT="ftp://ftp.NetBSD.org"}
	: ${PKGREPOSITORY="${PACKAGES}/All"}
#else
	: ${PORTSDIR="/usr/ports"}
	: ${PACKAGES="${PORTSDIR}/packages"}
#ifdef WITH_OPENBSD
	_SYSTEM=OpenBSD
	: ${PACKAGEROOT="ftp://ftp.OpenBSD.org"}
	: ${PACKAGE_REPOSITORY="${PACKAGES}/All"}
#else
	_SYSTEM=FreeBSD
	: ${PACKAGEROOT="ftp://ftp.FreeBSD.org"}
	: ${PKGREPOSITORY="${PACKAGES}/All"}
#endif
	: ${PACKAGESITE=""}
	: ${PKG_DBDIR="/var/db/pkg"}
#ifdef WITH_PKGSRC
	: ${PKG_SUFX=".tgz"}
#else
#ifdef WITH_OPENBSD
	: ${PKG_SUFX=".tgz"}
#else
	: ${PKG_SUFX=".tbz"}
#endif
#endif
	: ${MAKE="make"}
	MAKE_CMD=${MAKE}
	cwd=${PWD}

	[ -n "${CATEGORIES-}" ] || init_categories

#ifdef WITH_PKGSRC
	PORTSDIR=${PKGSRCDIR}
	export PKGSRCDIR PACKAGES PKGREPOSITORY \
#else
#ifdef WITH_OPENBSD
	PKGREPOSITORY=PACKAGE_REPOSITORY
	export PORTSDIR PACKAGES PACKAGE_REPOSITORY \
#else
	export PORTSDIR PACKAGES PKGREPOSITORY \
#endif
	       PKG_DBDIR PKG_SUFX
}

init_categories() {
	local - dir

	set +f
	for dir in ${PORTSDIR}/*/; do
		dir=${dir#${PORTSDIR}/}
		case ${dir%/} in
		[mM]k|[tT]emplates|Tools|bootstrap|distfiles|doc|infrastructure|licenses|packages) ;;
		*) CATEGORIES="${CATEGORIES:+${CATEGORIES} }${dir%/}" ;;
		esac
	done
}

extract_categories() {
	local pattern category

	pattern=$1
	if [ "${pattern}" = '*' ]; then
		categories="${CATEGORIES}"
	else
		categories=
		for category in ${CATEGORIES}; do
			case ${category} in
			${pattern}) categories="${categories} ${category}" ;;
			esac
		done
	fi
}

set_port_name() {
	port_name=
	if is_yes ${opt_use_index:-NO}; then
		port_name=$(grep --mmap -m 1 -o "^[^|]*|[^|]*${1#${PORTSDIR}/}|" "${PORTSDIR}/INDEX")
		port_name="${port_name%%|*}"
	elif [ -e "$1/Makefile" ] && cd "$1"; then
#ifdef WITH_PKGSRC
		port_name=$(${MAKE_CMD} -V '${PKGNAME}')
#else
#ifdef WITH_OPENBSD
		port_name=$(${MAKE_CMD} show=FULLPKGNAME\${SUBPACKAGE})
#else
		port_name=$(${MAKE_CMD} -VPKGNAME)
#endif
#endif
		cd ${cwd}
	else
		return 1
	fi

	if [ -z "${port_name}" ]; then
		return 1
	fi
}

set_pkg_name() {
	pkg_name=$1
}

set_pkg_origin() {
	local LINE

	pkg_origin=
#ifdef WITH_PKGSRC
	if [ -r "${PKG_DBDIR}/$1/+BUILD_INFO" ]; then
#else
	if [ -r "${PKG_DBDIR}/$1/+CONTENTS" ]; then
#endif
		while read LINE; do
			case ${LINE} in
#ifdef WITH_PKGSRC
			PKGPATH=*)
				pkg_origin=${LINE#PKGPATH=}
				break ;;
			esac
		done < "${PKG_DBDIR}/$1/+BUILD_INFO"
#else
			@comment\ ORIGIN:*)
				pkg_origin=${LINE#@comment ORIGIN:}
				break ;;
			[!@]*)	break ;;
			esac
		done < "${PKG_DBDIR}/$1/+CONTENTS"
#endif
	fi
}

set_pkg_pkgdir() {
	pkg_pkgdir=${PKG_DBDIR}/$1
}

set_pkg_portdir() {
	pkg_portdir=${PORTSDIR}/$1
}

correct_pkg() {
	if [ -z "${pkg_origin}" ]; then
		if [ -d "${pkg_pkgdir}" ]; then
			warn "'${pkg_name}' has no origin recorded."
		else
			warn "'${pkg_name}' is not installed. (broken dependencies?)"
		fi
		return 1
	elif [ ! -d "${pkg_portdir}" ]; then
		warn "${pkg_name} (${pkg_origin}): perhaps moved or obsoleted."
		trace_moved "${pkg_origin}" || return 1
	fi
}

set_system_info() {
	: ${ARCH=$(uname -m)}
	: ${OPSYS=$(uname -s)}
	: ${OS_VERSION=$(uname -r)}
	: ${OS_REVISION=${OS_VERSION%%-*}}
	: ${OS_MAJOR=${OS_REVISION%%.*}}
}

ports_glob() {
	local - OPT OPTARG OPTIND pattern name portdir categories category \
		opt_depends opt_required_by found

	opt_depends=
	opt_required_by=

	while getopts :rR OPT; do
		case ${OPT} in
		r)	opt_required_by=YES ;;
		R)	opt_depends=YES ;;
		\?)	return 1 ;;
		esac
	done
	shift $((${OPTIND}-1))

	set +f
	while [ $# -gt 0 ]; do
		pattern=${1#${PORTSDIR}/}; shift
		found=

		case ${pattern} in
		/*|*/|*/*/*|*./*|'') continue ;;
		*/*)
			name="${pattern#*/}"
			categories="${pattern%/*}" ;;
		*)
			name="${pattern}"
			categories='*' ;;
		esac

		extract_categories "${categories}"

		for category in ${categories}; do
			for portdir in ${PORTSDIR}/${category}/${name}/; do
				if [ -d "${portdir}" ]; then
					portdir=${portdir%/}

					if is_yes ${opt_depends}; then
						ports_depends "${portdir}"
					fi

					echo "${portdir#${PORTSDIR}/}"

					found=YES
				fi
			done
		done

		is_yes ${found} || warn "No such port: ${pattern}"
	done
}

ports_depends() {
	local IFS cwd port checked xvar depend

	case ${2-} in
	\$*)	xvar=${2#$}; eval ${xvar}=; set -- "$1" ;;
	*)	xvar= ;;
	esac

	IFS=' 
'
	checked=

	while [ $# -gt 0 ]; do
		port=$1; shift

		case ${checked} in
		*" ${port} "*) continue ;;
		esac

		cd "${port}" || continue
		checked="${checked:- }${port} "

#ifdef WITH_PKGSRC
		for depend in $(${MAKE} show-depends-pkgpaths); do
			set -- "${PORTSDIR}/${depnd}" ${1+"$@"}
#else
		for depend in $(${MAKE} -VRUN_DEPENDS -VLIB_DEPENDS); do
			set -- "${depend#*:}" ${1+"$@"}
#endif
		done
	done

	for depend in ${checked# * }; do
		if [ -n "${xvar}" ]; then
			eval ${xvar}=\"\$${xvar} ${depend#${PORTSDIR}/}\"
		else
			echo "${depend#${PORTSDIR}/}"
		fi
	done

	cd ${cwd}
}

try() {
	local exit

	if "$@"; exit=$?; [ ${exit} != 0 ]; then
		warn "Command failed (exit code ${exit}): $@"
		return ${exit}
	fi
}

run_make() {
	local port

	port=$1; shift
	if ! { cd "${port}" && try ${MAKE_CMD} ${1+"$@"}; }; then
		warn "Error in ${port}."
		return 1
	fi
}

trace_moved() {
#ifdef WITH_PKGSRC
	warn "See ${PORTSDIR}/doc/CHANGES-*."
	return 1
#else
#ifdef WITH_OPENBSD
	return 1
#else
	local IFS msg checked moved origin

	file_exist "${PORTSDIR}/MOVED" || return 1

	msg=
	origin=$1
	checked=
	IFS='|'

	while moved=$(grep -m1 "^$1|" "${PORTSDIR}/MOVED"); do
		set -- ${moved}

		if [ -n "$2" ]; then
			checked="${checked:-|}$1|"

			case ${checked} in
			*"|${2}|"*)
				warn "Failed. (${msg} -> ???)"
				break ;;
			esac

			msg="${msg:+${msg} }-> $2"
			set_pkg_portdir "$2"

			if [ -d "${pkg_portdir}" ]; then
				warn "'${origin}' has been moved. (${msg})"
				pkg_origin=$2
				return 0
			fi

			set -- "$2"
		else
			warn "'$1' has been removed from ports tree:"
			warn "\"$4\" ($3)"
			return 1
		fi
	done

	warn "Port directory not found: ${origin}"
	return 1
#endif
#endif
}

make_dirs() {
	while [ $# -gt 0 ]; do
		[ -d "$1" ] || try mkdir -p "$1" || return 1
		shift
	done
}

if [ "${0##*/}" = "portutil" ]; then
	usage() {
		echo "usage: ${0##*/} {info,install,fetch,update,config,rmconfig} ..."
		exit 0
	}

	case ${1-} in
	''|*[!a-zA-Z]*)	usage ;;
	info)		cmd="portinfo" ;;
	install)	cmd="portinfo -c $1 -c clean" ;;
	update)		cmd="pkg_replace" ;;
	*)		cmd="portinfo -c $1" ;;
	esac
	shift

	exec ${cmd} ${1+"$@"}
fi
