package jp.co.app.ynomoto.pmft;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;

import android.app.PendingIntent;
import android.appwidget.AppWidgetManager;
import android.appwidget.AppWidgetProvider;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.widget.RemoteViews;

import android.util.Log;

public class PowerMeter extends AppWidgetProvider {
	
	@Override
	public void onUpdate(Context c, AppWidgetManager awm, int[] appWidgetIds) {
		
		//length の内容が onUpdate が呼ばれる度に大きくなるらしいので final が必要だと思われる。
		final int N = appWidgetIds.length; 
		
		// ウィジェットがホームに配置されている分実行する。
		for (int i =0 ; i < N; i++) {
			int appWidgetId = appWidgetIds[i];
			
			// リモートビューに main.xml をセットする。
			RemoteViews rv = new RemoteViews(c.getPackageName(), R.layout.main);
	
			try {
				// csv ファイルをダウンロードする。
				String csvUrl = "http://powermeter.sourceforge.jp/juyo-j.csv";
//				String csvUrl = "http://www.tepco.co.jp/forecast/html/images/juyo-j.csv";				
				Log.v("jp.co.app.ynomoto.pmft.PowerMeter", "WebClient(" + csvUrl +")");
				WebClient wc = new WebClient(csvUrl);
				
				// 取得した csv の解析
				TempcoDataParser tdp = new TempcoDataParser(wc.getWebPage());
				
				//ブラウザを呼び出し、Google チャート API にグラフのデータを渡す。
				int[] maximumkW = new int[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, tdp.getMaximumKw()};
				GChart gc = new GChart(maximumkW, tdp.getTodayKw(), tdp.getYesterdayKw());
		    	Uri uri = gc.getChart();
		       Intent intent = new Intent(Intent.ACTION_VIEW, uri);
		       PendingIntent pendingIntent = PendingIntent.getActivity(c, 0, intent, 0);
				rv.setOnClickPendingIntent(R.id.powerMeter, pendingIntent);

				// ピーク時供給力供給量に対する電力の消費率を表示
				int usage = PowerUsageCalc.calculate(tdp.getCurrentKw(), tdp.getMaximumKw());
				Log.v("jp.co.app.ynomoto.pmft.PowerMeter", "usage=" + usage);
				rv.setTextViewText(R.id.textView1, Integer.toString(usage) + "%");
				
				// 電力消費率による顔文字の変化
				if(usage >= 90){
					rv.setTextViewText(R.id.textView2, "(´-ω-`)");
				} else if(usage >= 70){
					rv.setTextViewText(R.id.textView2, "(´・ω・`)");
				} else {
					rv.setTextViewText(R.id.textView2, "(`・ω・´)");
				} 
				
			} catch (MalformedURLException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (IOException e) {
				rv.setTextViewText(R.id.textView1, "Err");
			}

			// この AppWidgetManager appWidgetId のウィジェットを更新することを伝える。
			awm.updateAppWidget(appWidgetId, rv);
		}
	}
}

/*
 * 取得した csv は、String[]に格納して getPage() で情報の取得ができる。
 */

class WebClient {
	private String[] webPage = new String[30];

	WebClient(String url) throws MalformedURLException, IOException {

		String host[] = url.split("/");
		URL u;

		u = new URL(url);
		HttpURLConnection urlconn = (HttpURLConnection) u.openConnection();

		urlconn.setRequestMethod("GET");
		urlconn.setInstanceFollowRedirects(false);
		urlconn.setRequestProperty("Host", host[2]);
		urlconn.setRequestProperty("Accept", "*/*");
		urlconn.setRequestProperty("Referer", "http://" + host[2] + "/");
		urlconn.setRequestProperty("Accept-Language", "ja");
		urlconn.setRequestProperty("User-Agent", "pmft/0.1");
		urlconn.setRequestProperty("Connection:", "close");
		urlconn.connect();

		if (urlconn.getHeaderField(0).equals("HTTP/1.1 200 OK")) {
			BufferedReader reader = new BufferedReader(new InputStreamReader(
					urlconn.getInputStream(), "Shift_JIS"));
			int count = 0;
			while ((webPage[count] = reader.readLine()) != null) {
				count++;
			}
			reader.close();
		}

		urlconn.disconnect();
	}

	String[] getWebPage() {
		return webPage;
	}
}

/*
 * juyo-j.csv を解析して、各フィールドにデータを格納する。この csv のフォーマット は度々変更される可能性があるので注意すること。
 */
// 2011/3/28 時点での csv のフォーマット
// -----------------------------------------------------------------
// 2011/3/28 7:05 UPDATE
// ピーク時供給力(万kW),時台,供給力情報更新日,供給力情報更新時刻
// 3850,18:00,3/27,17:30
//
// DATE,TIME,当日実績(万kW),前日実績(万kW)
// 2011/3/28,0:00,2710,2904
// 2011/3/28,1:00,2576,2759
// 2011/3/28,2:00,2506,2661
// 2011/3/28,3:00,2457,2578
// 2011/3/28,4:00,2454,2520
// 2011/3/28,5:00,2589,2529
// 2011/3/28,6:00,2863,2598
// 2011/3/28,7:00,0,2760
// 2011/3/28,8:00,0,2908
// 2011/3/28,9:00,0,2946
// 2011/3/28,10:00,0,2913
// 2011/3/28,11:00,0,2857
// 2011/3/28,12:00,0,2826
// 2011/3/28,13:00,0,2786
// 2011/3/28,14:00,0,2772
// 2011/3/28,15:00,0,2786
// 2011/3/28,16:00,0,2860
// 2011/3/28,17:00,0,3073
// 2011/3/28,18:00,0,3371
// 2011/3/28,19:00,0,3392
// 2011/3/28,20:00,0,3345
// 2011/3/28,21:00,0,3251
// 2011/3/28,22:00,0,3103
// 2011/3/28,23:00,0,2939
// -----------------------------------------------------------------

class TempcoDataParser {
	private int maximumKw;
	private int currentKw;
	private int[] yesterdayKw = new int[24];
	private int[] todayKw = new int[24];;

	TempcoDataParser(String[] webPage) {
		// ピーク時供給力(万kW)を求める
		String[] tmp = webPage[2].split(",");
		maximumKw = Integer.parseInt(tmp[0]);
		
		// CSV の更新時間を求める
		tmp = webPage[0].split(" ");
		String[] time = tmp[1].split(":");
		String updateTime = time[0]; 
				
		// 0 時が CSV の6行目にあるので配列の index は 5
		for(int i = 5; webPage[i] != null; i++){
			if(webPage[i].matches(".*" + updateTime + ":.*")){
				if(i == 5){
					// updateTime が0時ならば、前日の23時の情報を取得する。CSV
					// の現在のフォーマットでは、0時の場合は、前日の情報全てを取得で
					// きる。
					tmp = webPage[28].split(",");
					currentKw = Integer.parseInt(tmp[2]);
					break;
				} else {
					// updateTime が1時ならば、その前の時間の消費電力量を取得。
					tmp = webPage[i-1].split(",");
					currentKw = Integer.parseInt(tmp[2]);
					break;
				}
			}
		}
		
		// yesterdayKw と todayKw を求める。
		for(int i = 5; webPage[i] != null; i++){
			tmp = webPage[i].split(",");
			yesterdayKw[i-5] = Integer.parseInt(tmp[3]);
			todayKw[i-5] = Integer.parseInt(tmp[2]);
		}
	}

	int getMaximumKw() {
		return maximumKw;
	}

	int getCurrentKw() {
		return currentKw;
	}
	
	public int[] getYesterdayKw() {
		return yesterdayKw;
	}

	public int[] getTodayKw() {
		return todayKw;
	}

}

/*
 * TempcoDataParser から得られた結果に基づいてピーク時供給力供給量に対する消費率を計算する。
 */
class PowerUsageCalc {
	public static int calculate(int currentKw, int maximumKw){
		return (int)((float)currentKw / maximumKw * 100);
	}
}

/*
 * Google Chart API 用の URL を生成する。
 */
	
//URL=http://chart.apis.google.com/chart
//chart に渡すパラメータの説明
//	chs=300x150 // グラフの幅(px)、グラフの高さ(px)
//	chd=t:60,80,83,85,87,88|85,80,90,80,70,90 //本日のデータ, 前日のデータ
//	cht=lc // グラフの種類は折れ線グラフ
//	chco=0000ff,00ff00// 本日のデータは青、前日のデータは緑
//	chtt=TEPCO+Power+Usage+%28million%20kW%29 //グラフタイトル
//	chdl=Today|Yesterday //本日のデータ、前日のデータ、
//	chxt=x,y // x 軸下方(index0)と y軸左辺(index1)にラベルを表示する
//	chxl=0:|0|12|23|1:|0|2000|4000 // index0, index1
// chg=25,-1// グリッドを表示
/*
 *  chd パラメータは 0 ～ 100 までの範囲でなければならない。従って、最大値を100%
 *  として、それに対する比率をデータとして用意する必要がある。
 */

class GChart {
	private String URL = "http://chart.apis.google.com/chart";
	private String chs,chd, cht, chco, chtt, chdl, chxt, chxl, chg;
	
	// Google チャートへ渡すパラメータの作成
	GChart(int[] maximumKw, int[] todayKw, int[] yesterdayKw){
		int graphMax = 6000;
		chs = "800x300";
		chd = "t:"+ toPercentage(maximumKw, graphMax) + "|" + toPercentage(todayKw, graphMax) + "|" + toPercentage(yesterdayKw, graphMax);
		cht = "bvo";
		chco = "11210D,3D7930,A2C180";
		chtt = "「本日の TEPCO 管轄内電力の使用状況グラフ(単位:万Kw)」";
		chdl = "ピーク時供給力|本日実績|前日実績";
		chxt = "x,y"; 
		chxl = "0:|0|1|2|3|4|5|6|7|8|9|10|11|12|13|14|15|16|17|18|19|20|21|22|23|1:|0|" + (graphMax / 2) + "|" + graphMax;
		chg = "25,-1";
	}
	
	private String toPercentage(int[] kWs, int graphMax){
		String percentages = String.valueOf((int)((float)kWs[0] / graphMax * 100));
		for(int i = 1; i < kWs.length; i++){
			percentages += "," + (int)((float)kWs[i] / graphMax * 100);
		}
		return percentages;
	}
	
	public Uri getChart() {
		String s = URL 
					+ "?" + "chs=" + chs 
					+ "&" + "chd=" + chd 
					+ "&" + "cht=" + cht 
					+ "&" + "chco=" + chco 
					+ "&" + "chtt=" + chtt 
					+ "&" + "chdl=" + chdl 
					+ "&" + "chxt=" + chxt
					+ "&" + "chxl=" + chxl
					+ "&" + "chg=" + chg;
		Log.v("jp.co.app.ynomoto.pmft.PowerMeter.GChart.getChart()", s);
		return Uri.parse(s);
	}
	
}