;;;-*- mode: lisp-interaction; syntax: elisp -*-;;;
;;
;; "predoc.el" is a WYSIWYG majar mode for Predoc document format.
;;
;;   Copyright (C) 2007 Kiyoka Nishiyama
;;
;;     $Date: 2007/02/18 13:52:34 $
;;
;; This file is part of Predoc
;;
;; Predoc is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;; 
;; Predoc is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with Predoc; see the file COPYING.
;;
;;
;;
;; How to nstall and How to use:
;;     http://www.netfort.gr.jp/~kiyoka/predoc/
;;
;;
;; ChangeLog:
;;   [0.9.1]
;;     1. Supported image resizing. ( width and height parameter of img tag )
;;     2. Supported predoc-insert-images. ( C-c l )
;;     3. Supported image cache.
;;
;;   [0.9.0]
;;     1. first release
;;
;;
(require 'sgml-mode)

(defconst predoc-version "0.9.1")

(defconst predoc-alink-face  'link)
(defconst predoc-entity-face 'highlight)
(defconst predoc-pre-face    'shadow)
(defconst predoc-image-face  'trailing-whitespace)

(defun predoc-warning (format &rest args)
  (apply 'message (concat "Predoc Warning: " format) args)
  (beep)
  (sleep-for 1))


(defconst predoc-image-icon-string
  "H_")

(defconst predoc-image-prefix-list
  '("bmp"    "gif"    "jpeg"    "jpg"    "png"    "svg"    "tiff"    "tif"    "xbm"    "xpm"))

(defconst predoc-entity-table
  '(
    (lt     .  "<"  )
    (gt     .  ">"  )
    (amp    .  "&"  )
    (quot   .  "\"" )
    (uml    .  ""  )
    (acute  .  ""  )
    (yen    .  ""  )
    (cent   .  ""  )
    (pound  .  ""  )
    (sect   .  ""  )
    (para   .  ""  )
    (copy   .  ""  )
    (reg    .  ""  )
    (not    .  ""  )
    (deg    .  ""  )
    (plusmn .  ""  )
    (times  .  ""  )
    (divide .  ""  )
    (laquo  .  ""   )))


(defcustom predoc-insert-image-size 100
  "Image size of predoc-insert-image command (C-c l)."
  :type  'integer
  :group 'predoc)

(defcustom predoc-convert-program "/usr/bin/convert"
  "The full-path of Imagemagick 'convert' program."
  :type  'string
  :group 'predoc)


;; --- utility ---
(defun assoc-ref (alist key)
  (let ((entry (assoc key alist)))
    (when entry
      (cdr entry))))


(defun predoc-entity-string-to-char (_entity)
  (let* (
	 (replace-pair
	  (assq (intern _entity) predoc-entity-table))h)
    (if replace-pair
	(car (string-to-list (cdr replace-pair)))
      nil)))


(defun predoc-insert-image (beg end image &rest args)
  "Display image on the current buffer.
Buffer string between BEG and END are replaced with IMAGE."
  (add-text-properties beg end (list 'display image
				     'intangible image
				     'invisible nil)))


(defun predoc-create-image (file &optional width height)
  (let (data pixel-width pixel-height)
    (when (file-readable-p file)
      ;;	(setq pixel-width (predoc-scale-pixel-width scale))
      (setq pixel-width (or width
			    ""))
      (setq pixel-height (or height
			     ""))
      (when (or width height)
	(if (not (file-executable-p predoc-convert-program))
	    (message (format "'%s' does not executable... Image can't be convert size." predoc-convert-program))
	  (setq data
		(with-temp-buffer
		  (let ((coding-system-for-read 'binary)
			(coding-system-for-write 'binary)
			(auto-image-file-mode nil))
		    (set-buffer-multibyte nil)
		    (insert-file-contents file)
		    ;; (message (format "1: min = %s  max = %s " (point-min) (point-max)))
		    (call-process-region (point-min) (point-max)
					 predoc-convert-program
					 t 
					 '(t nil)
					 nil
					 "-" 
					 "-resize"
					 (format "%sx%s" pixel-width pixel-height)
					 "PNG:-")
		    ;; (message (format "2: min = %s  max = %s " (point-min) (point-max)))
		    (buffer-substring-no-properties (point-min) (point-max)))))))
      (if data
	  (create-image data 'png 'data :ascent 'center)
	(create-image file nil nil :ascent 'center)))))


(defun predoc-insert-image-file (beg end attr-alist)
  "Display image on the current buffer
Buffer string between BEG and END are replaced with URL."

  (let ((image (assoc-ref predoc-image-cache attr-alist)))
    (when (not image)
      (let (
	    (src    (assoc-ref attr-alist 'src))
	    (width  (assoc-ref attr-alist 'width))
	    (height (assoc-ref attr-alist 'height)))
	
	;; get the image data file
	(setq image (predoc-create-image
		     (predoc-expand-full-path src)
		     width height))
	;; push to cache
	(push 
	 `(,attr-alist
	   .
	   ,image)
	 predoc-image-cache)
	))
    ;; insert to buffer
    (predoc-insert-image beg end image)))

;;
;; --- test code ---
;;
;;(setq  predoc-image-cache '())
;;
;;(predoc-insert-image-file 
;; (+ (point) 100)
;; (+ (point) 101)
;; '(
;;   (src    . "../doc/img/predoc_logo.png")
;;   (width  . "40")
;;   (height . "40")))
;;
;;(insert (pp predoc-image-cache))


(defun predoc-remove-image (beg end)
  "Remove an image which is inserted between BEG and END."
  (remove-text-properties beg end '(display nil intangible nil)))

(defun predoc-get-image (pos)
  "Get an image object which is indexed pos."
  (get-text-property pos 'display))

(defun predoc-expand-full-path (file)
  "Expand full path from relative path."
  (concat default-directory file))

;;
;; width=10 src="abc\"
;;   -> (("width" . "10") ("src" . "abc"))
;;
;; test pattern:
;;   (assq 'src (predoc-parse-attribute "width=10 src=\"abc\""))
;;
(defun predoc-parse-attribute (str)
  (let ((lst (split-string str "[ ]+"))
	(predoc-attribute-pattern "\\([a-zA-Z]+\\)=[\"]?\\([^ \"]+[/]?\\)"))
    (mapcar
     (lambda (x)
       (if (string-match predoc-attribute-pattern x)
	   (cons
	    (intern (match-string 1 x))
	    (match-string 2 x))
	 nil))
     lst)))

;; 
;; fontification
;;
(defun predoc-install-fontification ()
  (let ((_entity-pattern
	 "\\(&\\)\\([a-zA-Z]+\\)\\(;\\)")
	(_alink-pattern
	 "\\(<[aA][^>]*[>]\\)\\([^<]+\\)\\([<][/][aA][>]\\)")
	(_alink-elem-pattern
	 "[>]\\([^<]+\\)[<][/][aA][>]")
	(_pre-pattern
	 "\\(<pre>\\|<PRE>\\|</pre>\\|</PRE>\\)")
	(_image-pattern
 	 "\\(<img[ ]+\\)\\([^>]+\\)\\(>\\)"))

    (set (make-local-variable 'font-lock-defaults)
	 `((
	    ;; &xxxx;
	    (,_entity-pattern
	     (2
	      (let ((ch (predoc-entity-string-to-char (match-string 2))))
		(when ch
		  (compose-region (match-beginning 1)
				  (match-end       3)
				  ch)
		  nil)
		nil)))

	    (,_entity-pattern
	     .
	     predoc-entity-face)

	    ;; <pre> or </pre>
	    (,_pre-pattern
	     .
	     predoc-pre-face)
	    
	    ;; <img ... />
	    (,_image-pattern
	     2
	     (let* ((beg (match-beginning 1))
		    (end (match-end 3))
		    (attr-alist (predoc-parse-attribute (match-string-no-properties 2)))
		    (image-p (and window-system (assq 'src attr-alist))))
	       (compose-region beg
			       end
			       predoc-image-icon-string)
	       (when (not image-p)
		 (put-text-property beg
				    end
				    'face predoc-image-face))
	       (when image-p
		 (predoc-remove-image beg
				      end)
		 (predoc-insert-image-file beg
					   end
					   attr-alist)))
	     nil)

	    ;; <a ... /> ... </a>
	    (,_alink-elem-pattern
	     1
	     predoc-alink-face)

	    (,_alink-pattern
	     2
	     (progn
	       (compose-region (match-beginning 1)
			       (+ (match-end 1) 1)
			       (car (string-to-list (match-string 2))))
	       (compose-region (- (match-end    2) 1)
			       (match-end 3)
			       (car 
				(reverse
				 (string-to-list (match-string 2)))))
	       (put-text-property (match-beginning 1)
				  (+ (match-end 1) 1)				  
				  'face predoc-alink-face)
	       nil))
	    )))))


(defun predoc-mode-hookfunc-stuff ()

  ;; Remove character compositions
  (eval '(decompose-region (point-min) (point-max)))
  ;; Install fontification
  (when (and (boundp 'font-lock-keywords)
	     (symbol-value 'font-lock-keywords)
	     (not (featurep 'noweb-mode)))
    ;; This warning is not given if the `noweb-mode' package is installed.
    (predoc-warning "`font-lock-keywords' already set when hook ran."))
  (set (make-local-variable 'predoc-image-cache) '())

  (predoc-install-fontification))


(defun predoc-mode-hookfunc ()

  (defun predoc-modify-wysiwyg-object ()
    (interactive)
    (let ((cur (point)))
      (cond
       ((equal ?< (char-after (point)))
	(if (predoc-get-image (point))
	    (progn
	      (predoc-remove-image (point-at-bol)
				   (point-at-eol))
	      (message "image"))
	  (message"not image"))
	(delete-char 1))
       ((equal ?& (char-after (point)))
	(delete-char 1))
       ((re-search-forward "[a-zA-Z]+;" (point-at-eol) t)
	(goto-char cur)
	(if (eq (match-beginning 0) cur)
	    (progn
	      (insert "&")
	      (backward-char 1))))
       ((re-search-forward "\\([aA][ ]+\\|[iI][mM][gG][ ]+\\)" (point-at-eol) t)
	(goto-char cur)
	(if (eq (match-beginning 0) cur)
	    (progn
	      (insert "<")
	      (backward-char 1))))
       (t
	(message "No match predoc tags.")))))

  (defun predoc-insert-images ()
    "Insert img tags recursivly from current directory."
    (interactive)
    
    (defun search-image-files ()
      "search image files."
      (with-temp-buffer
	(shell-command (mapconcat
			(lambda (x)
			  (format "find ./ -iname '*.%s'; " x))
			predoc-image-prefix-list
			"")
		       (current-buffer))
	(split-string
	 (buffer-substring-no-properties (point-min) (point-max)))))
    
    (mapcar
     (lambda (x)
       (insert
	(format "<img src=\"%s\" width=\"%d\"/> " x predoc-insert-image-size)))
     (search-image-files)))

  (when
      (save-excursion
	(progn
	  (goto-char (point-min))
	  (re-search-forward "^[ ]*<pre>" (point-at-eol) t)))
    
    (predoc-mode-hookfunc-stuff)
    
    ;; Bind Return/Enter key.
    (local-set-key "\C-c\C-c" 'predoc-modify-wysiwyg-object)
    (local-set-key "\C-cl"    'predoc-insert-images)
    (setq mode-name "Predoc")))

(add-hook 'sgml-mode-hook            'predoc-mode-hookfunc)

(provide 'predoc)
;; predoc.el ends here
