;;;-*- mode: lisp-interaction; syntax: elisp -*-;;;
;;
;; "predoc-mode.el" is an WYSIWYG majar mode for Predoc document format.
;;
;;   Copyright (C) 2007 Kiyoka Nishiyama
;;
;;     $Date: 2007-04-11 23:14:37 +0900 (Wed, 11 Apr 2007) $
;;
;; This file is part of Predoc
;;
;; Predoc is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;; 
;; Predoc is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with Predoc; see the file COPYING.
;;
;;
;;
;; How to nstall and How to use:
;;     http://www.netfort.gr.jp/~kiyoka/predoc/
;;
;;
;; ChangeLog:
;;   [0.9.3]
;;     1. Added costomize variable `predoc-curl-program'
;;     2. Added predoc-open-alink() function ( and bind to '[return]' key )
;;     3. Supported "xxx.html  anchor-keyword" string convert into `a' tag.
;;     4. Fixed some bugs.
;;
;;   [0.9.2]
;;     1. Supported `img' tag which recognize "http://" URL.
;;     2. Supported "path/of/image/file.(jpg|png|bmp ... )" string convert into `img' tag.
;;     3. Supported "http://host/path/of/contents... anchor-keyword" string convert into `a' tag.
;;
;;   [0.9.1]
;;     1. Supported image resizing. ( width and height parameter of img tag )
;;     2. Supported predoc-insert-images. ( C-c l )
;;     3. Supported image cache.
;;
;;   [0.9.0]
;;     1. first release
;;
;;
(require 'sgml-mode)

(defconst predoc-version "0.9.3")

(defconst predoc-alink-face  'link)
(defconst predoc-entity-face 'highlight)
(defconst predoc-pre-face    'shadow)
(defconst predoc-image-face  'trailing-whitespace)

(defun predoc-warning (format &rest args)
  (apply 'message (concat "Predoc Warning: " format) args)
  (beep)
  (sleep-for 1))


(defconst predoc-image-icon-string
  "H_")

(defconst predoc-image-prefix-list
  '("bmp"    "gif"    "jpeg"    "jpg"    "png"    "svg"    "tiff"    "tif"    "xbm"    "xpm"))

(defconst predoc-entity-table
  '(
    (lt     .  "<"  )
    (gt     .  ">"  )
    (amp    .  "&"  )
    (quot   .  "\"" )
    (uml    .  ""  )
    (acute  .  ""  )
    (yen    .  ""  )
    (cent   .  ""  )
    (pound  .  ""  )
    (sect   .  ""  )
    (para   .  ""  )
    (copy   .  ""  )
    (reg    .  ""  )
    (not    .  ""  )
    (deg    .  ""  )
    (plusmn .  ""  )
    (times  .  ""  )
    (divide .  ""  )
    (laquo  .  ""   )))


(defcustom predoc-insert-image-size 100
  "Image size of predoc-insert-image command (C-c l)."
  :type  'integer
  :group 'predoc)

(defcustom predoc-convert-program "/usr/bin/convert"
  "The full-path of Imagemagick 'convert' program."
  :type  'string
  :group 'predoc)

(defcustom predoc-curl-program "/usr/bin/curl"
  "The full-path of 'curl' program."
  :type  'string
  :group 'predoc)
 

;; --- utility ---
(defun assoc-ref (alist key)
  (let ((entry (assoc key alist)))
    (when entry
      (cdr entry))))


(defun predoc-entity-string-to-char (_entity)
  (let* (
	 (replace-pair
	  (assq (intern _entity) predoc-entity-table))h)
    (if replace-pair
	(car (string-to-list (cdr replace-pair)))
      nil)))


(defun predoc-insert-image (beg end image &rest args)
  "Display image on the current buffer.
Buffer string between BEG and END are replaced with IMAGE."
  (add-text-properties beg end (list 'display image
				     'intangible image
				     'invisible nil)))



;; insert image data to current-buffer.
;;  e.x.)
;;     (predoc-insert-image-data "http://www.sumibi.org/sumibi/sumibi_picture.png")
(defun predoc-insert-image-data (url)
  (cond 
   ((string-match "http://" url)
    (call-process predoc-curl-program
		  nil
		  '(t nil)
		  nil
		  url))
   (t
    (insert-file-contents url))))


(defun predoc-create-image (url &optional width height)
  (let (data pixel-width pixel-height
	     (m (string-match "http://" url)))
    (when (or (file-readable-p url) m)
      (when (not m)
	(setq url (predoc-expand-full-path url)))
      (setq pixel-width (or width
			    ""))
      (setq pixel-height (or height
			     ""))
      (if (not (file-executable-p predoc-convert-program))
	  (predoc-warning "'%s' does not executable... Image can't be convert size." predoc-convert-program)
	(progn
	  (setq data
		(with-temp-buffer
		  (let ((coding-system-for-read 'binary)
			(coding-system-for-write 'binary)
			(auto-image-file-mode nil))
		    (set-buffer-multibyte nil)
		    (predoc-insert-image-data url)
		    ;; (message (format "1: min = %s  max = %s " (point-min) (point-max)))
		    (cond ((or (< 0 (length pixel-width))
			       (< 0 (length pixel-height)))
			   (call-process-region (point-min) (point-max)
						predoc-convert-program
						t 
						'(t nil)
						nil
						"-" 
						"-resize"
						(format "%sx%s" pixel-width pixel-height)
						"PNG:-"))
			  (t
			   (call-process-region (point-min) (point-max)
						predoc-convert-program
						t 
						'(t nil)
						nil
						"-" 
						"PNG:-")))
		    ;; (message (format "2: min = %s  max = %s " (point-min) (point-max)))
		    (buffer-substring-no-properties (point-min) (point-max)))))
	  (create-image data 'png 'data :ascent 'center))))))


(defun predoc-insert-image-file (beg end attr-alist)
  "Display image on the current buffer
Buffer string between BEG and END are replaced with URL."

  (let ((image (assoc-ref predoc-image-cache attr-alist)))
    (when (not image)
      (let (
	    (src    (assoc-ref attr-alist 'src))
	    (width  (assoc-ref attr-alist 'width))
	    (height (assoc-ref attr-alist 'height)))
	
	;; get the image data file
	(setq image (predoc-create-image
		     src
		     width height))
	;; push to cache
	(push 
	 `(,attr-alist
	   .
	   ,image)
	 predoc-image-cache)
	))
    ;; insert to buffer
    (predoc-insert-image beg end image)))

;;
;; --- test code ---
;;
;;(setq  predoc-image-cache '())
;;
;;(predoc-insert-image-file 
;; (+ (point) 100)
;; (+ (point) 101)
;; '(
;;   (src    . "../doc/img/predoc_logo.png")
;;   (width  . "40")
;;   (height . "40")))
;;
;;(insert (pp predoc-image-cache))


(defun predoc-remove-image (beg end)
  "Remove an image which is inserted between BEG and END."
  (remove-text-properties beg end '(display nil intangible nil)))

(defun predoc-get-image (pos)
  "Get an image object which is indexed pos."
  (get-text-property pos 'display))

(defun predoc-expand-full-path (file)
  "Expand full path from relative path."
  (concat default-directory file))

;;
;; width=10 src="abc\"
;;   -> (("width" . "10") ("src" . "abc"))
;;
;; test pattern:
;;   (assq 'src (predoc-parse-attribute "width=10 src=\"abc\""))
;;
(defun predoc-parse-attribute (str)
  (let ((lst (split-string str "[ ]+"))
	(predoc-attribute-pattern "\\([a-zA-Z]+\\)=[\"]?\\([^ \"]+[/]?\\)"))
    (mapcar
     (lambda (x)
       (if (string-match predoc-attribute-pattern x)
	   (cons
	    (intern (match-string 1 x))
	    (match-string 2 x))
	 nil))
     lst)))

;; 
;; fontification
;;
(defun predoc-install-fontification ()
  (let ((_entity-pattern
	 "\\(&\\)\\([a-zA-Z]+\\)\\(;\\)")
	(_alink-pattern
	 "\\(<[aA][^>]*[>]\\)\\([^<]+\\)\\([<][/][aA][>]\\)")
	(_alink-elem-pattern
	 "[>]\\([^<]+\\)[<][/][aA][>]")
	(_pre-pattern
	 "\\(<[pP][rR][eE]\\|</pre>\\|</PRE>\\)")
	(_image-pattern
 	 "\\(<img[ ]+\\)\\([^>]+\\)\\(>\\)"))

    (set (make-local-variable 'font-lock-defaults)
	 `((
	    ;; &xxxx;
	    (,_entity-pattern
	     (2
	      (let ((ch (predoc-entity-string-to-char (match-string 2))))
		(when ch
		  (compose-region (match-beginning 1)
				  (match-end       3)
				  ch)
		  nil)
		nil)))

	    (,_entity-pattern
	     .
	     predoc-entity-face)

	    ;; <pre> or </pre>
	    (,_pre-pattern
	     .
	     predoc-pre-face)
	    
	    ;; <img ... />
	    (,_image-pattern
	     2
	     (let* ((beg (match-beginning 1))
		    (end (match-end 3))
		    (attr-alist (predoc-parse-attribute (match-string-no-properties 2)))
		    (image-p (and window-system (assq 'src attr-alist))))
	       (compose-region beg
			       end
			       predoc-image-icon-string)
	       (when (not image-p)
		 (put-text-property beg
				    end
				    'face predoc-image-face))
	       (when image-p
		 (predoc-remove-image beg
				      end)
		 (predoc-insert-image-file beg
					   end
					   attr-alist)))
	     nil)

	    ;; <a ... /> ... </a>
	    (,_alink-elem-pattern
	     1
	     predoc-alink-face)

	    (,_alink-pattern
	     2
	     (progn
	       (compose-region (match-beginning 1)
			       (+ (match-end 1) 1)
			       (car (string-to-list (match-string 2))))
	       (compose-region (- (match-end    2) 1)
			       (match-end 3)
			       (car 
				(reverse
				 (string-to-list (match-string 2)))))
	       (put-text-property (match-beginning 1)
				  (+ (match-end 1) 1)				  
				  'face predoc-alink-face)
	       nil))
	    )))))


(defun predoc-mode-hookfunc-stuff ()

  ;; Remove character compositions
  (eval '(decompose-region (point-min) (point-max)))
  ;; Install fontification
  (when (and (boundp 'font-lock-keywords)
	     (symbol-value 'font-lock-keywords)
	     (not (featurep 'noweb-mode)))
    ;; This warning is not given if the `noweb-mode' package is installed.
    (predoc-warning "`font-lock-keywords' already set when hook ran."))
  (set (make-local-variable 'predoc-image-cache) '())

  (predoc-install-fontification))


(defun predoc-open-alink ()
  "Open href source of a tag."
  (interactive)
  (let ((start-pos nil))
    (save-excursion
      (let (
	    (_atag-pattern
	     "<[aA]\\([^>]*\\)[>]"))
	(setq start-pos (re-search-backward _atag-pattern (point-at-bol) t))
	(when start-pos
	  (let* ((attr-alist (predoc-parse-attribute (match-string-no-properties 1)))
		 (href (assoc-ref attr-alist 'href)))
	    (when href
	      (if (string-match "http://" href)
		  (browse-url href)
		(find-file href)))))))
    (if (not start-pos)
	(newline))))


(defun predoc-mode-hookfunc ()

  (defun predoc-modify-wysiwyg-object ()
    (interactive)
    (let (
	  (_alink-pattern
	   "\\(<[aA][^>]*[>]\\)\\([^<]+\\)\\([<][/][aA][>]\\)")
	  (_a_img-pattern
	   "\\([aA][ ]+\\|[iI][mM][gG][ ]+\\)")
	  (_entity-pattern
	   "[a-zA-Z]+;")
	  (_img-pattern
	   (concat "\\(.+\\)\\("
		   (mapconcat
		    (lambda (str) str)
		    predoc-image-prefix-list
		    "\\|")
		   "\\)"))
	  (_url_file-pattern
	   "\\(http://[^\t ]+\\|.+html?\\)[\t ]+\\([^\t ]+\\)"))

      (let ((cur    (point))
	    (str    (buffer-substring-no-properties (point) (point-at-eol))))
	(cond
	 ;; <a href...> ... </a>
	 ((string-match      (concat "^" _alink-pattern) str)
	  (re-search-forward             _alink-pattern  (point-at-eol) t)
	  (let ((start (match-string 1))
		(str   (match-string 2))
		(end   (match-string 3)))
	    (delete-region (match-beginning 1) (match-end 3))
	    (insert (substring start 1))
	    (let ((pos (point)))
	      (insert str)
	      (insert end)
	      (goto-char pos))))
	 ((equal ?< (char-after (point)))
	  (if (predoc-get-image (point))
	      (progn
		(predoc-remove-image (point-at-bol)
				     (point-at-eol))
		(message "image"))
	    (message"not image"))
	  (delete-char 1))
	 ;; &entity;
	 ((equal ?& (char-after (point)))
	  (delete-char 1))
	 ;; entity;
	 ((string-match      (concat "^" _entity-pattern) str)
	  (re-search-forward             _entity-pattern (point-at-eol) t)
	  (goto-char cur)
	  (if (eq (match-beginning 0) cur)
	      (progn
		(insert "&")
		(backward-char 1))))
	 ;; img src=...>   or   a href=...>
	 ((string-match      (concat "^" _a_img-pattern) str)
	  (re-search-forward             _a_img-pattern (point-at-eol) t)
	  (goto-char cur)
	  (if (eq (match-beginning 0) cur)
	      (progn
		(insert "<")
		(backward-char 1))))
	 ;; path/of/image/file.(jpg|png|bmp ... )
	 ((string-match      (concat "^" _img-pattern) str)
	  (re-search-forward             _img-pattern (point-at-eol) t)
	  (goto-char (match-end 2))
	  (insert (format "\" width=\"%d\"/> " predoc-insert-image-size))
	  (goto-char (match-beginning 1))
	  (let ((img-str "<img src=\""))
	    (insert img-str)
	    (backward-char (string-bytes img-str))))
	 ;; http://host/path/of/contents... anchor-keyword
	 ((string-match      (concat "^" _url_file-pattern) str)
	  (re-search-forward             _url_file-pattern (point-at-eol) t)
	  (let ((url (match-string 1))
		(str (match-string 2)))
	    (delete-region (match-beginning 1) (match-end 2))
	    (goto-char (match-beginning 1))
	    (insert (format "<a href=\"%s\" />%s</a>" url str))))
	 ;; No match
	 (t
	  (message "Predoc: <a> link format is URL_or_FILENAME  STRING' . "))))))

  (defun predoc-insert-images ()
    "Insert img tags recursivly from current directory."
    (interactive)
    
    (defun search-image-files ()
      "search image files."
      (with-temp-buffer
	(shell-command (mapconcat
			(lambda (x)
			  (format "find ./ -follow -iname '*.%s'; " x))
			predoc-image-prefix-list
			"")
		       (current-buffer))
	(split-string
	 (buffer-substring-no-properties (point-min) (point-max)))))
    
    (mapcar
     (lambda (x)
       (when (string-match "^./" x)
	 (insert
	  (format "<img src=\"%s\" width=\"%d\"/> " x predoc-insert-image-size))))
     (search-image-files)))

  (when
      (save-excursion
	(progn
	  (goto-char (point-min))
	  (re-search-forward "^[ ]*<[pP][rR][eE]" (point-at-eol) t)))
    
    (predoc-mode-hookfunc-stuff)
    
    ;; Bind Return/Enter key.
    (local-set-key "\C-c\C-c" 'predoc-modify-wysiwyg-object)
    (local-set-key "\C-cl"    'predoc-insert-images)
    (local-set-key [return]   'predoc-open-alink)
    (setq mode-name "Predoc")))

(add-hook 'sgml-mode-hook            'predoc-mode-hookfunc)

(provide 'predoc)
;; predoc.el ends here
