/* main.c - Last modified: 22-Oct-2018 (kobayasy)
 *
 * Copyright (c) 2018 by Yuichi Kobayashi <kobayasy@kobayasy.com>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <limits.h>
#include <signal.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgen.h>
#include <pthread.h>
#include <unistd.h>
#include <sys/wait.h>
#include "psync_utils.h"
#include "psync.h"
#include "popen3.h"

#define VERSION "1.7"
#define PROTID 0x01705370  /* 'p', 'S', 'p', 1 */
#define CONFFILE ".psync.conf"
#define EXPIRE_DEFAULT 400  /* [days] */

static const char *errmes(int status) {
    static const char *mes[] = {
        [0]              = "No error",
        [1]              = "Unknown",
        [-ERROR_FTYPE]   = "File type",
        [-ERROR_FPERM]   = "File permission",
        [-ERROR_FMAKE]   = "Make file",
        [-ERROR_FOPEN]   = "Open file",
        [-ERROR_FWRITE]  = "Write file",
        [-ERROR_FREAD]   = "Read file",
        [-ERROR_FLINK]   = "Link file",
        [-ERROR_FREMOVE] = "Remove file",
        [-ERROR_FMOVE]   = "Move file",
        [-ERROR_SWRITE]  = "Write file-stat",
        [-ERROR_SREAD]   = "Read file-stat",
        [-ERROR_SUPLD]   = "Upload file-stat",
        [-ERROR_SDNLD]   = "Download file-stat",
        [-ERROR_FUPLD]   = "Upload file",
        [-ERROR_FDNLD]   = "Download file",
        [-ERROR_DMAKE]   = "Make data-file",
        [-ERROR_DOPEN]   = "Open data-file",
        [-ERROR_DWRITE]  = "Write data-file",
        [-ERROR_DREAD]   = "Read data-file",
        [-ERROR_MEMORY]  = "Memory",
        [-ERROR_SYSTEM]  = "System",
        [-ERROR_STOP]    = "Interrupted"
    };
    int n;

    if (!ISERR(status))
        n = 0;
    else if (status < -(sizeof(mes)/sizeof(mes[0])))
        n = 1;
    else
        n = -status;
    return mes[n];
}

typedef struct {
    struct {
        bool set;
        struct sigaction act;
    } sighup, sigint, sigterm, sigpipe;
} SIGACT;

static void sigactinit(SIGACT *oldact) {
    oldact->sighup.set = false;
    oldact->sigint.set = false;
    oldact->sigterm.set = false;
    oldact->sigpipe.set = false;
}

static void sigactreset(SIGACT *oldact) {
    if (oldact->sighup.set)
        sigaction(SIGHUP, &oldact->sighup.act, NULL), oldact->sighup.set = false;
    if (oldact->sigint.set)
        sigaction(SIGINT, &oldact->sigint.act, NULL), oldact->sigint.set = false;
    if (oldact->sigterm.set)
        sigaction(SIGTERM, &oldact->sigterm.act, NULL), oldact->sigterm.set = false;
    if (oldact->sigpipe.set)
        sigaction(SIGPIPE, &oldact->sigpipe.act, NULL), oldact->sigpipe.set = false;
}

static int sigactset(void (*handler)(int signo), SIGACT *oldact) {
    int status = INT_MIN;
    struct sigaction act;

    act.sa_handler = handler;
    act.sa_flags = 0;
    if (sigemptyset(&act.sa_mask) == -1) {
        status = -1;
        goto error;
    }
    if (sigaction(SIGHUP, &act, &oldact->sighup.act) == -1) {
        sigactreset(oldact);
        status = -1;
        goto error;
    }
    oldact->sighup.set = true;
    if (sigaction(SIGINT, &act, &oldact->sigint.act) == -1) {
        sigactreset(oldact);
        status = -1;
        goto error;
    }
    oldact->sigint.set = true;
    if (sigaction(SIGTERM, &act, &oldact->sigterm.act) == -1) {
        sigactreset(oldact);
        status = -1;
        goto error;
    }
    oldact->sigterm.set = true;
    if (sigaction(SIGPIPE, &act, &oldact->sigpipe.act) == -1) {
        sigactreset(oldact);
        status = -1;
        goto error;
    }
    oldact->sigpipe.set = true;
    status = 0;
error:
    return status;
}

typedef struct s_clist {
    struct s_clist *next, *prev;
    char *dirname;
    time_t expire;
    char name[1];
} CLIST;

static CLIST *new_c(CLIST *clist) {
    LIST_NEW(clist);
    clist->dirname = clist->name;
    clist->expire = 0;
    return clist;
}

static CLIST *add_c(CLIST *clist, const char *name, const char *dirname) {
    CLIST *cnew = NULL;
    size_t length;

    if (!*name)
        goto error;
    length = strlen(name) + 1;
    cnew = malloc(offsetof(CLIST, name) + length + strlen(dirname) + 1);
    if (cnew == NULL)
        goto error;
    strcpy(cnew->name, name);
    cnew->dirname = strcpy(cnew->name + length, dirname);
    cnew->expire = 0;
    LIST_INSERT_NEXT(cnew, clist);
error:
    return cnew;
}

static int each_next_c(CLIST *clist,
                       int (*func)(CLIST *c, void *data),
                                             void *data,
                       volatile sig_atomic_t *stop ) {
    int status = INT_MIN;
    CLIST *c = clist;
    CLIST *cnext;

    if (*c->name)
        goto error;
    c = clist->next;
    while (*c->name) {
        ONSTOP(stop, -1);
        cnext = c->next;
        status = func(c, data);
        if (ISERR(status))
            goto error;
        c = cnext;
    }
    status = 0;
error:
    return status;
}

#if 0  // not used
static int each_prev_c(CLIST *clist,
                       int (*func)(CLIST *c, void *data),
                                             void *data,
                       volatile sig_atomic_t *stop ) {
    int status = INT_MIN;
    CLIST *c = clist;
    CLIST *cprev;

    if (*c->name)
        goto error;
    c = clist->prev;
    while (*c->name) {
        ONSTOP(stop, -1);
        cprev = c->prev;
        status = func(c, data);
        if (ISERR(status))
            goto error;
        c = cprev;
    }
    status = 0;
error:
    return status;
}
#endif  /* #if 0  // not used */

static int delete_func(CLIST *c, void *data) {
    int status = INT_MIN;

    LIST_DELETE(c);
    free(c);
    status = 0;
    return status;
}

#define CONFBUF 1024
#define CONFTOK " \t\r\n"
#define CONFREM '#'
static int get_clist(CLIST *clist, const char *confname) {
    int status = INT_MIN;
    FILE *fp = NULL;
    unsigned int line;
    char buffer[CONFBUF], *s;
    char *argv[CONFBUF/2];
    unsigned int argc;
    int seek;

    fp = fopen(confname, "r");
    if (fp == NULL) {
        fprintf(stderr, "Error: Can not open \"~/%s\".\n", confname);
        status = -1;
        goto error;
    }
    line = 0;
    while (fgets(buffer, sizeof(buffer), fp)) {
        ++line;
        s = strchr(buffer, CONFREM);
        if (s != NULL)
            *s = 0;
        s = strtok(buffer, CONFTOK);
        argv[argc = 0] = s;
        while (s != NULL) {
            s = strtok(NULL, CONFTOK);
            argv[++argc] = s;
        }
        switch (argc) {
        case 0:
            break;
        case 1:
            fprintf(stderr, "Error: Line %u in \"~/%s\": Too few parameters \"%s\".\n", line, confname, argv[0]);
            status = -1;
            goto error;
        case 2:
        case 3:
            LIST_SEEK_NEXT(clist, argv[0], seek);
            if (!seek) {
                fprintf(stderr, "Error: Line %u in \"~/%s\": Redefined \"%s\".\n", line, confname, argv[0]);
                status = -1;
                goto error;
            }
            clist = add_c(clist, argv[0], argv[1]);
            if (clist == NULL) {
                fprintf(stderr, "Error: Out of memory.\n");
                status = -1;
                goto error;
            }
            if (argv[2] != NULL) {
                clist->expire = strtoll(argv[2], &s, 10);
                if (*s) {
                    fprintf(stderr, "Error: Line %u in \"~/%s\": Invarid parameter \"%s\".\n", line, confname, argv[2]);
                    status = -1;
                    goto error;
                }
            }
            else
                clist->expire = EXPIRE_DEFAULT;
            break;
        default:
            fprintf(stderr, "Error: Line %u in \"~/%s\": Too many parameters \"%s\".\n", line, confname, argv[2]);
            status = -1;
            goto error;
        }
    }
    status = 0;
error:
    if (fp != NULL)
        fclose(fp);
    return status;
}

static int greeting(int fdin, int fdout) {
    int status = INT_MIN;
    uint32_t id = PROTID;

    WRITE_ONERR(id, fdout, -1);
    READ_ONERR(id, fdin, -1);
    if (id != PROTID) {
        status = -1;
        goto error;
    }
    status = 0;
error:
    return status;
}

#define INFOBUFFER_SIZE 256
typedef struct {
    void (*func)(const char *s);
    int fd;
} INFO_PARAM;
static void *info_thread(void *data) {
    INFO_PARAM *param = data;
    char buffer[INFOBUFFER_SIZE], *w, *s, *r;
    size_t size;
    ssize_t n;

    w = buffer, size = sizeof(buffer)-1;
    n = read(param->fd, w, size);
    if (n == -1)
        goto error;
    while (n > 0) {
        w[n] = 0, s = strchr(w, '\n');
        w += n, size -= n;
        if (s != NULL) {
            r = buffer;
            do {
                *s++ = 0;
                param->func(r);
                s = strchr(r = s, '\n');
            } while (s != NULL);
            n = r - buffer;
            w -= n, size += n;
            strcpy(buffer, r);
        }
        n = read(param->fd, w, size);
        if (n == -1)
            goto error;
    }
error:
    return NULL;
}

#define STATUS_NOTREADYLOCAL       1
#define STATUS_NOTREADYREMOTE      2
#define STATUS_PROTOCOLERROR    -101
#define STATUS_HANDSHAKEERROR   -102
#define STATUS_SYSTEMERROR      -103
#define STATUS_ARGUMENTERROR    -201
#define STATUS_ENVIRONMENTERROR -202
#define STATUS_CONFFILEERROR    -203
#define PROGRESS_BUFFER 256
static void info_func(const char *message) {
    static struct {
        intmax_t size;
        int length;
        char header[PROGRESS_BUFFER];
        char body[PROGRESS_BUFFER];
    } upload, download;
    static unsigned int newline = 1;
    int n;
    char *p;
    intmax_t progress;

    switch (*message++) {
    case 'L':
        if (strchr("+-", *message) != NULL) {
            n = strtol(message, &p, 10);
            message = p;
            if (strchr("+-", *message) == NULL) {
                while (strchr(" ", *message) != NULL)
                    ++message;
                switch (n) {
                case 0:
                    if (!newline)
                        fputc('\n', stdout), newline = 1;
                    break;
                case STATUS_NOTREADYLOCAL:
                    fputs("\n Skip, local not ready.\n" + newline, stdout), newline = 1;
                    break;
                case STATUS_NOTREADYREMOTE:
                    fputs("\n Skip, remote not ready.\n" + newline, stdout), newline = 1;
                    break;
                default:
                    if (ISERR(n))
                        fprintf(stdout, "\n Local error: %s\n" + newline, errmes(n)), newline = 1;
                    else
                        fprintf(stdout, "\n Local done: %d\n" + newline, n), newline = 1;
                }
            }
        }
        else {
            while (strchr(" ", *message) != NULL)
                ++message;
            fprintf(stdout, "\n%s\n" + newline, message), newline = 1;
        }
        break;
    case 'R':
        if (strchr("+-", *message) != NULL) {
            n = strtol(message, &p, 10);
            message = p;
            if (strchr("+-", *message) == NULL) {
                while (strchr(" ", *message) != NULL)
                    ++message;
                switch (n) {
                case 0:
                case STATUS_NOTREADYLOCAL:
                case STATUS_NOTREADYREMOTE:
                    break;
                default:
                    if (ISERR(n))
                        fprintf(stdout, "\n Remote error: %s (%s)\n" + newline, errmes(n), message), newline = 1;
                    else
                        fprintf(stdout, "\n Remote done: %d (%s)\n" + newline, n, message), newline = 1;
                }
            }
        }
        break;
    case 'P':
        if (strchr("+-", *message) != NULL) {
            upload.size = strtoll(message, &p, 10);
            upload.length = p - message - 1;
            message = p;
            if (upload.size > 0) {
                n = upload.length * 2 + 1;
                sprintf(upload.header, " %-*.*s", n, n, "Upload");
                progress = 0;
                sprintf(upload.body, " %*jd/%*jd", upload.length, progress, upload.length, upload.size);
            }
            else {
                *upload.header = 0;
                *upload.body = 0;
            }
        }
        if (strchr("+-", *message) != NULL) {
            download.size = strtoll(message, &p, 10);
            download.length = p - message - 1;
            message = p;
            if (download.size > 0) {
                n = download.length * 2 + 1;
                sprintf(download.header, " %-*.*s", n, n, "Download");
                progress = 0;
                sprintf(download.body, " %*jd/%*jd", download.length, progress, download.length, download.size);
            }
            else {
                *download.header = 0;
                *download.body = 0;
            }
        }
        break;
    case 'U':
        if (strchr("+-", *message) != NULL) {
            progress = strtoll(message, &p, 10);
            message = p;
            sprintf(upload.body, " %*jd/%*jd", upload.length, progress, upload.length, upload.size);
            if (newline)
                fprintf(stdout, " %s%s\n", upload.header, download.header);
            fprintf(stdout, "\r %s%s" + newline, upload.body, download.body), newline = 0;
            fflush(stdout);
        }
        break;
    case 'D':
        if (strchr("+-", *message) != NULL) {
            progress = strtoll(message, &p, 10);
            message = p;
            sprintf(download.body, " %*jd/%*jd", download.length, progress, download.length, download.size);
            if (newline)
                fprintf(stdout, " %s%s\n", upload.header, download.header);
            fprintf(stdout, "\r %s%s" + newline, upload.body, download.body), newline = 0;
            fflush(stdout);
        }
        break;
    }
}

static struct {
    volatile sig_atomic_t stop;
    int verbose;
} g = {
    .stop    = 0,
    .verbose = 1
};

static void sighandler(int signo) {
    g.stop = 1;
}

typedef struct {
    int fdin, fdout;
    int info;
} RUN_PARAM;
static int run_local_func(CLIST *clist, void *data) {
    int status = INT_MIN;
    RUN_PARAM *param = data;
    SIGACT oldact;
    size_t length, n;
    int rack, lack;
    char buffer[128];
    PSYNC *psync = NULL;

    sigactinit(&oldact);
    n = length = strlen(clist->name);
    WRITE_ONERR(n, param->fdout, STATUS_HANDSHAKEERROR);
    if (write_sp(param->fdout, clist->name, length) != length) {
        status = STATUS_HANDSHAKEERROR;
        goto error;
    }
    READ_ONERR(rack, param->fdin, STATUS_HANDSHAKEERROR);
    if (!rack) {
        if (g.verbose > 0) {
            length = sprintf(buffer, "L%s\n", clist->name);
            write(param->info, buffer, length);
        }
        ONERR(sigactset(sighandler, &oldact), STATUS_SYSTEMERROR);
        psync = psync_new(clist->dirname, clist->expire * (60*60*24), &g.stop);
        lack = psync == NULL ? -1 : 0;
        WRITE_ONERR(lack, param->fdout, STATUS_HANDSHAKEERROR);
        READ_ONERR(rack, param->fdin, STATUS_HANDSHAKEERROR);
        if (!lack) {
            if (!rack) {
                psync->fdin = param->fdin, psync->fdout = param->fdout;
                if (g.verbose > 1)
                    psync->info = param->info;
                status = psync_run(true, psync);
            }
            else
                status = STATUS_NOTREADYREMOTE;
            psync_free(psync), psync = NULL;
        }
        else
            status = STATUS_NOTREADYLOCAL;
        sigactreset(&oldact);
        if (g.verbose > 0) {
            length = sprintf(buffer, "L%+d %s\n", status, clist->name);
            write(param->info, buffer, length);
        }
        if (ISERR(status))
            goto error;
    }
    else
        status = 0;
error:
    if (psync != NULL)
        psync_free(psync);
    sigactreset(&oldact);
    return status;
}
static int run_local(int fdin, int fdout, int fderr, pid_t pid, void *data) {
    int status = INT_MIN;
    CLIST *clist = data;
    RUN_PARAM run_param = {fdout, fdin, -1};
    INFO_PARAM rinfo_param = {info_func, -1};
    pthread_t rinfo_tid;
    int info_pipe[2] = {-1, -1};
    INFO_PARAM linfo_param = {info_func, -1};
    pthread_t linfo_tid;
    size_t length;

    ONERR(greeting(run_param.fdin, run_param.fdout), STATUS_PROTOCOLERROR);
    if (g.verbose) {
        rinfo_param.fd = fderr;
        if (pthread_create(&rinfo_tid, NULL, info_thread, &rinfo_param) != 0) {
            status = STATUS_SYSTEMERROR;
            goto error;
        }
        if (pipe(info_pipe) == -1) {
            status = STATUS_SYSTEMERROR;
            goto error;
        }
        linfo_param.fd = info_pipe[0];
        if (pthread_create(&linfo_tid, NULL, info_thread, &linfo_param) != 0) {
            status = STATUS_SYSTEMERROR;
            goto error;
        }
        run_param.info = info_pipe[1];
    }
    status = each_next_c(clist, run_local_func, &run_param, NULL);
    if (ISERR(status))
        goto error;
    length = 0;
    WRITE_ONERR(length, run_param.fdout, STATUS_HANDSHAKEERROR);
    if (g.verbose) {
        close(info_pipe[1]), info_pipe[1] = -1;
        if (pthread_join(linfo_tid, NULL) != 0) {
            status = STATUS_SYSTEMERROR;
            goto error;
        }
        close(info_pipe[0]), info_pipe[0] = -1;
        if (pthread_join(rinfo_tid, NULL) != 0) {
            status = STATUS_SYSTEMERROR;
            goto error;
        }
    }
error:
    if (g.verbose) {
        if (info_pipe[0] != -1)
            close(info_pipe[0]), info_pipe[0] = -1;
        if (info_pipe[1] != -1)
            close(info_pipe[1]), info_pipe[1] = -1;
    }
    if (pid != 0)
        waitpid(pid, NULL, 0);
    return status;
}

static int run_remote(int fdin, int fdout, int fderr, pid_t pid, void *data) {
    int status = INT_MIN;
    CLIST *clist = data;
    SIGACT oldact;
    size_t length;
    char name[CONFBUF];
    int seek;
    int lack, rack;
    PSYNC *psync = NULL;

    sigactinit(&oldact);
    ONERR(greeting(fdin, fdout), STATUS_PROTOCOLERROR);
    READ_ONERR(length, fdin, STATUS_HANDSHAKEERROR);
    while (length > 0) {
        if (length > sizeof(name)-1) {
            status = STATUS_SYSTEMERROR;
            goto error;
        }
        if (read_sp(fdin, name, length) != length) {
            status = STATUS_HANDSHAKEERROR;
            goto error;
        }
        name[length] = 0;
        LIST_SEEK_NEXT(clist, name, seek);
        lack = seek ? -1 : 0;
        WRITE_ONERR(lack, fdout, STATUS_HANDSHAKEERROR);
        if (!lack) {
            fprintf(stderr, "R%s\n", clist->name);
            ONERR(sigactset(sighandler, &oldact), STATUS_SYSTEMERROR);
            psync = psync_new(clist->dirname, clist->expire * (60*60*24), &g.stop);
            lack = psync == NULL ? -1 : 0;
            WRITE_ONERR(lack, fdout, STATUS_HANDSHAKEERROR);
            READ_ONERR(rack, fdin, STATUS_HANDSHAKEERROR);
            if (!lack) {
                if (!rack) {
                    psync->fdin = fdin, psync->fdout = fdout;
                    status = psync_run(false, psync);
                }
                else
                    status = STATUS_NOTREADYREMOTE;
                psync_free(psync), psync = NULL;
            }
            else
                status = STATUS_NOTREADYLOCAL;
            sigactreset(&oldact);
            fprintf(stderr, "R%+d %s\n", status, clist->name);
            if (ISERR(status))
                goto error;
        }
        READ_ONERR(length, fdin, STATUS_HANDSHAKEERROR);
    }
    status = 0;
error:
    if (psync != NULL)
        psync_free(psync);
    sigactreset(&oldact);
    if (pid != 0)
        waitpid(pid, NULL, 0);
    return status;
}

int main(int argc, char *argv[]) {
    int status = INT_MIN;
    char *argv0 = *argv;
    CLIST clist;
    enum {MODE_HELP, MODE_RUN, MODE_ERROR} mode = MODE_HELP;
    char *args[] = {"ssh", "-qC", NULL, "psync", "--remote", 0};
    char *s;

    new_c(&clist);
    while (*++argv)
        if (**argv == '-') {
            if (!strcmp(*argv, "--help")) {
                mode = MODE_HELP;
                break;
            }
            else if (!strcmp(*argv, "--remote")) {
                mode = MODE_RUN;
                args[2] = NULL;
            }
            else if (!strcmp(*argv, "-q") ||
                     !strcmp(*argv, "--quiet") )
                g.verbose = 0;
            else if (!strcmp(*argv, "-v") ||
                     !strcmp(*argv, "--verbose") )
                g.verbose = 2;
            else {
                fprintf(stderr, "Error: Invarid option: %s\n", *argv);
                mode = MODE_ERROR;
                status = STATUS_ARGUMENTERROR;
                break;
            }
        }
        else if (args[2] == NULL) {
            mode = MODE_RUN;
            args[2] = *argv;
        }
        else {
            fprintf(stderr, "Error: Invarid argument: %s\n", *argv);
            mode = MODE_ERROR;
            status = STATUS_ARGUMENTERROR;
            break;
        }
    s = getenv("HOME");
    if (s == NULL){
        fputs("Error: $HOME is not set.\n", stderr);
        mode = MODE_ERROR;
        status = STATUS_ENVIRONMENTERROR;
    }
    else if (chdir(s) == -1) {
        fprintf(stderr, "Error: Can not change diractory to %s\n", s);
        mode = MODE_ERROR;
        status = STATUS_ENVIRONMENTERROR;
    }
    else if (ISERR(get_clist(&clist, CONFFILE))) {
        mode = MODE_ERROR;
        status = STATUS_CONFFILEERROR;
    }
    switch (mode) {
    case MODE_RUN:
        if (args[2] != NULL) {
            status = popen3(args, run_local, &clist);
            switch (status) {
            case STATUS_PROTOCOLERROR:
                fputs("Error: Protocol\n", stderr);
                break;
            case STATUS_HANDSHAKEERROR:
                fputs("Error: Handshake\n", stderr);
                break;
            case STATUS_SYSTEMERROR:
                fputs("Error: System\n", stderr);
                break;
            }
        }
        else
            status = run_remote(STDIN_FILENO, STDOUT_FILENO, STDERR_FILENO, 0, &clist);
        break;
    case MODE_ERROR:
        fputc('\n', stdout);
    default:
        s = basename(argv0);
        fprintf(stdout, "pSync version %s (protocol %c%c%c%u)\n",
                VERSION, PROTID, PROTID >> 8, PROTID >> 16, PROTID >> 24 );
        fputc('\n', stdout);
        fprintf(stdout, "Usage: %s [-v|-q] [USER@]HOST\n", s);
        fprintf(stdout, "       %s --help\n", s);
        fputc('\n', stdout);
        fputs("USER@HOST\n", stdout);
        fputs("  HOST           hostname\n", stdout);
        fputs("  USER           username (default: current login user)\n", stdout);
        fputc('\n', stdout);
        fputs("subcommand\n", stdout);
        fputs(" --help          show this help\n", stdout);
        fputc('\n', stdout);
        fputs("options\n", stdout);
        fputs("  -v, --verbose  increase verbosity\n", stdout);
        fputs("  -q, --quiet    suppress non-error messages\n", stdout);
        fputc('\n', stdout);
    }
    each_next_c(&clist, delete_func, NULL, NULL);
    if (status < 0)
        status = status < -255 ? 255 : -status;
    else
        status = 0;
    return status;
}
