//
//  PT1Device.cpp
//

#include <IOKit/IOLib.h>
#include <IOKit/assert.h>
#include "PT1Device.h"

#define __iomem
typedef UInt8 __u8;
typedef UInt16 __u16;
typedef UInt32 __u32;

extern "C" {
#include "pt1_com.h"
#include "pt1_i2c.h"
#include "pt1_tuner.h"
#include "pt1_tuner_data.h"

extern __u32 getfrequency_add(__u32 channel);
extern __u32 getfrequency(__u32 channel, int addfreq);

}

static int i2c_address[MAX_CHANNEL] = {T0_ISDB_S, T0_ISDB_T, T1_ISDB_S, T1_ISDB_T};

#define super IOService

// Even though we are defining the convenience macro super for the superclass, you must use the actual class name
// in the OS*MetaClass macros. Note that the class name is different when supporting Mac OS X 10.4.

#if MAC_OS_X_VERSION_MIN_REQUIRED <= MAC_OS_X_VERSION_10_4
OSDefineMetaClassAndStructors(PT1Device, IOService)
#else
OSDefineMetaClassAndStructors(PT1Device, IOService)
#endif

#pragma mark
// We override init only to log that it has been called to make it easier to follow the driver's lifecycle.
// Production drivers would only need to override init if they want to initialize data members.
bool PT1Device::init(OSDictionary* dictionary)
{
    if (!super::init(dictionary))
    {
        return false;
	}

	// This IOLog must follow super::init because getName relies on the superclass initialization.
	IOLog("%s[%p]::%s(%p)\n", getName(), this, __FUNCTION__, dictionary);

    // This is where you could set the initial value of your driver's data members.

    // 初期化
    _lnbPower = LNB_POWER_OFF;
    _lnbPowerWhenClose = LNB_POWER_OFF;
    _tunerPowerReset = kTunerPowerOff;
    for (int i = 0; i < 4; ++i)
    {
        _streamEnable[i] = false;
        _streamGray[i] = 0;
    }
    _transfer = false;

    // 初期化：DMAバッファ関連
    _bufferInfo.LockSize     = 0;
    _bufferInfo.VirtualSize  = 0;
    _bufferInfo.VirtualCount = 0;
    _bmdAddress = NULL;
    _bmdData = NULL;
    _dmaCmdAddress = NULL;
    _dmaCmdData    = NULL;

	return true;
}

// We override probe only to log that it has been called to make it easier to follow the driver's lifecycle.
// Production drivers can override probe if they need to make an active decision whether the driver is appropriate for
// the provider.
IOService* PT1Device::probe(IOService* provider, SInt32* score)
{
	IOLog("%s[%p]::%s(%p, %p)\n", getName(), this, __FUNCTION__, provider, score);
        
    IOService *res = super::probe(provider, score);

    return res;
}

bool PT1Device::start(IOService* provider)
{
	IOLog("%s[%p]::%s(%p)\n", getName(), this, __FUNCTION__, provider);
    
    if (!super::start(provider))
    {
        return false;
    }

    assert( OSDynamicCast( IOPCIDevice, provider ));
    _device = (IOPCIDevice *) provider;

    //
    // RevisionIDチェック
    //
    uint8_t revision = _device->configRead8(kIOPCIConfigRevisionID);
    if (revision != 0x01)
    {
        IOLog("RevisionID Check NG. %d\n", revision);
        return false;
    }
    IOLog("RevisionID = 0x%02X.\n", revision);

#if 0
    //
    // CheckBus
    //
    for (uint i = 0; i < 2; ++i)
    {
        uint data = (i == 0) ? 0x00000000 : 0xffffffff;
        for (uint j = 0; j < 32; ++j)
        {
            uint writeData = data ^ (BIT_1(1) << i);
            uint readData;

            fPCIDevice->configWrite32(A_CONFIG_DATA, writeData);
            readData = fPCIDevice->configRead32(A_CONFIG_DATA);

            if (readData != writeData)
            {
                IOLog("CheckBus PH1 NG.\n");
                return false;
            }
        }
    }

    uint writeData = 0;
    uint readData;

    fPCIDevice->configWrite32(A_CONFIG_DATA, writeData);
    readData = fPCIDevice->configRead32(A_CONFIG_DATA);

    if (readData != writeData)
    {
        IOLog("CheckBus PH2 NG.\n");
        return false;
    }

    for (uint i = 0; i < 4; ++i)
    {
        fPCIDevice->configWrite8(A_CONFIG_DATA + i, 0xff);
        readData = fPCIDevice->configRead32(A_CONFIG_DATA);
        writeData <<= 8;
        writeData |= 0xff;
        if (readData != writeData)
        {
            IOLog("CheckBus PH3 NG.\n");
            return false;
        }
    }
#endif

    //
    // ???
    //
    _device->configWrite16(kIOPCIConfigCommand, 0x03);


    //
    // Enable memory response from the card
    //
    IOLog("setMemoryEnable() = %d\n", _device->setMemoryEnable(true));
    IOLog("setMemoryEnable() = %d\n", _device->setMemoryEnable(true));

    //
    // Bus Master
    //
    IOLog("setBusMasterEnable() = %d\n", _device->setBusMasterEnable(true));
    IOLog("setBusMasterEnable() = %d\n", _device->setBusMasterEnable(true));


    /* map a range based on its config space base address register,
     * this is how the driver gets access to its memory mapped registers
     * the getVirtualAddress() method returns a kernel virtual address
     * for the register mapping */
    
    _map = _device->mapDeviceMemoryWithRegister(kIOPCIConfigBaseAddress0);
    if (!_map)
    {
        IOLog("Can't map device memory\n");
        return false;
    }
    _reg = (void *)_map->getVirtualAddress();

    /* publish ourselves so clients can find us */
    registerService();

    return( true );
}


// We override stop only to log that it has been called to make it easier to follow the driver's lifecycle.
void PT1Device::stop(IOService* provider)
{
	IOLog("%s[%p]::%s(%p)\n", getName(), this, __FUNCTION__, provider);

    if (_map != NULL)
    {
        _map->release();
        _reg = NULL;
    }
    
    super::stop(provider);
}

// We override free only to log that it has been called to make it easier to follow the driver's lifecycle.
void PT1Device::free(void)
{
	IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);
    
    super::free();
}

bool PT1Device::open(IOService *forClient, IOOptionBits options, void *arg)
{
    IOLog("%s[%p]::%s(%p, %d, %p)\n", getName(), this, __FUNCTION__, forClient, (int)options, arg);

    uint status = kStatusInternalError;

    if (super::open(forClient, options, arg))
    {
        if (xc3s_init(_reg) < 0)
        {
            super::close(forClient);
        }
        else
        {
            status = kStatusSuccess;
        }
    }

    if (arg != NULL)
    {
        *(uint32_t *)arg = status;
    }

    IOLog("%s[%p]::%s() %s\n", getName(), this, __FUNCTION__, ((status == kStatusSuccess) ? "ok." : "ng."));

    return (status == kStatusSuccess);
}

#pragma mark 未実装
void PT1Device::close(IOService *forClient, IOOptionBits options)
{
    IOLog("%s[%p]::%s(%p, %d)\n", getName(), this, __FUNCTION__, forClient, (int)options);

    if (_transfer)
    {
        uint32_t status;
        setTransferEnable(&status, false);
    }

    clearBuffer();
#if 0
			_ASSERTE(mDevice2);
			mDevice2->SetAT45DB041D(mAT45DB041D);

			for (uint i=0; i<sizeof(mTuner)/sizeof(*mTuner); i++) {
				Tuner *tuner = &mTuner[i];
				_ASSERTE(tuner);
				delete_(tuner->TC90512);
				delete_(tuner->TDA6651);
				delete_(tuner->QM1D1B0004);

				for (uint j=0; j<2; j++) {
					tuner->Sleep           [j] = true;
					tuner->FrequencyChannel[j] = 0;
					tuner->FrequencyOffset [j] = 0;
				}				
			}

			delete_(mXC3S);

			_ASSERTE(mDevice2);
			mDevice2->SetXC3S(mXC3S);

			if (mDevice) {
				mDevice->Delete();
				mDevice = NULL;
			}
#endif
    super::close(forClient, options);
}

// We override willTerminate only to log that it has been called to make it easier to follow the driver's lifecycle.
//
// willTerminate is called at the beginning of the termination process. It is a notification
// that a provider has been terminated, sent before recursing up the stack, in root-to-leaf order.
//
// This is where any pending I/O should be terminated. At this point the user client has been marked
// inactive and any further requests from the user process should be returned with an error.
bool PT1Device::willTerminate(IOService* provider, IOOptionBits options)
{
	IOLog("%s[%p]::%s(%p, %ld)\n", getName(), this, __FUNCTION__, provider, options);
	
	return super::willTerminate(provider, options);
}


// We override didTerminate only to log that it has been called to make it easier to follow the driver's lifecycle.
//
// didTerminate is called at the end of the termination process. It is a notification
// that a provider has been terminated, sent after recursing up the stack, in leaf-to-root order.
bool PT1Device::didTerminate(IOService* provider, IOOptionBits options, bool* defer)
{
	IOLog("%s[%p]::%s(%p, %ld, %p)\n", getName(), this, __FUNCTION__, provider, options, defer);
	
	return super::didTerminate(provider, options, defer);
}


// We override terminate only to log that it has been called to make it easier to follow the driver's lifecycle.
// Production drivers will rarely need to override terminate. Termination processing should be done in
// willTerminate or didTerminate instead.
bool PT1Device::terminate(IOOptionBits options)
{
    bool	success;
    
	IOLog("%s[%p]::%s(%ld)\n", getName(), this, __FUNCTION__, options);

    success = super::terminate(options);
    
    return success;
}


// We override finalize only to log that it has been called to make it easier to follow the driver's lifecycle.
// Production drivers will rarely need to override finalize.
bool PT1Device::finalize(IOOptionBits options)
{
    bool	success;
    
	IOLog("%s[%p]::%s(%ld)\n", getName(), this, __FUNCTION__, options);
    
    success = super::finalize(options);
    
    return success;
}

IOReturn PT1Device::clientMemoryForType(UInt32 type, IOOptionBits * options, IOMemoryDescriptor ** memory)
{
	IOLog("%s[%p]::%s(%ld)\n", getName(), this, __FUNCTION__, type);

    IOReturn result = kIOReturnNoMemory;

    *memory = NULL;
    *options = 0;

    switch (type)
    {
    case kIODefaultMemoryType:
        if (_bmdData != NULL)
        {
            // You don't need to balance with a release() because the user client does the release() for you.
            _bmdData->retain();

            *memory = _bmdData;
            *options = 0;
            result = kIOReturnSuccess;
        }
        break;

    default:
        IOLog("PT1Device - Unknown memory type %ld\n", type);
        break;
    }
    
    return result;
}

void PT1Device::clearBuffer(void)
{
    IOLog("clearBuffer().\n");
    if (_dmaCmdAddress != NULL)
    {
        if (_dmaCmdAddress->getMemoryDescriptor() != NULL)
        {
            _dmaCmdAddress->clearMemoryDescriptor();
        }
        _dmaCmdAddress->release();
        _dmaCmdAddress = NULL;
    }
    if (_dmaCmdData != NULL)
    {
        if (_dmaCmdData->getMemoryDescriptor() != NULL)
        {
            _dmaCmdData->clearMemoryDescriptor();
        }
        _dmaCmdData->release();
        _dmaCmdData = NULL;
    }
    if (_bmdAddress != NULL)
    {
        _bmdAddress->release();
        _bmdAddress = NULL;
    }
    if (_bmdData != NULL)
    {
        _bmdData->release();
        _bmdData = NULL;
    }

    _bufferInfo.VirtualSize = 0;
    _bufferInfo.VirtualCount = 0;
    _bufferInfo.LockSize = 0;
}

#pragma mark
#pragma mark ############### I/O用メソッド ###############
UInt8 PT1Device::memoryRead8(UInt16 offset)
{
    UInt8 value = 0xff;
    if (_map)
    {
        value = *(UInt8 *)(_map->getVirtualAddress() + offset);
    }
    return value;
}

UInt16 PT1Device::memoryRead16(UInt16 offset)
{
    UInt16 value = 0xff;
    if (_map)
    {
        value = *(UInt16 *)(_map->getVirtualAddress() + offset);
    }
    return value;
}

UInt32 PT1Device::memoryRead32(UInt16 offset)
{
    UInt32 value = 0xff;
    if (_map)
    {
        value = *(UInt32 *)(_map->getVirtualAddress() + offset);
    }
    return value;
}

void PT1Device::memoryWrite8(UInt16 offset, UInt8 value)
{
    if (_map)
    {
        *(UInt8 *)(_map->getVirtualAddress() + offset) = value;
    }
}

void PT1Device::memoryWrite16(UInt16 offset, UInt16 value)
{
    if (_map)
    {
        *(UInt16 *)(_map->getVirtualAddress() + offset) = value;
    }
}

void PT1Device::memoryWrite32(UInt16 offset, UInt32 value)
{
    if (_map)
    {
        *(UInt32 *)(_map->getVirtualAddress() + offset) = value;
    }
}

static bool ConvertData(LnbPower lnbPower, TunerPowerReset tunerPowerReset, uint32_t *data)
{
    if (data == NULL)
    {
        return false;
    }

    uint data_ = 0;

    switch (lnbPower)
    {
    case LNB_POWER_OFF:
        data_ |= (0 << 1 | 0 << 2);
        break;

    case LNB_POWER_15V:
        data_ |= (1 << 1 | 1 << 2);
        break;

    case LNB_POWER_11V:
        data_ |= (0 << 1 | 1 << 2);
        break;

    default:
        return false;
    }

    switch (tunerPowerReset)
    {
    case TUNER_POWER_OFF:
        data_ |= (0 << 0 | 0 << 3);
        break;

    case TUNER_POWER_ON_RESET_ENABLE:
        data_ |= (1 << 0 | 0 << 3);
        break;

    case TUNER_POWER_ON_RESET_DISABLE:
        data_ |= (1 << 0 | 1 << 3);
        break;

    default:
        return false;
    }

    *data = data_;

    return true;
}

#pragma mark
#pragma mark ############### User Interface ###############

IOReturn PT1Device::getDeviceInfo(DeviceInfo *deviceInfo, uint32_t *structSize)
{
	IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

    if (deviceInfo)
    {
        deviceInfo->Bus = _device->getBusNumber();
        deviceInfo->Slot = _device->getDeviceNumber();
        deviceInfo->Function = _device->getFunctionNumber();
        deviceInfo->BadBitCount = 0;
    }

    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::getPciClockCounter(uint32_t *status, uint32_t *counter)
{
	IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

    if ((status == NULL) || (counter == NULL))
    {
        return kIOReturnError;
    }

//    *counter = memoryRead32(0x04);
    *status = kStatusNotImplemented;

    return kIOReturnSuccess;
}

IOReturn PT1Device::setPciLatencyTimer(uint32_t *status, uint32_t latencyTimer)
{
	IOLog("%s[%p]::%s(latencyTimer = 0x%08x)\n", getName(), this, __FUNCTION__, latencyTimer);

    if (status == NULL)
    {
        return kIOReturnError;
    }

    _device->configWrite8(kIOPCIConfigLatencyTimer, (UInt8)latencyTimer);
    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

IOReturn PT1Device::getPciLatencyTimer(uint32_t *status, uint32_t *latencyTimer)
{
	IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

    if ((status == NULL) || (latencyTimer == NULL))
    {
        return kIOReturnError;
    }

    *latencyTimer = (uint32_t)_device->configRead8(kIOPCIConfigLatencyTimer);
    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

IOReturn PT1Device::setLnbPower(uint32_t *status, uint32_t lnbPower)
{
	IOLog("%s[%p]::%s(lnbPower = 0x%08x)\n", getName(), this, __FUNCTION__, lnbPower);

    if (status == NULL)
    {
        return kIOReturnError;
    }

    uint32_t data;
    if (ConvertData((LnbPower)lnbPower, _tunerPowerReset, &data))
    {
        memoryWrite32(0x04, data);
        _lnbPower = (LnbPower)lnbPower;
        *status = kStatusSuccess;
    }
    else
    {
        IOLog("%s[%p]::%s() Convert Error.\n", getName(), this, __FUNCTION__);
        *status = kStatusIOError;
    }

    return kIOReturnSuccess;
}

IOReturn PT1Device::getLnbPower(uint32_t *status, uint32_t *lnbPower)
{
	IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);
    
    if ((status == NULL) || (lnbPower == NULL))
    {
        return kIOReturnError;
    }

    *lnbPower = (uint32_t)_lnbPower;
    *status = kStatusSuccess;
    
    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::setLnbPowerWhenClose(uint32_t *status, uint32_t lnbPower)
{
	IOLog("%s[%p]::%s(lnbPower = 0x%08x)\n", getName(), this, __FUNCTION__, lnbPower);

    *status = kStatusNotImplemented;

    return kIOReturnSuccess;
}

IOReturn PT1Device::getLnbPowerWhenClose(uint32_t *status, uint32_t *lnbPower)
{
	IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

    if ((status == NULL) || (lnbPower))
    {
        return kIOReturnError;
    }

    *lnbPower = (uint32_t)_lnbPowerWhenClose;
    *status = kStatusSuccess; 

    return kIOReturnSuccess;
}

IOReturn PT1Device::setTunerPowerReset(uint32_t *status, uint32_t tunerPowerReset)
{
	IOLog("%s[%p]::%s(tunerPowerReset = 0x%08x)\n", getName(), this, __FUNCTION__, tunerPowerReset);

    if (status == NULL)
    {
        return kIOReturnError;
    }

    uint32_t data;
    if (ConvertData(_lnbPower, (TunerPowerReset)tunerPowerReset, &data))
    {
        memoryWrite32(0x04, data);
        _tunerPowerReset = (TunerPowerReset)tunerPowerReset;
        *status = kStatusSuccess;
    }
    else
    {
        IOLog("%s[%p]::%s() Convert Error.\n", getName(), this, __FUNCTION__);
        *status = kStatusIOError;
    }

    return kIOReturnSuccess;
}

IOReturn PT1Device::getTunerPowerReset(uint32_t *status, uint32_t *tunerPowerReset)
{
	IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

    if ((status == NULL) || (tunerPowerReset == NULL))
    {
        return kIOReturnError;
    }

    *tunerPowerReset = (uint32_t)_tunerPowerReset;
    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

IOReturn PT1Device::initTuner(uint32_t *status, uint32_t tuner)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x)\n", getName(), this, __FUNCTION__, tuner);

    if (status == NULL)
    {
        return kIOReturnError;
    }

    if (tuner_init(_reg, NULL, tuner) < 0)
    {
        *status = kStatusInternalError;
    }
    else
    {
        *status = kStatusSuccess;
    }

    IOLog("%s[%p]::%s() %s\n", getName(), this, __FUNCTION__, ((*status == kStatusSuccess) ? "ok." : "ng."));

    return kIOReturnSuccess;
}

IOReturn PT1Device::setTunerSleep(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t sleep)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x, sleep = 0x%08x)\n",
            getName(), this, __FUNCTION__, tuner, isdb, sleep);

    if (status == NULL)
    {
        return kIOReturnError;
    }

    if ((tuner > 1) || (isdb >= ISDB_COUNT))
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }

    int ch_type = ((isdb == ISDB_S) ? CHANNEL_TYPE_ISDB_S : CHANNEL_TYPE_ISDB_T);
    int type = ((sleep) ? TYPE_WAKEUP : TYPE_SLEEP);

    set_sleepmode(_reg, NULL, i2c_address[tuner * 2 + isdb], ch_type, type);
    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::getTunerSleep(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *sleep)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x)\n", getName(), this, __FUNCTION__, tuner, isdb);

    *status = kStatusNotImplemented;
    *sleep = 0x12345678;

    return kIOReturnSuccess;
}

uint32_t PT1Device::setFrequency_ISDB_S(uint32_t tuner, ISDB isdb, uint32_t channel, int32_t offset)
{
    uint32_t retval;
    WBLOCK wk;
    bool tmcclock = FALSE;

    // ISDB-S PLLロック
    for (int i = 0; i < MAX_BS_CHANNEL_PLL_COMMAND; ++i)
    {
        memcpy(&wk, bs_pll[channel].wblock[i], sizeof(WBLOCK));
        wk.addr = i2c_address[tuner * 2 + isdb];
        i2c_write(_reg, NULL, &wk);
    }
    // PLLロック確認
    // チェック用
    for (int i = 0; i < 200; ++i)
    {
        __u32 val;
        memcpy(&wk, &bs_pll_lock, sizeof(WBLOCK));
        wk.addr = i2c_address[tuner * 2 + isdb];
        val = i2c_read(_reg, NULL, &wk, 1);
        if (((val & 0xFF) != 0) && ((val & 0XFF) != 0XFF))
        {
            tmcclock = TRUE;
            break;
        }
    }

    if (tmcclock)
    {
        retval = kStatusSuccess;

        TmccS tmcc;
        uint32_t status_;
        getTmccS(&status_, tuner, &tmcc);
        if (status_ == kStatusSuccess)
        {
            setIdS(&status_, tuner, tmcc.Id[offset]);
        }
    }
    else
    {
        IOLog("PLL LOCK ERROR\n");
        retval = kStatusInternalError;
    }

    return retval;
}

uint32_t PT1Device::setFrequency_ISDB_T(uint32_t tuner, ISDB isdb, uint32_t channel, int32_t offset)
{
    uint32_t retval;
    WBLOCK wk;
    bool tmcclock = FALSE;

    union {
        __u8  charfreq[2];
        __u16 freq;
    } freq[2];

    freq[0].freq = getfrequency(channel, offset);
    freq[1].freq = getfrequency_add(channel);
    //指定周波数
    memcpy(&wk, &isdb_t_pll_base, sizeof(WBLOCK));
    wk.addr = i2c_address[tuner * 2 + isdb];
    // 計算した周波数を設定
    wk.value[wk.count] = freq[0].charfreq[1];
    wk.count += 1 ;
    wk.value[wk.count] = freq[0].charfreq[0];
    wk.count += 1 ;

    // 計算した周波数付加情報を設定
    wk.value[wk.count] = freq[1].charfreq[1];
    wk.count += 1 ;
    wk.value[wk.count] = freq[1].charfreq[0];
    wk.count += 1 ;

    i2c_write(_reg, NULL, &wk);

    __u32 val;
    for (int i = 0; i < 100; ++i)
    {
        memcpy(&wk, &isdb_t_pll_lock, sizeof(WBLOCK));
        wk.addr = i2c_address[tuner * 2 + isdb];
        val = i2c_read(_reg, NULL, &wk, 1);
        if (((val & 0xFF) != 0XFF) && ((val & 0X50) == 0x50))
        {
            tmcclock = TRUE;
            break;
        }
    }
    if (tmcclock)
    {
        IOLog("PT1:ISDB-T LOCK OK(%08x)\n", (unsigned int)val);
        retval = kStatusSuccess;

        TmccT tmcc;
        uint32_t status_;
        getTmccT(&status_, tuner, &tmcc);

    }
    else
    {
        IOLog("PT1:ISDB-T LOCK NG(%08x)\n", (unsigned int)val);
        retval = kStatusInternalError;
    }
    return retval;
}

#pragma mark 未実装
IOReturn PT1Device::setFrequency(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t channel, int32_t offset)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x, channel = 0x%08x, offset = 0x%08x)\n",
            getName(), this, __FUNCTION__, tuner, isdb, channel, offset);

    if (status == NULL)
    {
        return kIOReturnError;
    }

    if ((tuner > 1) || (isdb >= ISDB_COUNT) ||
        ((isdb == ISDB_S) && (channel >= MAX_BS_CHANNEL)) ||
        ((isdb == ISDB_T) && (channel >= MAX_ISDB_T_CHANNEL)))
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }

    WBLOCK wk;
    bool tmcclock = FALSE;
    switch (isdb)
    {
    case ISDB_S:
#if 0
        // ISDB-S PLLロック
        for (int i = 0; i < MAX_BS_CHANNEL_PLL_COMMAND; ++i)
        {
            memcpy(&wk, bs_pll[channel].wblock[i], sizeof(WBLOCK));
            wk.addr = i2c_address[tuner * 2 + isdb];
            i2c_write(_reg, NULL, &wk);
        }
        // PLLロック確認
        // チェック用
        for (int i = 0; i < 200; ++i)
        {
            __u32 val;
            memcpy(&wk, &bs_pll_lock, sizeof(WBLOCK));
            wk.addr = i2c_address[tuner * 2 + isdb];
            val = i2c_read(_reg, NULL, &wk, 1);
            if (((val & 0xFF) != 0) && ((val & 0XFF) != 0XFF))
            {
                tmcclock = TRUE;
                break;
            }
        }

        if (tmcclock)
        {
            *status = kStatusSuccess;

            TmccS tmcc;
            uint32_t status_;
            getTmccS(&status_, tuner, &tmcc);
            if (status_ == kStatusSuccess)
            {
                setIdS(&status_, tuner, tmcc.Id[offset]);
            }
        }
        else
        {
            IOLog("PLL LOCK ERROR\n");
            *status = kStatusInternalError;
        }
#else
        *status = setFrequency_ISDB_S(tuner, isdb, channel, offset);
#endif

        break;

    case ISDB_T:
#if 1
        union {
            __u8  charfreq[2];
            __u16 freq;
        } freq[2];

        freq[0].freq = getfrequency(channel, offset);
        freq[1].freq = getfrequency_add(channel);
        //指定周波数
        memcpy(&wk, &isdb_t_pll_base, sizeof(WBLOCK));
        wk.addr = i2c_address[tuner * 2 + isdb];
        // 計算した周波数を設定
        wk.value[wk.count] = freq[0].charfreq[1];
        wk.count += 1 ;
        wk.value[wk.count] = freq[0].charfreq[0];
        wk.count += 1 ;

        // 計算した周波数付加情報を設定
        wk.value[wk.count] = freq[1].charfreq[1];
        wk.count += 1 ;
        wk.value[wk.count] = freq[1].charfreq[0];
        wk.count += 1 ;
#if 1
    {
    WBLOCK isdb_t_chset[113] = {
    {0, 6, {0xfe, 0xc2, 0x04, 0x1b, 0x80, 0x81}},   //   0
    {0, 4, {0xfe, 0xc2, 0x04, 0x45, 0x00, 0x00}},   //   1
    {0, 4, {0xfe, 0xc2, 0x04, 0x6f, 0x00, 0x00}},   //   2
    {0, 4, {0xfe, 0xc2, 0x04, 0x99, 0x00, 0x00}},   //   3
    {0, 4, {0xfe, 0xc2, 0x04, 0xc3, 0x00, 0x00}},   //   4
    {0, 4, {0xfe, 0xc2, 0x04, 0xed, 0x00, 0x00}},   //   5
    {0, 4, {0xfe, 0xc2, 0x05, 0x17, 0x00, 0x00}},   //   6
    {0, 4, {0xfe, 0xc2, 0x05, 0x41, 0x00, 0x00}},   //   7
//    {0, 6, {0xfe, 0xc2, 0x05, 0x6b, 0x80, 0xa1}},   //   8
    {0, 4, {0xfe, 0xc2, 0x05, 0x6b, 0x00, 0x00}},   //   8
    {0, 4, {0xfe, 0xc2, 0x05, 0x95, 0x00, 0x00}},   //   9
    {0, 4, {0xfe, 0xc2, 0x05, 0xbf, 0x00, 0x00}},   //  10
    {0, 4, {0xfe, 0xc2, 0x05, 0xe9, 0x00, 0x00}},   //  11
    {0, 4, {0xfe, 0xc2, 0x06, 0x21, 0x00, 0x00}},   //  12
//    {0, 6, {0xfe, 0xc2, 0x06, 0x4b, 0x80, 0x62}},   //  13
    {0, 4, {0xfe, 0xc2, 0x06, 0x4b, 0x00, 0x00}},   //  13
    {0, 4, {0xfe, 0xc2, 0x06, 0x75, 0x00, 0x00}},   //  14
    {0, 4, {0xfe, 0xc2, 0x06, 0x9f, 0x00, 0x00}},   //  15
    {0, 4, {0xfe, 0xc2, 0x06, 0xc9, 0x00, 0x00}},   //  16
    {0, 4, {0xfe, 0xc2, 0x06, 0xe5, 0x00, 0x00}},   //  17
    {0, 4, {0xfe, 0xc2, 0x07, 0x0f, 0x00, 0x00}},   //  18
    {0, 4, {0xfe, 0xc2, 0x07, 0x39, 0x00, 0x00}},   //  19
    {0, 4, {0xfe, 0xc2, 0x07, 0x63, 0x00, 0x00}},   //  20
    {0, 4, {0xfe, 0xc2, 0x07, 0x8d, 0x00, 0x00}},   //  21
//   {0, 6, {0xfe, 0xc2, 0x07, 0xb7, 0x80, 0xa2}},   //  22
    {0, 4, {0xfe, 0xc2, 0x07, 0xb7, 0x00, 0x00}},   //  22
    {0, 4, {0xfe, 0xc2, 0x07, 0xe1, 0x00, 0x00}},   //  23
    {0, 4, {0xfe, 0xc2, 0x08, 0x0b, 0x00, 0x00}},   //  24
    {0, 4, {0xfe, 0xc2, 0x08, 0x35, 0x00, 0x00}},   //  25
    {0, 4, {0xfe, 0xc2, 0x08, 0x5f, 0x00, 0x00}},   //  26
    {0, 4, {0xfe, 0xc2, 0x08, 0x89, 0x00, 0x00}},   //  27
    {0, 4, {0xfe, 0xc2, 0x08, 0xb3, 0x00, 0x00}},   //  28
    {0, 4, {0xfe, 0xc2, 0x08, 0xdd, 0x00, 0x00}},   //  29
    {0, 4, {0xfe, 0xc2, 0x09, 0x07, 0x00, 0x00}},   //  30
    {0, 4, {0xfe, 0xc2, 0x09, 0x31, 0x00, 0x00}},   //  31
    {0, 4, {0xfe, 0xc2, 0x09, 0x5b, 0x00, 0x00}},   //  32
    {0, 4, {0xfe, 0xc2, 0x09, 0x85, 0x00, 0x00}},   //  33
    {0, 4, {0xfe, 0xc2, 0x09, 0xaf, 0x00, 0x00}},   //  34
    {0, 4, {0xfe, 0xc2, 0x09, 0xd9, 0x00, 0x00}},   //  35
    {0, 4, {0xfe, 0xc2, 0x0a, 0x03, 0x00, 0x00}},   //  36
    {0, 4, {0xfe, 0xc2, 0x0a, 0x2d, 0x00, 0x00}},   //  37
    {0, 4, {0xfe, 0xc2, 0x0a, 0x57, 0x00, 0x00}},   //  38
    {0, 4, {0xfe, 0xc2, 0x0a, 0x81, 0x00, 0x00}},   //  39
//    {0, 6, {0xfe, 0xc2, 0x0a, 0xab, 0x80, 0xe2}},   //  40
    {0, 4, {0xfe, 0xc2, 0x0a, 0xab, 0x00, 0x00}},   //  40
    {0, 4, {0xfe, 0xc2, 0x0a, 0xd5, 0x00, 0x00}},   //  41
    {0, 4, {0xfe, 0xc2, 0x0a, 0xff, 0x00, 0x00}},   //  42
    {0, 4, {0xfe, 0xc2, 0x0b, 0x29, 0x00, 0x00}},   //  43
    {0, 4, {0xfe, 0xc2, 0x0b, 0x53, 0x00, 0x00}},   //  44
    {0, 4, {0xfe, 0xc2, 0x0b, 0x7d, 0x00, 0x00}},   //  45
    {0, 4, {0xfe, 0xc2, 0x0b, 0xa7, 0x00, 0x00}},   //  46
    {0, 4, {0xfe, 0xc2, 0x0b, 0xd1, 0x00, 0x00}},   //  47
    {0, 4, {0xfe, 0xc2, 0x0b, 0xfb, 0x00, 0x00}},   //  48
    {0, 4, {0xfe, 0xc2, 0x0c, 0x25, 0x00, 0x00}},   //  49
    {0, 4, {0xfe, 0xc2, 0x0c, 0x4f, 0x00, 0x00}},   //  50
    {0, 4, {0xfe, 0xc2, 0x0c, 0x79, 0x00, 0x00}},   //  51
//    {0, 6, {0xfe, 0xc2, 0x0c, 0xa3, 0x80, 0x64}},   //  52
    {0, 4, {0xfe, 0xc2, 0x0c, 0xa3, 0x00, 0x00}},   //  52
    {0, 4, {0xfe, 0xc2, 0x0c, 0xcd, 0x00, 0x00}},   //  53
    {0, 4, {0xfe, 0xc2, 0x0c, 0xf7, 0x00, 0x00}},   //  54
    {0, 4, {0xfe, 0xc2, 0x0d, 0x21, 0x00, 0x00}},   //  55
    {0, 4, {0xfe, 0xc2, 0x0d, 0x4b, 0x00, 0x00}},   //  56
    {0, 4, {0xfe, 0xc2, 0x0d, 0x75, 0x00, 0x00}},   //  57
    {0, 4, {0xfe, 0xc2, 0x0d, 0x9f, 0x00, 0x00}},   //  58
    {0, 4, {0xfe, 0xc2, 0x0d, 0xc9, 0x00, 0x00}},   //  59
    {0, 6, {0xfe, 0xc2, 0x0d, 0xf3, 0x80, 0x84}},   //  60
    {0, 4, {0xfe, 0xc2, 0x0e, 0x1d, 0x00, 0x00}},   //  61
    {0, 4, {0xfe, 0xc2, 0x0e, 0x47, 0x00, 0x00}},   //  62
    {0, 4, {0xfe, 0xc2, 0x0e, 0x7f, 0x00, 0x00}},   //  63
    {0, 4, {0xfe, 0xc2, 0x0e, 0xa9, 0x00, 0x00}},   //  64
    {0, 4, {0xfe, 0xc2, 0x0e, 0xd3, 0x00, 0x00}},   //  65
    {0, 4, {0xfe, 0xc2, 0x0e, 0xfd, 0x00, 0x00}},   //  66
    {0, 4, {0xfe, 0xc2, 0x0f, 0x27, 0x00, 0x00}},   //  67
    {0, 4, {0xfe, 0xc2, 0x0f, 0x51, 0x00, 0x00}},   //  68
    {0, 4, {0xfe, 0xc2, 0x0f, 0x7b, 0x00, 0x00}},   //  69
    {0, 4, {0xfe, 0xc2, 0x0f, 0xa5, 0x00, 0x00}},   //  70
    {0, 4, {0xfe, 0xc2, 0x0f, 0xcf, 0x00, 0x00}},   //  71
    {0, 4, {0xfe, 0xc2, 0x0f, 0xf9, 0x00, 0x00}},   //  72
    {0, 4, {0xfe, 0xc2, 0x10, 0x23, 0x00, 0x00}},   //  73
    {0, 4, {0xfe, 0xc2, 0x10, 0x4d, 0x00, 0x00}},   //  74
    {0, 4, {0xfe, 0xc2, 0x10, 0x77, 0x00, 0x00}},   //  75
//    {0, 6, {0xfe, 0xc2, 0x10, 0xa1, 0x80, 0xa4}},   //  76
    {0, 4, {0xfe, 0xc2, 0x10, 0xa1, 0x00, 0x00}},   //  76
    {0, 4, {0xfe, 0xc2, 0x10, 0xcb, 0x00, 0x00}},   //  77
    {0, 4, {0xfe, 0xc2, 0x10, 0xf5, 0x00, 0x00}},   //  78
    {0, 4, {0xfe, 0xc2, 0x11, 0x1f, 0x00, 0x00}},   //  79
    {0, 4, {0xfe, 0xc2, 0x11, 0x49, 0x00, 0x00}},   //  80
    {0, 4, {0xfe, 0xc2, 0x11, 0x73, 0x00, 0x00}},   //  81
    {0, 4, {0xfe, 0xc2, 0x11, 0x9d, 0x00, 0x00}},   //  82
    {0, 4, {0xfe, 0xc2, 0x11, 0xc7, 0x00, 0x00}},   //  83
    {0, 4, {0xfe, 0xc2, 0x11, 0xf1, 0x00, 0x00}},   //  84
//    {0, 6, {0xfe, 0xc2, 0x12, 0x1b, 0x80, 0xc4}},   //  85
    {0, 4, {0xfe, 0xc2, 0x12, 0x1b, 0x00, 0x00}},   //  85
    {0, 4, {0xfe, 0xc2, 0x12, 0x45, 0x00, 0x00}},   //  86
    {0, 4, {0xfe, 0xc2, 0x12, 0x6f, 0x00, 0x00}},   //  87
    {0, 4, {0xfe, 0xc2, 0x12, 0x99, 0x00, 0x00}},   //  88
    {0, 4, {0xfe, 0xc2, 0x12, 0xc3, 0x00, 0x00}},   //  89
    {0, 4, {0xfe, 0xc2, 0x12, 0xed, 0x00, 0x00}},   //  90
    {0, 4, {0xfe, 0xc2, 0x13, 0x17, 0x00, 0x00}},   //  91
    {0, 4, {0xfe, 0xc2, 0x13, 0x41, 0x00, 0x00}},   //  92
    {0, 4, {0xfe, 0xc2, 0x13, 0x6b, 0x00, 0x00}},   //  93
    {0, 4, {0xfe, 0xc2, 0x13, 0x95, 0x00, 0x00}},   //  94
    {0, 4, {0xfe, 0xc2, 0x13, 0xbf, 0x00, 0x00}},   //  95
    {0, 4, {0xfe, 0xc2, 0x13, 0xe9, 0x00, 0x00}},   //  96
    {0, 4, {0xfe, 0xc2, 0x14, 0x13, 0x00, 0x00}},   //  97
    {0, 4, {0xfe, 0xc2, 0x14, 0x3d, 0x00, 0x00}},   //  98
    {0, 4, {0xfe, 0xc2, 0x14, 0x67, 0x00, 0x00}},   //  99
    {0, 4, {0xfe, 0xc2, 0x14, 0x91, 0x00, 0x00}},   // 100
//    {0, 6, {0xfe, 0xc2, 0x14, 0xbb, 0x80, 0xe4}},   // 101
    {0, 4, {0xfe, 0xc2, 0x14, 0xbb, 0x00, 0x00}},   // 101
    {0, 4, {0xfe, 0xc2, 0x14, 0xe5, 0x00, 0x00}},   // 102
    {0, 4, {0xfe, 0xc2, 0x15, 0x0f, 0x00, 0x00}},   // 103
    {0, 4, {0xfe, 0xc2, 0x15, 0x39, 0x00, 0x00}},   // 104
    {0, 4, {0xfe, 0xc2, 0x15, 0x63, 0x00, 0x00}},   // 105
    {0, 4, {0xfe, 0xc2, 0x15, 0x8d, 0x00, 0x00}},   // 106
    {0, 4, {0xfe, 0xc2, 0x15, 0xb7, 0x00, 0x00}},   // 107
    {0, 4, {0xfe, 0xc2, 0x15, 0xe1, 0x00, 0x00}},   // 108
    {0, 4, {0xfe, 0xc2, 0x16, 0x0b, 0x00, 0x00}},   // 109
    {0, 4, {0xfe, 0xc2, 0x16, 0x35, 0x00, 0x00}},   // 110
    {0, 4, {0xfe, 0xc2, 0x16, 0x5f, 0x00, 0x00}},   // 111
    {0, 4, {0xfe, 0xc2, 0x16, 0x89, 0x00, 0x00}}};  // 112

IOLog("%d, %02X %02X %02X %02X\n", wk.count, wk.value[2], wk.value[3], wk.value[4], wk.value[5]);
        memcpy(&wk, &isdb_t_chset[channel], sizeof(WBLOCK));

        // 計算した周波数付加情報を設定
        wk.value[wk.count] = freq[1].charfreq[1];
        wk.count += 1 ;
        wk.value[wk.count] = freq[1].charfreq[0];
        wk.count += 1 ;


IOLog("%d, %02X %02X %02X %02X\n", wk.count, wk.value[2], wk.value[3], wk.value[4], wk.value[5]);
        wk.addr = i2c_address[tuner * 2 + isdb];

    }
#endif
        i2c_write(_reg, NULL, &wk);

        __u32 val;
        for (int i = 0; i < 100; ++i)
        {
            memcpy(&wk, &isdb_t_pll_lock, sizeof(WBLOCK));
            wk.addr = i2c_address[tuner * 2 + isdb];
            val = i2c_read(_reg, NULL, &wk, 1);
            if (((val & 0xFF) != 0XFF) && ((val & 0X50) == 0x50))
            {
                tmcclock = TRUE;
                break;
            }
        }
        if (tmcclock)
        {
            *status = kStatusSuccess;
        }
        else
        {
            IOLog("PT1:ISDB-T LOCK NG(%08x)\n", (unsigned int)val);
            *status = kStatusInternalError;
        }
#else
//        *status = setFrequency_ISDB_T(tuner, isdb, channel, offset);
        if (isdb_t_frequency(_reg, NULL, i2c_address[tuner * 2 + ISDB_T], channel, offset) < 0)
        {
            *status = kStatusInternalError;
        }
#endif
        break;

    default:
        *status = kStatusParamError;
        break;
	}

    IOLog("%s[%p]::%s() %s\n", getName(), this, __FUNCTION__, ((*status == kStatusSuccess) ? "ok." : "ng."));

    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::getFrequency(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *channel, int32_t *offset)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x)\n", getName(), this, __FUNCTION__, tuner, isdb);

    *status = kStatusNotImplemented;
    *channel = 0x12345678;
    *offset  = 0x23456789;

    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::getFrequencyOffset(uint32_t *status, uint32_t tuner, ISDB isdb, int32_t *clock, int32_t *carrier)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x)\n", getName(), this, __FUNCTION__, tuner, isdb);

    if ((status == NULL) || (clock == NULL) || (carrier == NULL))
    {
        return kIOReturnError;
    }
    if ((tuner > 1) || (isdb >= ISDB_COUNT))
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }

    WBLOCK wk;
    switch (isdb)
    {
    case ISDB_S:
        memcpy(&wk, &bs_get_clock, sizeof(WBLOCK));
        wk.addr = i2c_address[tuner * 2 + isdb];
        *clock = i2c_read(_reg, NULL, &wk, 1) & 0xff;

        memcpy(&wk, &bs_get_carrir, sizeof(WBLOCK));
        wk.addr = i2c_address[tuner * 2 + isdb];
        *carrier = i2c_read(_reg, NULL, &wk, 1) & 0xff;
        *carrier *= 80000;
        *status = kStatusSuccess;
        break;

    case ISDB_T:
        memcpy(&wk, &isdb_t_get_clock, sizeof(WBLOCK));
        wk.addr = i2c_address[tuner * 2 + isdb];
        *clock = i2c_read(_reg, NULL, &wk, 1) & 0xff;

        memcpy(&wk, &bs_get_carrir, sizeof(WBLOCK));
        wk.addr = i2c_address[tuner * 2 + isdb];
        *carrier = i2c_read(_reg, NULL, &wk, 1) & 0xff;
        *status = kStatusSuccess;
        break;

    default:
        *status = kStatusParamError;
        break;
    }

	IOLog("%s[%p]::%s() clock = %d, carrier = %d\n", getName(), this, __FUNCTION__, *clock, *carrier);

    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::getCnAgc(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *cn100, uint32_t *currentAgc, uint32_t *maxAgc)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x)\n", getName(), this, __FUNCTION__, tuner, isdb);

    if ((status == NULL) || (cn100 == NULL) || (currentAgc == NULL) || (maxAgc == NULL))
    {
        return kIOReturnError;
    }
    if ((tuner > 1) || (isdb >= ISDB_COUNT))
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }

    WBLOCK wk;
    switch (isdb)
    {
    case ISDB_S:
        *cn100 = isdb_s_read_signal_strength(_reg, NULL, i2c_address[tuner * 2 + isdb]);
        memcpy(&wk, &bs_get_agc, sizeof(WBLOCK));
        wk.addr = i2c_address[tuner * 2 + isdb];
        *currentAgc = i2c_read(_reg, NULL, &wk, 1) & 0xff;
        *maxAgc = 127;
        *status = kStatusSuccess;
        break;

    case ISDB_T:
        *cn100 = isdb_t_read_signal_strength(_reg, NULL, i2c_address[tuner * 2 + isdb]);
        memcpy(&wk, &isdb_t_agc2, sizeof(WBLOCK));
        wk.addr = i2c_address[tuner * 2 + isdb];
        *currentAgc = i2c_read(_reg, NULL, &wk, 1) & 0xff;
        *maxAgc = 255;
        *status = kStatusSuccess;
        break;

    default:
        *status = kStatusParamError;
        break;
    }

	IOLog("%s[%p]::%s() %s.\n", getName(), this, __FUNCTION__, (*status == kStatusSuccess) ? "ok" : "ng");

    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::setIdS(uint32_t *status, uint32_t tuner, uint32_t tsid)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, tsid = 0x%08x)\n", getName(), this, __FUNCTION__, tuner, tsid);

    if (status == NULL)
    {
        return kIOReturnError;
    }
    if (tuner > 1)
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }

    WBLOCK wk;
    union {
        __u8  ts[2];
        __u16 tsid;
    } uts_id;

    uts_id.tsid = tsid;
    memcpy(&wk, &bs_set_ts_lock, sizeof(WBLOCK));
    wk.addr = i2c_address[tuner * 2 + ISDB_S];
	// TS-ID設定
    wk.value[1] = uts_id.ts[1];
    wk.value[2] = uts_id.ts[0];
    i2c_write(_reg, NULL, &wk);

    *status = kStatusInternalError;
    for (int i = 0; i < 100; ++i)
    {
        memcpy(&wk, &bs_get_ts_lock, sizeof(WBLOCK));
        wk.addr = i2c_address[tuner * 2 + ISDB_S];
        __u32 val = i2c_read(_reg, NULL, &wk, 2);
        if ((val & 0xFFFF) == tsid)
        {
            *status = kStatusSuccess;
            break;
		}
	}

	IOLog("%s[%p]::%s() %s.\n", getName(), this, __FUNCTION__, (*status == kStatusSuccess) ? "ok" : "ng");

    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::getIdS(uint32_t *status, uint32_t tuner, uint32_t *tsid)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x)\n", getName(), this, __FUNCTION__, tuner);

    if ((status == NULL) || (tsid == NULL))
    {
        return kIOReturnError;
    }
    if (tuner > 1)
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }

    WBLOCK wk;
    memcpy(&wk, &bs_get_ts_lock, sizeof(WBLOCK));
    wk.addr = i2c_address[tuner * 2 + ISDB_S];
    *tsid = i2c_read(_reg, NULL, &wk, 2) & 0xffff;

    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::getCorrectedErrorRate(uint32_t *status, uint32_t tuner, ISDB isdb, LayerIndex layerIndex, ErrorRate *errorRate)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x, layerIndex = 0x%08x)\n",
            getName(), this, __FUNCTION__, tuner, isdb, layerIndex);

    *status = kStatusNotImplemented;
    errorRate->Numerator = 0x12345678;
    errorRate->Denominator = 0x23456789;

    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::resetCorrectedErrorCount(uint32_t *status, uint32_t tuner, ISDB isdb)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x)\n", getName(), this, __FUNCTION__, tuner, isdb);

    *status = kStatusNotImplemented;

    return kIOReturnSuccess;
}

IOReturn PT1Device::getErrorCount(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *count)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x)\n", getName(), this, __FUNCTION__, tuner, isdb);

    if ((status == NULL) || (count == NULL))
    {
        return kIOReturnError;
    }
    if ((tuner > 1) || (isdb >= ISDB_COUNT))
    {
        return kIOReturnError;
    }

    uint32_t gray = memoryRead32((0x04 + (tuner * 2 + isdb)) * 4);    

    uint32_t binary = 0;

    for (uint32_t i = 0; i < 32; ++i)
    {
        uint32_t k = 0;
        for (uint32_t j = i; j < 32; ++j)
        {
            k = k ^ BIT_SHIFT_MASK(gray, j, 1);
        }
        binary |= (k << i);
    }

    *count = binary;
    
    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::getTmccS(uint32_t *status, uint32_t tuner, TmccS *tmcc)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x)\n", getName(), this, __FUNCTION__, tuner);

    if ((status == NULL) || (tmcc == NULL))
    {
        return kIOReturnError;
    }
    if (tuner > 1)
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }

    WBLOCK wk;
    union {
        __u16 ts[2];
        __u32 tsid;
    } ts_id;

    memcpy(&wk, &bs_tmcc_get_1, sizeof(WBLOCK));
    wk.addr = i2c_address[tuner * 2 + ISDB_S];
    i2c_write(_reg, NULL, &wk);

    bool tmcclock = FALSE ;

    for(int i = 0; i < 200; ++i)
    {
        __u32 val;
        memcpy(&wk, &bs_tmcc_get_2, sizeof(WBLOCK));
        wk.addr = i2c_address[tuner * 2 + ISDB_S];
		val = i2c_read(_reg, NULL, &wk, 1);
        if (((val & 0XFF) != 0XFF) && (!(val & 0x10)))
        {
            tmcclock = TRUE;
            break;
        }
    }

    if (tmcclock)
    {
        // 該当周波数のTS-IDを取得
        for (int i = 0; i < (MAX_BS_TS_ID / 2); ++i)
        {
            for (int j = 0; j < 100; ++j)
            {
                memcpy(&wk, bs_get_ts_id[i], sizeof(WBLOCK));
                wk.addr = i2c_address[tuner * 2 + ISDB_S];
                ts_id.tsid = i2c_read(_reg, NULL, &wk, 4);
                // TS-IDが0の場合は再取得する
                if ((ts_id.ts[0] != 0) && (ts_id.ts[1] != 0))
                {
                    break;
                }
            }
            tmcc->Id[i * 2    ] = ts_id.ts[1];
            tmcc->Id[i * 2 + 1] = ts_id.ts[0] ;
        }
    }
    else
    {
        IOLog("TMCC LOCK ERROR\n");
        *status = kStatusInternalError;
	}

    IOLog("%s[%p]::%s() %s\n", getName(), this, __FUNCTION__, ((*status == kStatusSuccess) ? "ok." : "ng."));

    return kIOReturnSuccess;
}

IOReturn PT1Device::getLayerS(uint32_t *status, uint32_t tuner, LayerS *layer)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x)\n", getName(), this, __FUNCTION__, tuner);

    if ((status == NULL) || (layer == NULL))
    {
        return kIOReturnError;
    }
    if (tuner > 1)
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }

    union {
        __u8  slot[4];
        __u32 u32slot;
    } ts_slot;
    WBLOCK wk;

    //スロット取得
    memcpy(&wk, &bs_get_slot, sizeof(WBLOCK));
    wk.addr = i2c_address[tuner * 2 + ISDB_S];
    ts_slot.u32slot = i2c_read(_reg, NULL, &wk, 3);
    layer->Count[0] = ts_slot.slot[0];
    layer->Count[1] = ts_slot.slot[1];
    layer->Mode[0] = ts_slot.slot[2];
    layer->Mode[1] = 0;

    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

IOReturn PT1Device::getTmccT(uint32_t *status, uint32_t tuner, TmccT *tmcc)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x)\n", getName(), this, __FUNCTION__, tuner);

    if ((status == NULL) || (tmcc == NULL))
    {
        return kIOReturnError;
    }
    if (tuner > 1)
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }

    WBLOCK wk;
    memcpy(&wk, &isdb_t_check_tune, sizeof(WBLOCK));
    wk.addr = i2c_address[tuner * 2 + ISDB_T];
    i2c_write(_reg, NULL, &wk);

    __u32 val;
    bool tmcclock = FALSE;
    for (int i = 0; i < 2000; ++i)
    {
        memcpy(&wk, &isdb_t_tune_read, sizeof(WBLOCK));
        wk.addr = i2c_address[tuner * 2 + ISDB_T];
        val = i2c_read(_reg, NULL, &wk, 1);
        if (((val & 0xFF) != 0xFF) && ((val & 0x8) != 8))
        {
            tmcclock = TRUE;
            break;
        }
    }

    if (tmcclock)
    {
        tmcc->System = 0;
        for (int i = 0; i < 3; ++i)
        {
            tmcc->Mode[i] = 0;
            tmcc->Rate[i] = 0;
            tmcc->Interleave[i] = 0;
            tmcc->Segment[i] = 0;
        }
    }
    else
    {
        IOLog("TMCC LOCK ERROR. val = 0x%08x\n", val);
        *status = kStatusInternalError;
	}

    IOLog("%s[%p]::%s() %s\n", getName(), this, __FUNCTION__, ((*status == kStatusSuccess) ? "ok." : "ng."));

    return kIOReturnSuccess;
}

IOReturn PT1Device::getLockedT(uint32_t *status, uint32_t tuner, bool locked[LAYER_COUNT_T])
{
	IOLog("%s[%p]::%s(tuner = 0x%08x)\n", getName(), this, __FUNCTION__, tuner);

    *status = kStatusSuccess;

    for (int i = 0; i < LAYER_COUNT_T; ++i)
    {
        locked[i] = (bool)(i % 2);
    }

    return kIOReturnSuccess;
}

IOReturn PT1Device::setLayerEnable(uint32_t *status, uint32_t tuner, ISDB isdb, LayerMask layerMask)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x, layerMask = 0x%08x)\n",
            getName(), this, __FUNCTION__, tuner, isdb, layerMask);

    *status = kStatusNotImplemented;

    return kIOReturnSuccess;
}

IOReturn PT1Device::getLayerEnable(uint32_t *status, uint32_t tuner, ISDB isdb, LayerMask *layerMask)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x)\n", getName(), this, __FUNCTION__, tuner, isdb);

    *status = kStatusNotImplemented;
    *layerMask = (LayerMask)0x12345678;

    return kIOReturnSuccess;
}

IOReturn PT1Device::setBufferInfo(uint32_t *status, const BufferInfo *bufferInfo)
{
	IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

    // パラメータチェック
    if ((status == NULL) || (bufferInfo == NULL))
    {
        return kIOReturnError;
    }
    if ((bufferInfo->LockSize == 0) || (bufferInfo->VirtualSize == 0) || (bufferInfo->VirtualCount == 0))
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }
    if ((bufferInfo->VirtualSize % bufferInfo->LockSize) != 0)
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }
    if (_bufferInfo.VirtualSize != 0)
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }

    *status = kStatusInternalError;

    do {
        // アドレス用DMAバッファ
        mach_vm_size_t bufferSize = kPageSize * bufferInfo->VirtualSize * bufferInfo->VirtualCount;
//        IOOptionBits options = kIOMemoryKernelUserShared | kIOMemoryPhysicallyContiguous;
        IOOptionBits options = kIOMemoryKernelUserShared | kIOMemoryPhysicallyContiguous | kIODirectionOut;
        _bmdAddress = IOBufferMemoryDescriptor::inTaskWithPhysicalMask(kernel_task, options, bufferSize, 0x00000000FFFFF000ULL);
        if (_bmdAddress == NULL)
        {
            IOLog("_bmdAddress ng.\n");
            break;
        }

        _dmaCmdAddress = IODMACommand::withSpecification(kIODMACommandOutputHost32, 32, 0, IODMACommand::kMapped, 0, 1);
        if (_dmaCmdAddress == NULL)
        {
            IOLog("_dmaCmdAddress ng.\n");
            break;
        }

        if (_dmaCmdAddress->setMemoryDescriptor(_bmdAddress) != kIOReturnSuccess)
        {
            IOLog("_dmaCmdAddress->setMemoryDescriptor failed.\n");
            break;
        }

        IOReturn err = kIOReturnSuccess;
        UInt64 offset = 0;
        while ((kIOReturnSuccess == err) && (offset < _bmdAddress->getLength()))
        {
            // use the 32 bit variant to match outSegFunc
            IODMACommand::Segment32 segments[1];
            UInt32 numSeg = 1;

            // use the 32 bit variant to match outSegFunc
            err = _dmaCmdAddress->gen32IOVMSegments(&offset, &segments[0], &numSeg);
            IOLog("addr: gen32IOVMSegments(%x) addr 0x%lx, len 0x%lx, nsegs %ld\n",
                  err, segments[0].fIOVMAddr, segments[0].fLength, numSeg);
        }

        // データ用DMAバッファ
        bufferSize = kPageSize * kBufferPageCount * bufferInfo->VirtualSize * bufferInfo->VirtualCount;
//        options = kIOMemoryKernelUserShared | kIOMemoryPhysicallyContiguous;
        options = kIOMemoryKernelUserShared | kIOMemoryPhysicallyContiguous | kIODirectionIn;
        _bmdData = IOBufferMemoryDescriptor::inTaskWithPhysicalMask(kernel_task, options, bufferSize, 0x00000000FFFFF000ULL);
        if (_bmdData == NULL)
        {
            IOLog("_bmdData ng.\n");
            break;
        }
        _dmaCmdData = IODMACommand::withSpecification(kIODMACommandOutputHost32, 32, 0, IODMACommand::kMapped, 0, 1);
        if (_dmaCmdData == NULL)
        {
            IOLog("_dmaCmdData ng.\n");
            break;
        }
        if (_dmaCmdData->setMemoryDescriptor(_bmdData) != kIOReturnSuccess)
        {
            IOLog("_dmaCmdData->setMemoryDescriptor failed.\n");
            break;
        }

        err = kIOReturnSuccess;
        offset = 0;
        while ((kIOReturnSuccess == err) && (offset < _bmdData->getLength()))
        {
            // use the 32 bit variant to match outSegFunc
            IODMACommand::Segment32 segments[1];
            UInt32 numSeg = 1;

            // use the 32 bit variant to match outSegFunc
            err = _dmaCmdData->gen32IOVMSegments(&offset, &segments[0], &numSeg);
            IOLog("data: gen32IOVMSegments(%x) addr 0x%lx, len 0x%lx, nsegs %ld\n",
                  err, segments[0].fIOVMAddr, segments[0].fLength, numSeg);
        }

        //
        IOMemoryMap *map = _bmdAddress->map();
        for (uint i = 0; i < bufferInfo->VirtualSize * bufferInfo->VirtualCount; ++i)
        {
            uint next = (i + 1) % (bufferInfo->VirtualSize * bufferInfo->VirtualCount);
            IOPhysicalAddress addr = _bmdAddress->getPhysicalSegment(next * kPageSize, NULL);
            uint32_t *ptr = (uint32_t *)(map->getVirtualAddress() + kPageSize * i);
            *(ptr++) = (addr >> 12);

            for (uint j = 0; j < kBufferPageCount; ++j)
            {
                addr = _bmdData->getPhysicalSegment(i * kPageSize * kBufferPageCount + j * kPageSize, NULL);
                *(ptr++) = (addr >> 12);
            }
        }
        map->release();

        *status = kStatusSuccess;

    } while (false);

    if (*status == kStatusSuccess)
    {
        IOLog("success.\n");
        _bufferInfo = *bufferInfo;
    }
    else
    {
        IOLog("failed.\n");
        clearBuffer();
    }

    return kIOReturnSuccess;
}

IOReturn PT1Device::getBufferInfo(uint32_t *status, BufferInfo *bufferInfo)
{
//  IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

    if ((status == NULL) || (bufferInfo == NULL))
    {
        return kIOReturnError;
    }

    *bufferInfo = _bufferInfo;
    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::resetTransferCounter(uint32_t *status)
{
	IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

    if (status == NULL)
    {
        return kIOReturnError;
    }

    memoryWrite32(0x00, 0x00000010);

    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::incrementTransferCounter(uint32_t *status)
{
//    IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

    if (status == NULL)
    {
        return kIOReturnError;
    }

    memoryWrite32(0x00, 0x00000020);

    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

IOReturn PT1Device::setStreamEnable(uint32_t *status, uint32_t tuner, ISDB isdb, bool enable)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x, enable = 0x%08x)\n",
            getName(), this, __FUNCTION__, tuner, isdb, enable);

    if (status == NULL)
    {
        return kIOReturnError;
    }
    if ((tuner > 1) || (isdb >= ISDB_COUNT))
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }

    SetStream(_reg, (tuner * 2  + isdb), enable);
	_streamEnable[tuner * 2  + isdb] = enable;

    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

IOReturn PT1Device::getStreamEnable(uint32_t *status, uint32_t tuner, ISDB isdb, bool *enable)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x)\n", getName(), this, __FUNCTION__, tuner, isdb);

    if ((status == NULL) || (enable == NULL))
    {
        return kIOReturnError;
    }
    if ((tuner > 1) || (isdb >= ISDB_COUNT))
    {
        return kIOReturnError;
    }

    *enable = _streamEnable[tuner * 2 + isdb];
    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

IOReturn PT1Device::setStreamGray(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t gray)
{
//  IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x, gray = 0x%08x)\n", getName(), this, __FUNCTION__, tuner, isdb, gray);

    if (status == NULL)
    {
        return kIOReturnError;
    }
    if ((tuner > 1) || (isdb >= ISDB_COUNT) ||((1<<(3)) <= gray))
    {
        *status = kStatusParamError;
        return kIOReturnSuccess;
    }

    uint32_t index = tuner * 2 + isdb;
    uint32_t data = (BIT_1(1) << 3 | gray) << (4 * index);

    _streamGray[index] = gray;

    memoryWrite32(0x0c, data);

    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

IOReturn PT1Device::getStreamGray(uint32_t *status, uint32_t tuner, ISDB isdb, uint32_t *gray)
{
	IOLog("%s[%p]::%s(tuner = 0x%08x, isdb = 0x%08x)\n", getName(), this, __FUNCTION__, tuner, isdb);

    if ((status == NULL) || (gray == NULL))
    {
        return kIOReturnError;
    }
    if ((tuner > 1) || (isdb >= ISDB_COUNT))
    {
        return kIOReturnError;
    }

    *gray = _streamGray[tuner * 2 + isdb];
    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

IOReturn PT1Device::setTransferEnable(uint32_t *status, bool enable)
{
	IOLog("%s[%p]::%s(enable = 0x%08x)\n", getName(), this, __FUNCTION__, enable);

    if (status == NULL)
    {
        return kIOReturnError;
    }

    _transfer = enable;
    if (_transfer)
    {
        // 開始
        IOPhysicalAddress addr = _bmdAddress->getPhysicalSegment(0, NULL);
        memoryWrite32(0x14, (addr >> 12));
        memoryWrite32(0x00, 0x0c000040);
    }
    else
    {
        // 停止
        memoryWrite32(0x00, 0x08080000);
        for (uint i = 0; i < 10; ++i)
        {
            UInt32 val = memoryRead32(0x00);
            if (!(val & (1 << 6)))
            {
                break;
            }
            IOSleep(1);
        }
    }

    *status = kStatusSuccess;

    return kIOReturnSuccess;
}

IOReturn PT1Device::getTransferEnable(uint32_t *status, bool *enable)
{
	IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

    if ((status == NULL) || (enable == NULL))
    {
        return kIOReturnError;
    }

    *status = kStatusSuccess;
    *enable = _transfer;

    return kIOReturnSuccess;
}

#pragma mark 未実装
IOReturn PT1Device::getTransferInfo(uint32_t *status, TransferInfo *transferInfo)
{
	IOLog("%s[%p]::%s()\n", getName(), this, __FUNCTION__);

    *status = kStatusNotImplemented;
    if ((status == NULL) || (transferInfo == NULL))
    {
        return kIOReturnError;
    }

    uint32_t data = memoryRead32(0x00);

    enum {
        BIT_RAM_OVERFLOW = 3,
        BIT_INITIATOR_ERROR,
        BIT_INITIATOR_WARNING,
        BIT_INITIATOR_BUSY
    };

    transferInfo->BufferOverflow   = BIT_SHIFT_MASK(data, BIT_RAM_OVERFLOW     , 1);
    transferInfo->TransferCounter0 = BIT_SHIFT_MASK(data, BIT_INITIATOR_ERROR  , 1);
    transferInfo->TransferCounter1 = BIT_SHIFT_MASK(data, BIT_INITIATOR_WARNING, 1);
    //            InitiatorBusy    = BIT_SHIFT_MASK(data, BIT_INITIATOR_BUSY   , 1);

    *status = kStatusSuccess;

    return kIOReturnSuccess;
}



