////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// PB_LAS.v
// Copyright(C) 2009 OHSAWA Naotaka. All rights reserved.
// 
// $Rev$
// $Date$
// $Author$

`include "defs.vh"

////////////////////////////////////////////////////////////////////////////////
// PB_LAS
// Reduction core module using Left Ancestor Stack Algorithm
module PB_LAS
(
	input				CLK,
	input				RSTn,

	// Interface for init and start
	input				INIT,
	input				RUN,
	input				LAZYK,
	input				CONT,
	input				SET_CDR,
	input  [`PB_DS_BITS-1:0]	SET_CDR_D,

	output				BUSY,
	output [`PB_DS_BITS-1:0]	CAR,
	output [`PB_DS_BITS-1:0]	CDR,

	// Evaluation Stack Interface
	output				EREQ,
	output [`PB_ES_LBITS-1:0]	ELEN,
	output [`PB_DS_BITS-1:0]	EWD,
	output [`PB_DS_BITS-1:0]	EWD_1,

	input  [`PB_ES_ABITS-1:0]	ENUM,
	input  [`PB_DS_BITS-1:0]	ETOP,
	input  [`PB_DS_BITS-1:0]	ETOP_1,
	input  [`PB_DS_BITS-1:0]	ETOP_2,

        input				EVF,
	
	// External Memory Interface
	output				MAREQ,
	input				MABSY,
	output				MAWRITE,
	output [`PB_MI_ABITS-1:0]	MAA,
	output [`PB_MI_DBITS-1:0]	MAD,
	output [`PB_MI_DBITS/8-1:0]	MAS,

	input				MRREQ,
	output				MRBSY,
	input  [`PB_MI_DBITS-1:0]	MRD,
	
	// Free Pointer Interface
	output				FPREQ,
	input  [`PB_DS_BITS-1:0]	FPA,
        input				FPVF,


	// Input Stream Interface
	output				ISREQ,
	input				ISBSY,
	input				ISLAST,
	input  [`PB_NUM_BITS-1:0]	ISD,
	
	// Output Stream Interface
	output				OSREQ,
	input				OSBSY,
	output				OSLAST,
	output [`PB_NUM_BITS-1:0]	OSD
);

	///////////////////////////////////////////////////
	// Reduction core
	
	// CAR reg & CDR reg
	reg  [`PB_DS_BITS-1:0]		car_r;
	reg  [`PB_DS_BITS-1:0]		cdr_r;

	// L Address & L car reg
	reg  [`PB_DS_BITS-1:0]		la;

	// state reg
	reg [3:0]			st;
	reg				st_car;

	// RUN mode
	reg				lazyk_mode;
	reg				cont_mode;

	///////////////////////////////
	// tag check
	wire	is_cons, is_sym, is_num, is_s, is_k, is_i, is_l, is_n;
	wire	halt, halt_num;

	assign	is_cons = car_r[`PB_DS_TAG_BITS-1:0] == `PB_DS_TAG_CONS ?
				1'b1 : 1'b0;
	assign	is_sym  = car_r[`PB_DS_TAG_BITS-1:0] == `PB_DS_TAG_SYM ?
				1'b1 : 1'b0;
	assign	is_num  = car_r[`PB_DS_TAG_BITS-1:0] == `PB_DS_TAG_NUM ?
				1'b1 : 1'b0;

	assign	is_s	= car_r[`PB_DS_BITS-1:`PB_DS_TAG_BITS] == `PB_DS_S ?
				1'b1 : 1'b0;
	assign	is_k	= car_r[`PB_DS_BITS-1:`PB_DS_TAG_BITS] == `PB_DS_K ?
				1'b1 : 1'b0;
	assign	is_i	= car_r[`PB_DS_BITS-1:`PB_DS_TAG_BITS] == `PB_DS_I ?
				1'b1 : 1'b0;
	assign	is_l	= car_r[`PB_DS_BITS-1:`PB_DS_TAG_BITS] == `PB_DS_L ?
				1'b1 : 1'b0;
	assign	is_n	= car_r[`PB_DS_BITS-1:`PB_DS_TAG_BITS] == `PB_DS_N ?
				1'b1 : 1'b0;

	assign	halt	= (is_s && ENUM >= 'h3 || is_k && ENUM >= 'h2 || (is_i || is_n | is_l) && ENUM >= 'h1) ? 1'b0 : 1'b1;

	assign	halt_num= (ETOP != {`PB_DS_N, `PB_DS_TAG_SYM} || ENUM < 'h1) ?
				1'b1 : 1'b0;

	///////////////////////////////
	// RUN mode
	always @(posedge CLK `RST_TYPE) begin
		if(!RSTn) begin
				lazyk_mode <= 1'b0;
		end else if(RUN) begin
			if(LAZYK) begin
				lazyk_mode <= `TPD 1'b1;
			end else begin
				lazyk_mode <= `TPD 1'b0;
			end
		end
	end

	always @(posedge CLK `RST_TYPE) begin
		if(!RSTn) begin
				cont_mode <= 1'b0;
		end else if(RUN) begin
			if(CONT) begin
				cont_mode <= `TPD 1'b1;
			end else begin
				cont_mode <= `TPD 1'b0;
			end
		end
	end

	///////////////////////////////
	// State machine
	parameter	r_st_idle	= 4'h0,
			r_st_setup_car	= 4'h1,
			r_st_car	= 4'h2,
			r_st_cons_fetch	= 4'h3,
			r_st_output	= 4'h4,
			r_st_lazy_read	= 4'h5,
			r_st_lazy_read_2= 4'h6,
			r_st_lazy_read_3= 4'h7,
			r_st_update_cdr	= 4'h8;
	reg [3:0]			st_nxt;
	always @* begin
		case(st)
		r_st_idle: begin
			if(RUN) begin
				if(LAZYK) begin
					st_nxt	= r_st_setup_car;
				end else begin
					st_nxt	= r_st_car;
				end
			end else begin
					st_nxt	= r_st_idle;
			end
		end

		r_st_setup_car:		st_nxt	= r_st_car;

		r_st_car: begin
			if(is_cons) begin
				if(lazyk_mode && !st_car && ENUM < 'h1) begin
					st_nxt	= r_st_update_cdr;
				end else if(!MABSY) begin
					st_nxt	= r_st_cons_fetch;
				end else begin
					st_nxt	= r_st_car;
				end
			end else begin
				if(is_sym && halt ||
				   is_num && halt_num && !lazyk_mode) begin
					st_nxt	= r_st_idle;
				end else if( st_car && is_num && halt_num) begin
					st_nxt	= r_st_output;
				end else if(!st_car && ENUM < 'h1) begin
					st_nxt	= r_st_update_cdr;
				end else if(is_sym && is_l && !ISBSY) begin
					st_nxt	= r_st_lazy_read;
				end else begin
					st_nxt	= r_st_car;
				end
			end
		end

		r_st_cons_fetch: begin
			if(MRREQ) begin
					st_nxt	= r_st_car;
			end else begin
					st_nxt	= r_st_cons_fetch;
			end
		end

		r_st_output: begin
			if(OSBSY) begin
					st_nxt	= r_st_output;
			end else begin
				if(OSLAST) begin
					st_nxt	= r_st_idle;
				end else begin
					st_nxt	= r_st_setup_car;
				end
			end
		end

		r_st_update_cdr: begin
			if(cont_mode) begin
				st_nxt	= r_st_setup_car;
			end else begin
				st_nxt	= r_st_idle;
			end
		end

		r_st_lazy_read: begin
			if(MABSY) begin
					st_nxt	= r_st_lazy_read;
			end else begin
					st_nxt	= r_st_lazy_read_2;
			end
		end

		r_st_lazy_read_2: begin
			if(MABSY) begin
					st_nxt	= r_st_lazy_read_2;
			end else begin
					st_nxt	= r_st_lazy_read_3;
			end
		end

		r_st_lazy_read_3: begin
			if(MABSY) begin
					st_nxt	= r_st_lazy_read_3;
			end else begin
					st_nxt	= r_st_car;
			end
		end

		default:		st_nxt	= 4'bxxxx;
		endcase
	end

	always @(posedge CLK `RST_TYPE) begin
		if(!RSTn) begin
				st <= r_st_idle;
		end else if(INIT || EVF || FPVF) begin
			if(RUN) begin
				st <= `TPD r_st_car;
			end else begin
				st <= `TPD r_st_idle;
			end
		end else if(st != r_st_idle || RUN) begin
				st <= `TPD st_nxt;
		end
	end

	always @(posedge CLK `RST_TYPE) begin
		if(!RSTn) begin
				st_car <= 1'b1;
		end else if(RUN || st == r_st_update_cdr) begin
				st_car <= `TPD 1'b1;
		end else if(st == r_st_output && !OSBSY) begin
				st_car <= `TPD 1'b0;
		end
	end


	///////////////////////////////
	// Evaluation Stack control
	always @* begin
		case(st)
		r_st_setup_car:			EREQ = 1'b1;

		r_st_car: begin
			if(is_sym && !halt) begin
				if(is_s) begin
					if(MABSY) begin
						EREQ = 1'b0;
					end else begin
						EREQ = 1'b1;
					end
				end else if(is_l) begin
						EREQ = 1'b0;
				end else begin
						EREQ = 1'b1;
				end
			end else if(is_num && ENUM != 'h0) begin
						EREQ = 1'b1;
			end else begin
						EREQ = 1'b0;
			end
		end

		r_st_cons_fetch: begin
			if(MRREQ) begin
						EREQ = 1'b1;
			end else begin
						EREQ = 1'b0;
			end
		end

		default:			EREQ = 1'b0;
		endcase
	
	end

	always @* begin
		case(st)
		r_st_setup_car: begin
			if(st_car) begin
				ELEN = `PB_ES_LEN_SETUP_CAR;
			end else begin
				ELEN = `PB_ES_LEN_SETUP_CDR;
			end
		end

		r_st_car: begin
			if(is_num) begin
				ELEN = `PB_ES_LEN_POP1;
			end else if(is_s) begin
				ELEN = `PB_ES_LEN_POP3_PUSH2;
			end else if(is_k) begin
				ELEN = `PB_ES_LEN_POP2;
			end else if(is_i) begin
				ELEN = `PB_ES_LEN_POP1;
			end else if(is_n) begin
				ELEN = `PB_ES_LEN_POP1_PUSH1;
			end else begin
				ELEN = {`PB_ES_LBITS{1'bx}};
			end
		end

		r_st_cons_fetch: begin
				ELEN = `PB_ES_LEN_PUSH;
		end

		default:	ELEN = {`PB_ES_LBITS{1'bx}};
		endcase
	end

	always @* begin
		case(st)
		r_st_car: begin
			if(is_n) begin
					EWD  = car_r;
			end else begin
					EWD  = ETOP_2;
			end
		end

		r_st_cons_fetch: begin
					EWD  = MRD[`PB_DS_BITS-1:0];
		end

		default:		EWD  = {`PB_DS_BITS{1'bx}};
		endcase
	end

	assign	EWD_1	= FPA;

	///////////////////////////////
	// External Memory control
	wire [`PB_DS_BITS-1:0]	maa_w;
	assign	MAREQ	= st == r_st_car && (is_cons && !(lazyk_mode && !st_car & ENUM < 'h1) || is_sym && is_s && !halt) || st == r_st_lazy_read || st == r_st_lazy_read_2 || st == r_st_lazy_read_3 ? 1'b1 : 1'b0;
	assign	MAWRITE	= st == r_st_car && is_cons ? 1'b0 : 1'b1;
	assign	maa_w	= st == r_st_car && is_cons ? car_r : 
			  st == r_st_lazy_read_2 ||
			  st == r_st_lazy_read_3    ? la    : FPA;
	assign	MAA	= maa_w[`PB_DS_BITS-1:`PB_DS_TAG_BITS];

	always @* begin
		if(st == r_st_lazy_read) begin
			if(ISLAST) begin
				MAD = {`PB_DS_CONS, `PB_DS_TAG_CONS,
					`PB_DS_CHURHC_NUM_256, `PB_DS_TAG_CONS};
			end else begin
				MAD = {`PB_DS_CONS, `PB_DS_TAG_CONS,
					{`PB_DS_BITS-`PB_DS_TAG_BITS-`PB_NUM_BITS{1'b0}}, ISD, `PB_DS_TAG_CONS};
			end
		end else if(st == r_st_lazy_read_2) begin
				MAD = {car_r, FPA};
		end else if(st == r_st_lazy_read_3) begin
				MAD = {`PB_DS_I, `PB_DS_TAG_SYM,
				       `PB_DS_L, `PB_DS_TAG_SYM};
		end else begin
				MAD = {ETOP_1, ETOP_2};
		end
	end
					
	assign	MAS	= {`PB_MI_DBITS/8{1'b1}};

	assign	MRBSY	= 1'b0;



	///////////////////////////////
	// Free Pointer Interface
	assign	FPREQ	= st == r_st_car && is_sym && is_s && !halt && !MABSY ||
			  st == r_st_lazy_read   && !MABSY ||
			  st == r_st_lazy_read_2 && !MABSY ? 1'b1 : 1'b0;


	///////////////////////////////
	// CAR reg control
	always @(posedge CLK `RST_TYPE) begin
		if(!RSTn) begin
				car_r <= {`PB_DS_BITS{1'b0}};
		end else if(RUN && !LAZYK || st == r_st_setup_car) begin
				car_r <= `TPD cdr_r;
		end else if(!MABSY && st == r_st_lazy_read) begin
				car_r <= `TPD FPA;
		end else if(!MABSY && st == r_st_lazy_read_2) begin
				car_r <= `TPD la;
		end else if(EREQ && st != r_st_setup_car) begin
			if(st != r_st_car) begin	// r_st_cons_fetch
				car_r <= `TPD MRD[`PB_MI_DBITS-1:`PB_DS_BITS];
			end else if(is_num) begin	// r_st_car
				car_r <= `TPD car_r + 'h4;	// inc
			end else if(!(is_sym && is_l && ISBSY)) begin
				car_r <= `TPD ETOP;
			end
		end
	end

	///////////////////////////////
	// CDR reg control
	always @(posedge CLK `RST_TYPE) begin
		if(!RSTn) begin
				cdr_r <= {`PB_DS_BITS{1'b0}};
		end else if(SET_CDR) begin
				cdr_r <= `TPD SET_CDR_D;
		end else if(st == r_st_update_cdr) begin
				cdr_r <= `TPD car_r;
		end
	end

	///////////////////////////////
	// LA reg control (LazyRead cons address)
	always @(posedge CLK `RST_TYPE) begin
		if(!RSTn) begin
				la <= {`PB_DS_ZERO, `PB_DS_TAG_SYM};
		end else if(INIT) begin
				la <= `TPD {`PB_DS_ZERO, `PB_DS_TAG_SYM};
		end else if(!MABSY && st == r_st_lazy_read_2) begin
				la <= `TPD FPA;
		end else if(!MRBSY && st == r_st_cons_fetch &&
			la == {`PB_DS_ZERO, `PB_DS_TAG_SYM} &&
			MRD[`PB_DS_BITS-1:0] == {`PB_DS_L,`PB_DS_TAG_SYM}) begin
				la <= `TPD car_r;
		end
	end

	///////////////////////////////
	// stream input
	assign	ISREQ	= st == r_st_car && is_sym && is_l ? 1'b1 : 1'b0;

	///////////////////////////////
	// stream output
	assign	OSREQ	= st == r_st_output ? 1'b1 : 1'b0;
	assign	OSD	= car_r[`PB_NUM_BITS+`PB_DS_TAG_BITS-1:`PB_DS_TAG_BITS];
	assign	OSLAST	= car_r[`PB_NUM_BITS+`PB_DS_TAG_BITS];

	///////////////////////////////
	// debug outputs
	assign	BUSY	= st != r_st_idle ? 1'b1 : 1'b0;
	assign	CAR	= car_r;
	assign	CDR	= cdr_r;

endmodule

////////////////////////////////////////////////////////////////////////////////
// END OF IMPLEMENTATION
////////////////////////////////////////////////////////////////////////////////
