#! /usr/bin/env python2.1
"""NDTP client module.

  Author: tgz <tgz@users.sourceforge.jp>
  Version: 1.0.0
  Last Modified: 2003/03/29

  This program is based on LGPL

  Example:

  >>> import ndtplib
  >>> n = ndtplib.NDTP('localhost.localdomain')
  >>> n.version()
  'ndtpd version 2.3.6 on localhost.localdomain'
  >>> n.listdic()
  {2: '<Dictionary 2>', 1: '<Dictionary 1>'}
  >>> n.quit()
  (Connection Closed.)


"""
import socket
import string
import re

(NDTP_HOST, NDTP_PORT) = ('localhost', 2010)
CR        = '\n'
BUFSIZE   = 32768
FORWARD, BACKWARD = (0, 1)
ALFOR, ALBACK, KANFOR, KANBACK = ('a', 'A', 'k', 'K')
KEYWORD, HEADNUM = (0, 1)
DIFFERENTIAL, BITMAP = (0, 1)
prog1 = re.compile('[\x20-\x7f]')
prog2 = re.compile('^[\x20-\x7f]*$')

# Exception classes used by this module. 
class NDTPException(Exception):
    """Base class for all exceptions raised by this module."""

class NDTPServerDisconnected(NDTPException):
    """Not connected to any NDTP server.

    This exception is raised when the server unexpectedly disconnects,
    or when an attempt is made to use the NDTP instance before
    connecting it to a server.
    """

class NDTPPermissionDenied(NDTPException):
    """Permission to access denied."""

class NDTPSelectDictionaryError(NDTPException):
    """Failed to select dictionary."""

class NDTPSetDictionaryError(NDTPException):
    """Not set dictionary."""

class NDTPBitmapError(NDTPException):
    """Failed to use bitmap."""

class NDTP:
    file = None
    debuglevel = 0

    def __init__(self, host = '', port = 2010):
        """Initialize a new instance."""
        if host:
            self.connect(host, port)

    def set_debuglevel(self, debuglevel = 0):
        """Set the debug output level."""
        self.debuglevel = debuglevel

    def connect(self, host, port = 2010):
        if not port:
            port = NDTP_PORT
        message = 'host \'%s\' is not found.' % host
        self.sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        try:
            if self.debuglevel > 0:
                print 'connect:', (host, port)
            self.sock.connect((host, port))
        except socket.error:
            raise NDTPServerDisconnected(message)

    def send(self, str):
        """(Private)Send `str' to the server."""
        message = 'connection is not established.'
        cmd = str + CR
        if self.debuglevel > 0:
            print 'send:', cmd
        try:
            if self.sock:
                self.sock.send(cmd)
            else:
                raise NDTPServerDisconnected('please run connect() first.')
        except (socket.error,AttributeError):
            raise NDTPServerDisconnected(message)

    def version(self):
        """NDTP 'v' command, Get the NDTP server-version."""
        message = 'connection is not established.'
        try:
            self.send('v')
            if self.file is None:
                self.file = self.sock.makefile('r', BUFSIZE)
            CONT = 1
            while CONT:
                line = self.file.readline()
                if line[:2] == '$v':
                    version = line[2:-1]
                    CONT = 0
            return version
        except:
            raise NDTPServerDisconnected(message)

    def listdic(self):
        """NDTP 'T' command, Show the list of dictionaries."""
        self.send('T')
        resp = {}
        if self.file is None:
            self.file = self.sock.makefile('r', BUFSIZE)
        CONT = 1
        while CONT:
            line = self.file.readline()
            if self.debuglevel > 0:
                print '(line)',line
            if not line:
                CONT = 0
            elif line[:2] == '$*':
                CONT = 0
            else:
                antw, msg = (int(line[:2]),line[2:-1])
                resp[antw] = msg
        return resp

    def allow(self, user):
        """NDTP 'A' command, Get permission to use."""
        self.send('A%s' % user)
        message = 'permission denied.'
        if self.file is None:
            self.file = self.sock.makefile('r', BUFSIZE)
        CONT = 1
        while CONT:
            line = self.file.readline()
            if line[1] == 'A':
                if self.debuglevel > 0:
                    print '(Succeeded!)', user
                return 1
            elif line[1] == 'N':
                raise NDTPPermissionDenied(message)
            else:
                CONT = 0

    def setdic(self, dicnum = 1):
        """NDTP 'L' command, Set a dictionary."""
        message = ('dictionary No.\'%d\' is used by other user \
or no such a dictionary.' % dicnum)
        if dicnum == 0:
            raise NDTPSetDictionaryError('no such a dictionary No.\'%d\'.' % \
dicnum)
        self.send('L%s' % dicnum)
        if self.file is None:
            self.file = self.sock.makefile('r', BUFSIZE)
        line = ''
        CONT = 1
        while CONT:
            line = self.file.readline()
            if line[1] == '*':
                if self.debuglevel > 0:
                    print '(Succeeded!)', dicnum
                return 1
            elif line[1] == '&':
                raise NDTPSetDictionaryError(message)
            else:
                CONT = 0

    def listbitmapsize(self):
        """NDTP 'XI' command, Get list of available bitmap sizes."""
        self.send('XI')
        message = 'dictionary has not been set yet.'
        antw = []
        CONT = 1
        if self.file is None:
            self.file = self.sock.makefile('r', BUFSIZE)
        line = None
        while CONT:
            line = self.file.readline()
            if not line or line[:2] == '$$':
                CONT = 0
            elif line[:2] == '$I':
                pass
            elif line[:2] == '$<':
                raise NDTPSetDictionaryError(message)
            else:
                if self.debuglevel > 0:
                    print '(line)',line
                antw.append(line[:-1])
        return antw

    def setbitmapsize(self, size):
        """NDTP 'XL' command, Set bitmap size."""
        self.send('XL%s' % size)
        message1 = 'dictionary has not been set yet.'
        #message2 = 'no such bitmap size, \'%d\'.' % size
        message2 = 'no such bitmap size.'
        CONT = 1
        if self.file is None:
            self.file = self.sock.makefile('r', BUFSIZE)
        line = ''
        while CONT:
                line = self.file.readline()
                if not line:
                    CONT = 0
                elif line[:2] == '$*':
                    break
                elif line[:2] == '$<':
                    raise NDTPSetDictionaryError(message1)
                else:
                    raise NDTPBitmapError(message2)
        return 1

    def getbitmap(self, gaiji):
        """NDTP 'Xb' command, Get a bitmap datum."""
        self.send('Xb%s' % gaiji)
        if self.file is None:
            self.file = self.sock.makefile('r', BUFSIZE)
        CONT = 1
        bitmapdata = ''
        while CONT:
            line = self.file.readline()
            if not line or line[:2] == '$$':
                CONT = 0
            elif line[:2] == '$I':
                pass
            else:
                if self.debuglevel > 0:
                    print '(line)',line
                bitmapdata = bitmapdata + line
        return bitmapdata

    def getbitmaps(self):
        """NDTP 'XB' command, Get bitmap data."""
        self.send('XB')
        antw = {}
        if self.file is None:
            self.file = self.sock.makefile('r', BUFSIZE)
        gaiji = None
        bitmapdata = ''
        CONT = 1
        flag = DIFFERENTIAL
        while CONT:
            line = self.file.readline()
            if not line or line[:2] == '$$':
                CONT = 0
            elif line[:2] == '$I':
                pass
            else:
                if self.debuglevel > 0:
                    print '(line)',line
                if line[:2] == '$=':
                    if gaiji != None:
                        antw[gaiji] = bitmapdata
                        bitmapdata = ''
                    gaiji = line[2:-1]
                    flag = BITMAP
                else:
                    if flag == DIFFERENTIAL:
                        flag = BITMAP
                    bitmapdata = bitmapdata + line
        return antw

    def getindex(self):
        """NDTP 'I' command, Get index data."""
        self.send('I')
        message = 'dictionary has not been set yet.'
        antw = {}
        CONT = 1
        if self.file is None:
            self.file = self.sock.makefile('r', BUFSIZE)
        line = ''
        while CONT:
            line = self.file.readline()
            if not line or line[:2] == '$$':
                CONT = 0
            elif line[:2] == '$I':
                pass
            elif line[:2] == '$<':
                raise NDTPSetDictionaryError(message)
            else:
                if self.debuglevel > 0:
                    print '(line)',line
                index, frame = string.split(line[:-1])
                antw[index] = frame + ':0'
        return antw

    def copyright(self):
        """Show copyright of selected dictionary."""
        return self.getmain(self.getindex()['OK'])

    def getmain(self, header):
        """NDTP 'S' command, Send 'header' and get main data."""
        if self.file is None:
            self.file = self.sock.makefile('r', BUFSIZE)
        CONT = 1
        antw = ''
        self.send('S%s' % header)
        while CONT:
                line = self.file.readline()
                if not line or line[:2] == '$$':
                    CONT = 0
                elif line[:2] == '$1':
                    pass
                else:
                    if self.debuglevel > 0:
                        print '(line)',line
                    antw = antw + line
        return antw

    def ask(self, pattern):
        """Search 'pattern' and get result & header."""
        self.send(self.genquery(pattern))
        antw = {}
        if self.file is None:
            self.file = self.sock.makefile('r', BUFSIZE)
        CONT = 1
        flag = KEYWORD
        while CONT:
            line = self.file.readline()
            if not line or line[:2] == '$$':
                CONT = 0
            elif line[:2] == '$0':
                pass
            else:
                line = line[:-1]
                if self.debuglevel > 0:
                    print '(line)',line
                if flag == KEYWORD:
                   result = line
                   flag = HEADNUM
                else:
                   header = line
                   antw[result] = header
                   flag = KEYWORD
        return antw

    def quit(self):
        """NDTP 'Q' command, Terminate the NDTP session."""
        self.send('Q')
        self.close()
        if self.debuglevel > 0:
            print '(Connection Closed.)'

    def revchar(self, char):
        """(Private)Reverse multibyte-mixed character."""
        list = []
        i = len(char)
        while i:
            result = prog1.match(char[i-1:i])
            if result == None:
                flag = 2
            else:
                flag = 1
            list.append(char[i-flag:i])
            i = i - flag
        return string.join(list, '')

    def forb(self, pattern):
        """(Private)Discriminate pattern(forward/backward)."""
        flag = FORWARD
        if pattern[0] == '*':
            flag = BACKWARD
            pattern = self.revchar(pattern)
        return flag, pattern

    def kora(self, pattuple):
        """(Private)Discriminate if character include a non-ASCII char."""
        flag = ALFOR
        result = prog2.match(pattuple[1])
        if result == None:
            """Kana"""
            flag = KANFOR
            if pattuple[0] == BACKWARD: 
                flag = KANBACK
        else:
            """Alphabet"""
            if pattuple[0] == BACKWARD: 
                flag = ALBACK
        return flag + pattuple[1]

    def genquery(self, plain):
        """(Private)Generate query."""
        query = 'P' + self.kora(self.forb(plain))
        return query

    def close(self):
        """(Private)Close the connection to the NDTP server."""
        if self.file:
            self.file.close()
        self.file = None
        if self.sock:
            self.sock.close()
        self.sock = None

if __name__ == '__main__':
        server = NDTP('localhost.localdomain')
        print server.version()
        server.quit()
