/*!
  \file
  \brief つくばチャレンジ用のロガープログラム

  \author Satofumi KAMIMURA

  $Id$
*/

#include "mDifferentialDrive.h"
#include "mUrgCtrl.h"
#include "mCameraCtrl.h"
#include "Thread.h"
#include "delay.h"

using namespace qrk;


namespace
{
  enum {
    UrgFrameInterval = 4,
  };


  void connectDevices(mDifferentialDrive& run,
                      mUrgCtrl& urg, mCameraCtrl& camera,
                      int argc, char *argv[])
  {
    if (! run.connect("/dev/usb/ttyUSB0", 115200)) {
      printf("RunCtrl::connect: %s\n", run.what());
      exit(1);
    }

    if (! urg.connect("/dev/ttyACM0")) {
      printf("UrgCtrl::connect: %s\n", urg.what());
      exit(1);
    }

    if (! camera.connect()) {
      fprintf(stderr, "Warnning: CameraCtrl::connect: %s\n", camera.what());
    }
  }


  int runCapture_thread(void* args)
  {
    mDifferentialDrive* run = static_cast<mDifferentialDrive*>(args);
    FILE* fd = fopen("run_position.txt", "w");

    while (1) {
      int timestamp = -1;
      Position<int> pos = run->position(NULL, &timestamp);
      fprintf(fd, "%d, %d, %d, %d\n", timestamp, pos.x, pos.y, pos.to_deg());
      fflush(fd);

      delay(100);
    }
  }


  // URG のデータを読み捨て
  int urgCapture_thread(void* args)
  {
    mUrgCtrl* urg = static_cast<mUrgCtrl*>(args);
    static int times = 0;

    while (1) {
      std::vector<long> data;
      int n = urg->capture(data);
      if (n <= 0) {
        delay(urg->scanMsec() * UrgFrameInterval);

      } else {
        printf("%d: n = %d\n", times, n);
        ++times;
      }
    }
    //return n;
  }


  // ID を増やしながらカメラ画像取得
  int cameraCapture_thread(void* args)
  {
    mCameraCtrl* camera = static_cast<mCameraCtrl*>(args);
    static int capture_id = 0;

    while (1) {
      char buffer[256];
      snprintf(buffer, 256, "camera_%04d.jpg", capture_id);
      camera->capture(buffer);
      ++capture_id;

      delay(1000);
    }
    //return capture_id;
  }
};


int main(int argc, char *argv[])
{
  mDifferentialDrive run(argc, argv);
  mUrgCtrl urg(argc, argv);
  mCameraCtrl camera;

  connectDevices(run, urg, camera, argc, argv);

  // サーボフリー
  run.setServo(false);

  // URG データの取得間隔を調整
  urg.setCaptureMode(AutoCapture);
  urg.setCaptureTimes(UrgCtrl::Infinity);
  urg.setCaptureFrameInterval(UrgFrameInterval);

  // 定期的なデータ記録
  Thread run_thread(runCapture_thread, &run);
  Thread urg_thread(urgCapture_thread, &urg);
  Thread camera_thread(cameraCapture_thread, &camera);

  run_thread.run();
  urg_thread.run();
  camera_thread.run();

  run_thread.wait();
  urg_thread.wait();
  camera_thread.wait();
  printf("end.");

  return 0;
}
