/*!
  \file
  \brief 独立駆動二輪の制御

  \author Satofumi KAMIMURA

  $Id: DifferentialDrive.cpp 1219 2009-08-14 02:59:46Z satofumi $
*/

#include "DifferentialDrive.h"
#include "CommandPacket.h"
#include "SerialDevice.h"
#include "differential_drive.h"
#include "MathUtils.h"

using namespace qrk;
using namespace std;


struct DifferentialDrive::pImpl
{
  string error_message_;
  Connection* connection_;
  bool serial_created_;
  CommandPacket command_packet_;


  pImpl(void)
    : error_message_("no error."),
      connection_(NULL), serial_created_(false)
  {
  }


  ~pImpl(void)
  {
    if (serial_created_) {
      delete connection_;
    }
  }


  bool connect(const char* device, long baudrate)
  {
    disconnect();
    if (! connection_) {
      setSerialDevice();
    }

    if (! connection_->connect(device, baudrate)) {
      error_message_ = connection_->what();
      return false;
    }

    unsigned char major;
    unsigned char minor;
    unsigned char micro;
    if (! command_packet_.version(&major, &minor, &micro)) {
      error_message_ = "no response from controller.";
      connection_->disconnect();
      return false;
    }

    if ((major != Major) || (minor != Minor)) {
      error_message_ = "please update controller version.";
      connection_->disconnect();
      return false;
    }

    // サーボ状態にする
    stop();

    return true;
  }


  void disconnect(void)
  {
    if (connection_) {
      connection_->disconnect();
    }
  }


  bool isConnected(void)
  {
    return (connection_ == NULL) ? false : connection_->isConnected();
  }


  void setSerialDevice(void)
  {
    // シリアル接続のオブジェクトを生成する
    setConnection(new SerialDevice);
    serial_created_ = true;
  }


  void setConnection(Connection* connection)
  {
    connection_ = connection;
    command_packet_.setConnection(connection);

    // SerialDevice のリソース管理はユーザ側に任せる
    serial_created_ = false;
  }


  void stop(void)
  {
    if (! command_packet_.stop()) {
      // !!! 例外を投げるべき
    }
  }


  long convertDir16(const Angle& angle)
  {
    return static_cast<long>(65536.0 * angle.to_rad() / (2.0 * M_PI));
  }
};


DifferentialDrive::DifferentialDrive(void) : pimpl(new pImpl)
{
}


DifferentialDrive::~DifferentialDrive(void)
{
}


const char* DifferentialDrive::what(void) const
{
  return pimpl->error_message_.c_str();
}


bool DifferentialDrive::connect(const char* device, long baudrate)
{
  return pimpl->connect(device, baudrate);
}


void DifferentialDrive::disconnect(void)
{
  pimpl->disconnect();
}


bool DifferentialDrive::isConnected(void) const
{
  return pimpl->isConnected();
}


void DifferentialDrive::setConnection(Connection* connection)
{
  pimpl->setConnection(connection);
}


Connection* DifferentialDrive::connection(void)
{
  return pimpl->connection_;
}


void DifferentialDrive::followLine(const Position<long>& line,
                                   const Coordinate* coordinate)
{
  Position<long> target_position(line);
  if (coordinate) {
    target_position = coordinate->pointPosition(target_position, this);
  }

  long dir16 = pimpl->convertDir16(target_position.angle);
  if (! pimpl->command_packet_.followLine(target_position.x,
                                          target_position.y, dir16)) {
    // !!! 例外を投げるべき
  }
}


void DifferentialDrive::stopLine(const Position<long>& line,
                                 const Coordinate* coordinate)
{
  static_cast<void>(line);
  static_cast<void>(coordinate);

  // !!!
}


void DifferentialDrive::followCircle(const Point<long>& center, long radius,
                                     const Coordinate* coordinate )
{
  Position<long> target_point(center.x, center.y, deg(0));
  if (coordinate) {
    target_point = coordinate->pointPosition(target_point, this);
  }

  if (! pimpl->command_packet_.followCircle(target_point.x,
                                            target_point.y, radius)) {
    // !!! 例外を投げるべき
  }
}


void DifferentialDrive::stopCircle(const Point<long>& center, long radius,
                                   const Angle& angle,
                                   const Coordinate* coordinate)
{
  static_cast<void>(center);
  static_cast<void>(radius);
  static_cast<void>(angle);
  static_cast<void>(coordinate);

  // !!!
}


void DifferentialDrive::spin(const Angle& angle_per_sec)
{
  long dir16 = pimpl->convertDir16(angle_per_sec);
  if (! pimpl->command_packet_.spin(dir16)) {
    // !!! 例外を投げるべき
  }
}


void DifferentialDrive::rotate(const Angle& angle, const Coordinate* coordinate)
{
  Position<long> target_position(0, 0, angle);
  if (coordinate) {
    target_position = coordinate->pointPosition(target_position, this);
  }

  long dir16 = pimpl->convertDir16(target_position.angle);
  if (! pimpl->command_packet_.rotate(dir16)) {
    // !!! 例外を投げるべき
  }
}


void DifferentialDrive::stop(void)
{
  pimpl->stop();
}


Position<long> DifferentialDrive::position(Coordinate* coordinate) const
{
  Position<long> target_position;

  long mm[2] = { 0, 0 };
  unsigned short dir16 = 0;
  if (! pimpl->command_packet_.position(mm, &dir16)) {
    pimpl->error_message_ = "no response from controller.";
    pimpl->connection_->disconnect();

    // !!! 例外を投げるべき

    // (0, 0, deg(0)) のままデータを返す
    return target_position;
  }

  target_position.x = mm[0];
  target_position.y = mm[1];
  target_position.angle = rad(2.0 * M_PI * dir16 / 65536.0);

  if (coordinate) {
    return coordinate->pointPosition(target_position, this);
  } else {
    return target_position;
  }
}


long DifferentialDrive::straightVelocity(void) const
{
  // !!!
  return 0;
}


Angle DifferentialDrive::rotateVelocity(void) const
{
  // !!!
  return deg(0);
}


void DifferentialDrive::setRobotPosition(const Position<long>& position,
                                         const Coordinate* coordinate)
{
  static_cast<void>(position);
  static_cast<void>(coordinate);

  // !!!
}


void DifferentialDrive::setStraightVelocity(long mm_per_sec)
{
  static_cast<void>(mm_per_sec);
  // !!!
}


void DifferentialDrive::setRotateVelocity(const Angle& angle_per_sec)
{
  static_cast<void>(angle_per_sec);

  // !!!
}


void DifferentialDrive::setStraightAcceleration(long mm_per_sec2)
{
  static_cast<void>(mm_per_sec2);

  // !!!
}


void DifferentialDrive::setRotateAcceleration(const Angle& angle_per_sec2)
{
  static_cast<void>(angle_per_sec2);

  // !!!
}


bool DifferentialDrive::isStable(void) const
{
  // !!!
  return false;
}


void DifferentialDrive::setWheelVelocity(int id, long mm_per_sec)
{
  pimpl->command_packet_.setWheelVelocity(id, mm_per_sec);
}


string DifferentialDrive::currentCommand(void) const
{
  // !!!
  return "Not implemented.";
}


void DifferentialDrive::saveCommand(void)
{
  // !!!
}


void DifferentialDrive::restoreCommand(void)
{
  // !!!
}


void DifferentialDrive::commitParameters(void)
{
  // !!!
}


#if 0
void DifferentialDrive::setMotorParameter(int id, const motor_t& motor)
{
  static_cast<void>(id);
  static_cast<void>(motor);
  // !!!
}


void DifferentialDrive::setEncoderParameter(int id, const encoder_t& encoder)
{
  static_cast<void>(id);
  static_cast<void>(encoder);
  // !!!
}


void DifferentialDrive::setWheelParameter(int id, const wheel_t& wheel)
{
  static_cast<void>(id);
  static_cast<void>(wheel);
  // !!!
}


void DifferentialDrive::setBodyParameter(const body_t& body)
{
  static_cast<void>(body);
  // !!!
}
#endif
