/*!
  \file
  \brief イベントのスケジュール管理

  \author Satofumi KAMIMURA

  $Id: EventScheduler.cpp 1241 2009-08-22 10:16:44Z satofumi $
*/

#include "EventScheduler.h"
#include "Device.h"
#include "ManagedTicks.h"
#include "OdeHandler.h"
#include "ModelManager.h"
#include "Thread.h"
#include "Lock.h"
#include "LockGuard.h"
#include "system_ticks.h"
#include "system_delay.h"
#include <vector>
#include <map>
#include <cstdlib>

using namespace qrk;
using namespace std;


namespace
{
  enum {
    X = 0,
    Y = 1,
    Z = 2,
  };

  typedef multimap<long, Device*> Devices;

  typedef map<dGeomID, long*> RayIds;


  Lock mutex_;
  ManagedTicks timer_;
  long current_ticks_ = 0;
  bool stop_ = false;

  Devices devices_;
  ConditionVariable* delay_condition_ = NULL;
  long delay_wakeup_ticks_ = 0;

  dWorldID world_;
  dSpaceID space_;
  dGeomID ground_;
  dJointGroupID contact_group_;

  RayIds ray_ids_;
  ModelManager model_manager_;


  bool measerLaserDistance(int n, dContact contact[], dGeomID o1, dGeomID o2)
  {
    if (n > 0) {
      RayIds::iterator ray_it = ray_ids_.find(o1);
      if (ray_it == ray_ids_.end()) {
        ray_it = ray_ids_.find(o2);
      }
      if (ray_it != ray_ids_.end()) {
        // レーザーの判定処理
        dVector3 start;
        dVector3 dir;
        dGeomID ray_id = ray_it->first;
        dGeomRayGet(ray_id, start, dir);

        // 計算した距離には、ランダムで 10 [mm] の誤差を載せる
        // !!! 誤差を付加することで、エラーを示す距離になってもよいことにする
        // !!! 誤差を付加する処理は、range_finder 内で行うように修正する
        double ray_distance =
          sqrt(pow(contact[0].geom.pos[X] - start[X], 2) +
               pow(contact[0].geom.pos[Y] - start[Y], 2) +
               pow(contact[0].geom.pos[Z] - start[Z], 2)) * 1000.0;
        ray_distance += (20.0 * rand() / (RAND_MAX + 1.0)) - 10.0;
        *(ray_it->second) = static_cast<long>(ray_distance);
        return true;
      }
    }
    return false;
  }


  bool isObstacle(dGeomID id)
  {
    dBodyID body_id = dGeomGetBody(id);
    return model_manager_.obstacle(body_id);
  }


  void nearCallback(void* data, dGeomID o1, dGeomID o2)
  {
    // !!! data が何なのかを確認すべき
    static_cast<void>(data);

    enum { N = 10 };            // 接触点数の最大値
    dContact contact[N];

    int n = dCollide(o1, o2, N, &contact[0].geom, sizeof(dContact));
    if (measerLaserDistance(n, contact, o1, o2)) {
      return;
    }

    bool is_obstacle = isObstacle(o1) || isObstacle(o2);
    bool is_ground = (ground_ == o1) || (ground_ == o2);
    if (! is_ground) {
      if ((n <= 0) || (! is_obstacle)) {
        // 衝突判定させたい物体のときは return させない
        return;
      }
    }

    for (int i = 0; i < n; ++i) {
      contact[i].surface.mode =
        dContactBounce | dContactSoftCFM | dContactSoftERP;
      contact[i].surface.bounce = 0.5;
      contact[i].surface.bounce_vel = 0.1;

      contact[i].surface.soft_cfm = 0.01;
      contact[i].surface.soft_erp = 0.1;

      // !!! 車輪かそうでないかによって、摩擦係数を変更すべき
      // !!! 車輪でないときは、摩擦係数を 1.0 ぐらいにする
      contact[i].surface.mu = 6.2;

      dJointID c = dJointCreateContact(world_, contact_group_, &contact[i]);
      dJointAttach(c,
                   dGeomGetBody(contact[i].geom.g1),
                   dGeomGetBody(contact[i].geom.g2));
    }
  }


  void clearLasers(void)
  {
    for (RayIds::iterator it = ray_ids_.begin(); it != ray_ids_.end(); ++it) {
      dGeomDestroy(it->first);
    }
    ray_ids_.clear();
  }
}


struct EventScheduler::pImpl
{
  Thread thread_;


  pImpl(void)
    : thread_(scheduler_loop, NULL)
  {
  }


  static pImpl* object(void)
  {
    static pImpl singleton_object;
    return &singleton_object;
  }


  void run(void)
  {
    scheduler_loop(NULL);
  }


  static int scheduler_loop(void*)
  {
    timer_.setTicksFunction(system_ticks);
    timer_.play();

    enum {
      MaxLoopTimes = 100,
    };

    OdeHandler ode_;
    world_ = ode_.world();
    space_ = ode_.space();
    ground_ = ode_.ground();
    contact_group_ = ode_.contactGroup();

    while (! stop_) {
      mutex_.lock();

      // システム時刻を取得
      long target_ticks = timer_.ticks();
      int loop_times = min(target_ticks - current_ticks_,
                           static_cast<long>(MaxLoopTimes));

      for (int i = 0; i < loop_times; ++i) {
        // その時刻にイベント・リストに登録されているタスクがあれば、処理
        if (! devices_.empty()) {

          for (Devices::iterator it = devices_.begin();
               it != devices_.upper_bound(current_ticks_); ++it) {
            Device* device = it->second;

            // 実行し、次のタイミングを取得して再登録する
            device->execute();
            long next_ticks = current_ticks_ + device->nextExecuteInterval();
            devices_.insert(pair<long, Device*>(next_ticks, device));
          }
          devices_.erase(devices_.begin(),
                         devices_.upper_bound(current_ticks_));
        }

        // delay() の起床
        // !!! 複数スレッドから delay() が呼ばれる場合には変更する
        if (delay_condition_ && (current_ticks_ >= delay_wakeup_ticks_)) {
          delay_condition_->wakeup();
          delay_condition_ = NULL;
        }

        // ODE の時間を進める
        // !!! モデルがないときには、ODE 処理をしないようにする
        // !!! 遅ければ、dWorldQuickStep(), dWorldStepFirst1() の使用を検討する
        dSpaceCollide(space_, 0, &nearCallback);
        dWorldStep(world_, 0.001);
        clearLasers();
        dJointGroupEmpty(contact_group_);

        ++current_ticks_;
      }
      mutex_.unlock();

      // !!! QThread::msleep(1); に変更する
      system_delay(1);
    }

    return 0;
  }


  void setLaser(long* buffer, dGeomID laser_id)
  {
    ray_ids_[laser_id] = buffer;
  }
};


EventScheduler::EventScheduler(void) : pimpl(pImpl::object())
{
}


EventScheduler::~EventScheduler(void)
{
}


void EventScheduler::lock(void)
{
  mutex_.lock();
}


void EventScheduler::unlock(void)
{
  mutex_.unlock();
}


void EventScheduler::start(void)
{
  pimpl->thread_.run();
}


void EventScheduler::terminate(void)
{
  LockGuard guard(mutex_);
  stop_ = true;

  if (delay_condition_) {
    delay_condition_->wakeup();
    delay_condition_ = NULL;
  }
}


long EventScheduler::ticks(void) const
{
  mutex_.lock();
  long ticks = current_ticks_;
  mutex_.unlock();

  return ticks;
}


void EventScheduler::play(void)
{
  LockGuard guard(mutex_);
  timer_.play();
}


void EventScheduler::pause(void)
{
  LockGuard guard(mutex_);
  timer_.pause();
}


void EventScheduler::moreFaster(void)
{
  LockGuard guard(mutex_);
  timer_.moreFaster();
}


void EventScheduler::moreSlower(void)
{
  LockGuard guard(mutex_);
  timer_.moreSlower();
}


void EventScheduler::registerDevice(Device* device)
{
  LockGuard guard(mutex_);
  devices_.insert(pair<long, Device*>(current_ticks_, device));
}


void EventScheduler::removeDevice(Device* device)
{
  LockGuard guard(mutex_);
  for (Devices::iterator it = devices_.begin(); it != devices_.end();) {
    if (it->second == device) {
      devices_.erase(it++);
    } else {
      ++it;
    }
  }
}


bool EventScheduler::registerDelayEvent(ConditionVariable* condition,
                                        size_t msec)
{
  LockGuard gaurd(mutex_);
  if (stop_) {
    return false;
  }
  delay_wakeup_ticks_ = current_ticks_ + msec;
  delay_condition_ = condition;

  return true;
}


void EventScheduler::setLaser(long* buffer, dGeomID laser_id)
{
  pimpl->setLaser(buffer, laser_id);
}
