/*!
  \file
  \brief テーマ時計の描画

  \author Satofumi KAMIMURA

  $Id$

  \todo ドラッグ＆ドロップでテーマをロードできるようにする
*/

#include <QLabel>
#include <QPixmap>
#include <QBitmap>
#include <QTimer>
#include <QTime>
#include <QAction>
#include <QtGui>
#include <math.h>
#include "ThemeClockWidget.h"
#include "ThemeProperty.h"


struct ThemeClockWidget::pImpl {
  enum {
    BaseOrg = 0,
    Hari_h,
    Hari_m,
    Hari_s,
    Digital,
  };

  enum {
    H_pos = 0,
    M_pos,
    S_pos,

    H1_pos,
    H2_pos,
    Cl_pos,
    M1_pos,
    M2_pos,
    S1_pos,
    S2_pos,

    Ap_pos,

    Last_pos,
  };

  ThemeProperty* theme_;
  bool theme_loaded_;
  QVector<QLabel*> labels_;
  QVector<QPixmap*> pixmaps_;
  QVector<QPoint> positions_;
  QRegion* region_;
  QTimer clock_timer_;
  QPoint drag_position_;

  pImpl(ThemeProperty* theme)
    : theme_(theme), theme_loaded_(false) {
    clock_timer_.setInterval(1000);

    // !!! 秒の描画、点滅、などの設定は、ここで変数に代入してしまうべき

  }

  void initialize(ThemeClockWidget* parent) {
    // 描画の初期化
    if (! initForm(parent)) {
      return;
    }

    // 時計の動作開始
    startClock(parent);
  }

  void startClock(ThemeClockWidget* parent) {
    updateClock();
    connect(&clock_timer_, SIGNAL(timeout()), parent, SLOT(updateClock()));
    firstUpdate(parent);
  }

  void updateClock(void) {
    QTime time = QTime::currentTime();
    updateAnalog(time.hour(), time.minute(), time.second());
    updateDigital(time.hour(), time.minute(), time.second());
  }

  void firstUpdate(ThemeClockWidget* parent) {
    // PC 時刻を取得し、更新開始のミリ秒タイミングを調整する
    QTime current_time = QTime::currentTime();

    clock_timer_.stop();
    QTimer::singleShot(1000 - current_time.msec(), parent, SLOT(startTimer()));
  }

  // !!! main() でフォーム位置が取得できなければ、~pImpl() で settings.save() を呼ぶ

  bool initForm(ThemeClockWidget* parent) {
    if (theme_loaded_) {
      // 既に初期化済みだったら、戻る
      return false;
    }

    // 画像の読み出し
    theme_loaded_ |= loadImages(parent);
    if (! theme_loaded_) {
      initTemporaryForm(parent);
      return false;
    }

    // "base_org" 画像幅で Widget サイズを更新
    region_ = new QRegion(pixmaps_[BaseOrg]->mask());
    parent->resize(pixmaps_[BaseOrg]->size());
    //parent->setBaseSize(pixmaps_[BaseOrg]->size());

    // 描画位置の読み出し
    loadPositions();

    // ラベル位置の更新
    moveLabels();

    // 説明の表示設定
    parent->
      setToolTip(tr("Drag the click with the left mouse button.\n"
                    "Use the right mouse button to open a context menu."));

    return true;
  }

  void initTemporaryForm(ThemeClockWidget* parent) {
    parent->setMinimumSize(QSize(200, 100));
    parent->setToolTip(tr("Drop theme file."));

    parent->setAcceptDrops(true);
  }

  void initMenu(ThemeClockWidget* parent) {
    QAction* quit_action = new QAction(tr("E&xit"), parent);
    quit_action->setShortcut(tr("Ctrl-Q"));
    connect(quit_action, SIGNAL(triggered()), qApp, SLOT(quit()));
    parent->addAction(quit_action);

    parent->setContextMenuPolicy(Qt::ActionsContextMenu);
  }

  bool setPixmap(QPixmap* pixmap, const char* tag) {

    if (! theme_->getPixmap(*pixmap, tag)) {
      return false;
    }
    return true;
  }

  bool loadImages(ThemeClockWidget* parent) {
    const char* ImageTags[] = {
      "base/base_org",
      "analog/hari_h",
      "analog/hari_m",
      "analog/hari_s",
      "digital/parts",
    };
    size_t n = sizeof(ImageTags) / sizeof(ImageTags[0]);
    for (size_t i = 0; i < n; ++i) {
      QLabel* label = new QLabel(parent);
      QPixmap* pixmap = new QPixmap;

      // 画像の読み出し
      if (! setPixmap(pixmap, ImageTags[i])) {
        // !!! 何もしないかも
        // !!! base_org がエラーになったら、戻るべきか？
        return false;
      }

      if (i <= Hari_s) {
        label->setPixmap(*pixmap);
      }

      label->resize(pixmap->size());
      labels_.push_back(label);
      pixmaps_.push_back(pixmap);
    }

    // !!! デジタル文字用の追加。もう少し、適切な実装にする
    // !!! このままだと、何が 9 なのかとか、全然わからない
    for (size_t i = 0; i < 9; ++i) {
      QLabel* label = new QLabel(parent);
      labels_.push_back(label);
    }

    return true;
  }

  bool loadPositions(void) {
    const char* PositionTags[] = {
      "analog/h_xpos", "analog/h_ypos",
      "analog/m_xpos", "analog/m_ypos",
      "analog/s_xpos", "analog/s_ypos",
      "digital/h1_xpos", "digital/h1_ypos",
      "digital/h2_xpos", "digital/h2_ypos",
      "digital/cl_xpos", "digital/cl_ypos",
      "digital/m1_xpos", "digital/m1_ypos",
      "digital/m2_xpos", "digital/m2_ypos",
      "digital/s1_xpos", "digital/s1_ypos",
      "digital/s2_xpos", "digital/s2_ypos",
      "digital/ap_xpos", "digital/ap_ypos",
    };

    size_t n = sizeof(PositionTags) / sizeof(PositionTags[0]);
    for (size_t i = 0; i < n; i += 2) {
      int x = theme_->getPosition(PositionTags[i]);
      int y = theme_->getPosition(PositionTags[i + 1]);
      positions_.push_back(QPoint(x, y));
    }

    return true;
  }

  void moveLabels(void) {
    for (size_t i = 0; i < 3; ++i) {
      const QSize& size = labels_[i + 1]->size();
      int x = positions_[i].x() - size.width() / 2;
      int y = positions_[i].y() - size.height() / 2;
      labels_[i + 1]->move(x, y);

      // !!! 描画のときは、isNull() で画像の有効判定をすべき
    }
  }

  void updateAnalog(int hour, int minute, int second) {
    // !!! 画像を回転させて、label_ に張り付ける

    int degrees[3];
    double adjusted_hour = (hour > 12) ? hour - 12 : hour;
    degrees[0] =
      static_cast<int>((5.0 * adjusted_hour) + (5.0 * minute / 60.0));
    degrees[1] = minute;
    degrees[2] = second;

    for (size_t i = 0; i < 3; ++i) {
      QTransform transform;
      double degree = 360.0 * degrees[i] / 60.0;
      transform.rotate(degree);
      QPixmap rotated_pixmap = pixmaps_[Hari_h + i]->transformed(transform);

      const QSize& size = rotated_pixmap.size();
      int x = positions_[i].x() - size.width() / 2;
      int y = positions_[i].y() - size.height() / 2;
      labels_[Hari_h + i]->setPixmap(rotated_pixmap);
      labels_[Hari_h + i]->resize(rotated_pixmap.size());
      labels_[Hari_h + i]->move(x, y);
    }
  }

  void updateDigital(int hour, int minute, int second) {
    // !!! <digital/colon> の on, off で、コロンの点滅を制御する

    // !!! digital/sec で、判定するべき
    // !!! digital/s1_xpos があれば、秒の描画を行わせる

    char buffer[9];
    int adjusted_hour = (hour > 12) ? hour - 12 : hour;
    sprintf(buffer, "%02d%c%02d", adjusted_hour,
            (second % 2) ? ':' : ' ', minute);
    //':', minute);
    //fprintf(stderr, "%s\n", buffer);

    for (size_t i = 0; i < 7; ++i) {
      drawDigital(i, buffer[i]);
    }
    drawDigital(7, (hour >= 12) ? 'P' : 'A');

#if 1
    sprintf(buffer, "%02d", second);
    drawDigital(5, buffer[0]);
    drawDigital(6, buffer[1]);
#endif
  }

  void drawDigital(int index, char ch) {
    // !!! 文字毎の画像をつくり出して、使いまわすべき

    QPoint point = positions_[H1_pos + index];
    if (point.x() < 0) {
      return;
    }

    // !!! 汚いな...
    const QSize& size = pixmaps_[Digital]->size();
    QPixmap digital;
    if ((ch >= '0') && (ch <= '9')) {
      digital = pixmaps_[Digital]->copy(size.width() * (ch - '0') / 14,
                                        0, size.width() / 14, size.height());

    } else if (ch == ':') {
      digital = pixmaps_[Digital]->copy(size.width() * 11 / 14,
                                        0, size.width() / 14, size.height());

    } else if (ch == ' ') {
      digital = pixmaps_[Digital]->copy(size.width() * 10 / 14,
                                        0, size.width() / 14, size.height());

    } else if ((ch == 'A') || (ch == 'P')) {
      int sub_index = (ch == 'A') ? 12 : 13;
      digital = pixmaps_[Digital]->copy(size.width() * sub_index / 14,
                                        0, size.width() / 14, size.height());

    } else {
      return;
    }
    labels_[Digital + index]->setPixmap(digital);
    labels_[Digital + index]->resize(digital.size());
    labels_[Digital + index]->move(positions_[H1_pos + index]);
  }
};


ThemeClockWidget::ThemeClockWidget(ThemeProperty* theme, QWidget* parent)
  : QWidget(parent, Qt::FramelessWindowHint), pimpl(new pImpl(theme)) {
  pimpl->initialize(this);

  // 右クリックメニューの追加
  pimpl->initMenu(this);
}


ThemeClockWidget::~ThemeClockWidget(void) {
}


void ThemeClockWidget::resizeEvent(QResizeEvent* /* event */) {
  if (pimpl->theme_loaded_) {
    setMask(*pimpl->region_);
  }
}


void ThemeClockWidget::firstUpdate(void) {
  pimpl->firstUpdate(this);
}


void ThemeClockWidget::startTimer(void) {

  pimpl->clock_timer_.start();
  updateClock();
}


void ThemeClockWidget::updateClock(void) {

  setUpdatesEnabled(false);
  pimpl->updateClock();
  setUpdatesEnabled(true);
}


void ThemeClockWidget::mousePressEvent(QMouseEvent* event) {

  if (event->button() == Qt::LeftButton) {
    pimpl->drag_position_ = event->globalPos() - frameGeometry().topLeft();
    event->accept();
  }
}


void ThemeClockWidget::mouseMoveEvent(QMouseEvent* event) {
  if (event->buttons() & Qt::LeftButton) {
    move(event->globalPos() - pimpl->drag_position_);
    event->accept();
  }
}


void ThemeClockWidget::dragEnterEvent(QDragEnterEvent* event) {
  setBackgroundRole(QPalette::Highlight);
  event->acceptProposedAction();
}


void ThemeClockWidget::dragMoveEvent(QDragEnterEvent* event) {
  event->acceptProposedAction();
}


void ThemeClockWidget::dropEvent(QDropEvent* event) {
  const QMimeData* mime_data = event->mimeData();

  if (mime_data->hasUrls()) {
    QList<QUrl> url_list = mime_data->urls();
    for (int i = 0; i < url_list.count(); ++i) {
      fprintf(stderr, "%s\n",
	      url_list.at(i).path().toStdString().c_str());

      // テーマを作り直して登録
      // !!! めちゃくちゃ、変な実装だな！
      // !!! つか、メモリリークさせてるしな！
      //ThemeProperty* theme = new ThemeProperty(url_list.at(i).path());
      ThemeProperty* theme = new ThemeProperty("C:\\lm_clocktheme_02.dat");
      std::swap(theme, pimpl->theme_);
      pimpl->initialize(this);
      //autoFillBackground();
      //hide();
      //show();
      //repaint(*pimpl->region_);
      //setUpdatesEnabled(true);

      // !!! ロードできたら、処理を中断
      // !!! とりあえず中断
      break;
    }
  }

  event->acceptProposedAction();
}
