/*!
  \file
  \brief テーマデータの読み出し

  \author Satofumi KAMIMURA

  $Id$

  \todo テーマがないときには、デフォルトのテーマを適用する
*/

#include <QString>
#include <QMap>
#include <QXmlDefaultHandler>
#include <QPixmap>
#include "ThemeProperty.h"
#include "lm_rrda.h"


struct ThemeProperty::pImpl {
  class XmlHandler : public QXmlDefaultHandler {
    QString current_text_;
    QVector<QString> tag_stack_;

  public:
    QMap<QString, QString> tag_map_;

    bool startElement(const QString&, /* namespaceURI */
                      const QString&, /* localName */
                      const QString& qName,
                      const QXmlAttributes& /* attributes */) {
      tag_stack_.push_back(qName);
      current_text_.clear();
      return true;
    }

    bool endElement(const QString&, /* namespaceURI */
                    const QString&, /* localName */
                    const QString& /* qName */) {
      QString value = current_text_.trimmed();
      current_text_.clear();

      // 前後の空白を除去して空になったら、終端とみなして何もしない
      if (! value.isEmpty()) {

        QString key;
        for (QVector<QString>::iterator it = tag_stack_.begin();
             it != tag_stack_.end(); ++it) {
          if (it == tag_stack_.begin()) {
            // !!! 格好悪いなぁ...
            continue;
          }
          //fprintf(stderr, "%s : ", it->toStdString().c_str());
          key += *it + '/';
        }
        key.chop(1);

        // タグ名の登録
        tag_map_.insert(key, value);
        //fprintf(stderr, "%s : ", key.toStdString().c_str());
        //fprintf(stderr, "=> %s\n", value.toStdString().c_str());
      }
      tag_stack_.pop_back();

      return true;
    }

    bool characters(const QString& str) {
      current_text_ += str;
      return true;
    }

    bool fatalError(const QXmlParseException& /*exception */) {
      // !!! 現状、XML のデータ以降に変なデータが返されるため
      // !!! それを読み飛ばすために false にしてある
      return false;
    }
  };

  XmlHandler xml_handler_;


  pImpl(const QString& theme_name) {
    // テーマデータを開く
    // !!! key は、登録できるようにする
    if (lm_rrda_open(theme_name.toStdString().c_str(), "LMTC") == FALSE) {
      // デフォルト設定を使う
      // !!! エラーメッセージが表示されていれば、それでよい気もしてきた
      // !!!
      return;
    }
    QByteArray data;
    if (! readRrdaData(data, "config")) {
      // !!! エラー
      // デフォルト設定を使う
      return;
    }

    // 設定ファイル(XML) の読みだし
    readConfigXml(data);
  }

  ~pImpl(void) {
    close();
  }

  void close(void) {
    lm_rrda_close();
  }

  void readConfigXml(QByteArray& data) {

    // 登録内容のクリア
    xml_handler_.tag_map_.clear();

    // XML 内容の読み出し
    QXmlSimpleReader reader;
    reader.setContentHandler(&xml_handler_);
    reader.setErrorHandler(&xml_handler_);

    QXmlInputSource xml_resource;
    xml_resource.setData(data);
    reader.parse(xml_resource);
  }

  bool readRrdaData(QByteArray& data, const char* data_name) {

    int data_size = lm_rrda_data_size(data_name);
    if (data_size == 0) {
      // !!! エラーメッセージを更新すべき
      // !!! あと、エラーメッセージは要求されたときのみ表示するようにする
      fprintf(stderr, "tm_config: config data is not found.\n");
      return false;
    }

    const char* raw_data = (char *)lm_rrda_load_data(data_name);
    for (int i = 0; i < data_size; ++i) {
      data.push_back(raw_data[i]);
    }
    // !!! 本当は、開放すべきなのかもしれないが、とりあえず保留
    // !!! lm_rrda.c の実装を確認してから

    return true;
  }
};


ThemeProperty::ThemeProperty(const QString& theme_name)
  : pimpl(new pImpl(theme_name)) {
}


ThemeProperty::~ThemeProperty(void) {
}


bool ThemeProperty::getPixmap(QPixmap& pixmap, const char* tag) const {
  QMap<QString, QString>& tag_map = pimpl->xml_handler_.tag_map_;
  QMap<QString, QString>::iterator p = tag_map.find(tag);
  if (p == tag_map.end()) {
    return false;
  }
  QString pixmap_name = tag_map.value(tag);
  QByteArray data;
  if (! pimpl->readRrdaData(data, pixmap_name.toStdString().c_str())) {
    return false;
  }
  return pixmap.loadFromData(data);
}


int ThemeProperty::getPosition(const char* tag) const {
  QMap<QString, QString>& tag_map = pimpl->xml_handler_.tag_map_;
  QMap<QString, QString>::iterator p = tag_map.find(tag);
  if (p == tag_map.end()) {
    return -1;
  }
  QString position_str = tag_map.value(tag);
  int value = position_str.toUInt();

  return value;
}


void ThemeProperty::close(void) {
  pimpl->close();
}
