/*!
  \file
  \brief テーマファイル作成用のメインウィンドウ管理

  \author Satofumi KAMIMURA

  $Id$
*/

#include <QHeaderView>
#include <QCloseEvent>
#include <QFileDialog>
#include <QGraphicsPixmapItem>
#include <QShortcut>
#include <cmath>
#include "ThemeConfigWindow.h"
#include "ArchiveConfig.h"
#include "SettingConfig.h"
#include "ReleaseDialog.h"
#include "ReleaseDataFile.h"


namespace {
  typedef QMap<QTreeWidgetItem*, std::string> ItemToName;
  typedef QMap<std::string, QTreeWidgetItem*> NameToItem;

  typedef QMap<std::string, QPixmap*> Pixmaps;
  typedef QMap<std::string, QIcon*> Icons;
  typedef QMap<std::string, QGraphicsItem*> Items;

  typedef QMap<std::string, QPoint> PositionMap;

  typedef struct {
    const char* name;
    const char* x_tag;
    const char* y_tag;
  } position_tag_t;
};


static const char* ImageTags[] = {
  "base/base_org",
  "analog/hari_h", "analog/hari_m", "analog/hari_s",
  "digital/parts",
};


static const position_tag_t PositionTags[] = {
  { "analog/hari_h", "analog/h_xpos", "analog/h_ypos", },
  { "analog/hari_m", "analog/m_xpos", "analog/m_ypos", },
  { "analog/hari_s", "analog/s_xpos", "analog/s_ypos", },

  { "digital/h1", "digital/h1_xpos", "digital/h1_ypos", },
  { "digital/h2", "digital/h2_xpos", "digital/h2_ypos", },
  { "digital/m1", "digital/m1_xpos", "digital/m1_ypos", },
  { "digital/m2", "digital/m2_xpos", "digital/m2_ypos", },
  { "digital/s1", "digital/s1_xpos", "digital/s1_ypos", },
  { "digital/s2", "digital/s2_xpos", "digital/s2_ypos", },
  { "digital/cl", "digital/cl_xpos", "digital/cl_ypos", },
  { "digital/ap", "digital/ap_xpos", "digital/ap_ypos", },
};



struct ThemeConfigWindow::pImpl {

  QAction* new_act_;
  QAction* open_act_;
  QAction* save_act_;

  ArchiveConfig* archive_config_;
  SettingConfig* setting_config_;

  bool save_ok_;
  QString first_title_;

  bool same_hands_center_;

  std::string current_tag_;

  ItemToName item_to_name_;
  NameToItem name_to_item_;

  Pixmaps pixmaps_;
  Icons icons_;

  QGraphicsScene* scene_;
  Items items_;
  std::vector<QPixmap> digital_parts_;

  PositionMap parts_positions_;

  ReleaseDialog release_dialog_;

  pImpl(void)
    : new_act_(NULL), open_act_(NULL), save_act_(NULL),
      archive_config_(NULL), setting_config_(NULL), save_ok_(false),
      same_hands_center_(false), scene_(NULL) {
  }


  // フォームの初期化
  void initializeForm(ThemeConfigWindow* parent) {
    static_cast<void>(parent);

    // Window Title の取得
    first_title_ = parent->windowTitle();

    // 描画ウィンドウの設定
    initializeGraphicsView(parent);

    // ツリー表示の幅を調整
    parent->parts_tree_->header()->setResizeMode(0, QHeaderView::Stretch);
    parent->parts_tree_->header()->setResizeMode(1, QHeaderView::Stretch);

    // チェックボックス
    connect(parent->center_checkbox_, SIGNAL(clicked(bool)),
            parent, SLOT(centerCheckHandler(bool)));
    connect(parent->blink_checkbox_, SIGNAL(clicked(bool)),
            parent, SLOT(blinkCheckHandler(bool)));
    connect(parent->analog_checkbox_, SIGNAL(clicked(bool)),
            parent, SLOT(analogCheckHandler(bool)));
    connect(parent->digital_checkbox_, SIGNAL(clicked(bool)),
            parent, SLOT(digitalCheckHandler(bool)));
    same_hands_center_ = parent->center_checkbox_->isChecked();

    // メニューイベントの接続
    connect(parent->action_exit_, SIGNAL(triggered()), parent, SLOT(close()));
    connect(parent->action_exit_, SIGNAL(triggered()),
            &release_dialog_, SLOT(close()));

    // カーソルによる、選択画像の移動
    (void) new QShortcut(Qt::Key_Left, parent, SLOT(leftPressed()));
    (void) new QShortcut(Qt::Key_Right, parent, SLOT(rightPressed()));
    (void) new QShortcut(Qt::Key_Up, parent, SLOT(upPressed()));
    (void) new QShortcut(Qt::Key_Down, parent, SLOT(downPressed()));

    // 部品画像の更新
    connect(parent->image_lineedit_, SIGNAL(returnPressed()),
            parent, SLOT(partsEditHandler()));
    connect(parent->update_button_, SIGNAL(clicked()),
            parent, SLOT(updateHandlelr()));

    // 部品表示ツリー
    connect(parent->parts_tree_, SIGNAL(itemSelectionChanged()),
            parent, SLOT(imageUploadHandler()));

    // リリース
    connect(&release_dialog_, SIGNAL(release(QString, QString)),
            parent, SLOT(releaseHandler(QString, QString)));

    // 再生、停止
    // !!! イベント接続
  }


  // 描画ウィンドウの更新
  void initializeGraphicsView(ThemeConfigWindow* parent) {

    scene_ = new QGraphicsScene(parent);
    parent->parts_view_->setScene(scene_);
  }


  // ツリー情報ポインタの格納
  void treeTagInitialize(ThemeConfigWindow* parent) {

    // ベース画像
    QTreeWidgetItem* base = parent->parts_tree_->itemAt(0, 0);
    registerTreeItem("base/base_org", base);

    // アナログ部品
    QTreeWidgetItem* analog = parent->parts_tree_->itemBelow(base);
    registerTreeItem("analog", analog);

    // 短針の画像
    QTreeWidgetItem* short_hand = analog->child(0);
    registerTreeItem("analog/hari_h", short_hand);

    // 長針の画像
    QTreeWidgetItem* long_hand = analog->child(1);
    registerTreeItem("analog/hari_m", long_hand);

    // 秒針の画像
    QTreeWidgetItem* second_hand = analog->child(2);
    registerTreeItem("analog/hari_s", second_hand);

    // デジタル文字
    QTreeWidgetItem* digital = parent->parts_tree_->itemBelow(analog);
    registerTreeItem("digital/parts", digital);

    // h1
    QTreeWidgetItem* h1 = digital->child(0);
    registerTreeItem("digital/h1", h1);

    // h2
    QTreeWidgetItem* h2 = h1->child(0);
    registerTreeItem("digital/h2", h2);

    // colon
    QTreeWidgetItem* colon = digital->child(1);
    registerTreeItem("digital/cl", colon);

    // m1
    QTreeWidgetItem* m1 = digital->child(2);
    registerTreeItem("digital/m1", m1);

    // m2
    QTreeWidgetItem* m2 = m1->child(0);
    registerTreeItem("digital/m2", m2);

    // s1
    QTreeWidgetItem* s1 = digital->child(3);
    registerTreeItem("digital/s1", s1);

    // s2
    QTreeWidgetItem* s2 = s1->child(0);
    registerTreeItem("digital/s2", s2);

    // am/mp
    QTreeWidgetItem* ap = digital->child(4);
    registerTreeItem("digital/ap", ap);
  }


  // 部品名とポインタの対応を登録
  void registerTreeItem(const std::string& tag, QTreeWidgetItem* item) {

    item_to_name_.insert(item, tag);
    name_to_item_.insert(tag, item);
  }


  // アナログ部品のノードを開く
  void expandAnalogNode(ThemeConfigWindow* parent) {

    QTreeWidgetItem* base = parent->parts_tree_->itemAt(0, 0);
    QTreeWidgetItem* analog = parent->parts_tree_->itemBelow(base);
    parent->parts_tree_->expandItem(analog);
  }


  // enable / disable の設定
  void setActionsEnabled(ThemeConfigWindow* parent, bool enable) {

    // 保存
    setSaveEnabled(parent, enable);

    // メニュー、ツールボタン
    parent->action_save_as_->setEnabled(enable);
    parent->action_release_->setEnabled(enable);
    parent->action_preview_->setEnabled(enable);

    // 再生ボタン
    parent->play_button_->setEnabled(enable);
    parent->offset_slidebar_->setEnabled(enable);

    // 編集用 Widget
    parent->groupbox_->setEnabled(enable);
    parent->parts_tree_->setEnabled(enable);
  }


  // 保存機能の enable / disable 設定
  void setSaveEnabled(ThemeConfigWindow* parent, bool enable) {

    parent->action_save_->setEnabled(enable);
    save_act_->setEnabled(enable);
  }


  // 画像の更新機能の enable / disable
  void setUpdateEnabled(ThemeConfigWindow* parent, bool enable) {

    parent->image_label_->setEnabled(enable);
    parent->image_lineedit_->setEnabled(enable);
    parent->update_button_->setEnabled(enable);
  }


  // 設定ファイルオブジェクトの生成
  void loadArchiveFile(ThemeConfigWindow* parent, const char* name) {

    ArchiveConfig* archive_config = new ArchiveConfig(name);
    std::swap(archive_config, archive_config_);
    delete archive_config;

    const char* xml_file = archive_config_->getNormalData("config");
    const char* adjusted_xml_file = (strlen(xml_file) > 0) ? xml_file : "";
    SettingConfig* setting_config = new SettingConfig(adjusted_xml_file);
    std::swap(setting_config, setting_config_);
    delete setting_config;

    load(parent);
  }


  // 読み出し
  void load(ThemeConfigWindow* parent) {

    // 設定を読み出して反映する
    loadSettings(parent);

    // 保存可能、リリース可能にする
    setActionsEnabled(parent, true);

    // 現在のファイル名をタイトルバーに表示する
    setTitleToText(parent, archive_config_->getArchiveFileName());
  }


  // 設定の読み出し
  bool loadSettings(ThemeConfigWindow* parent) {

    // 部品位置の反映
    loadPosition();

    // チェックボックスへの反映
    // !!! "針の中心をそろえる" は、設定ファイルを作ったときに管理する
    // !!! または、データの中心が同じならば、チェックを有効にする
    // !!! 画像サイズがわからないと、確認できないが...

    // 秒針の 表示 / 非表示
    bool analog_sec = ! strcmp("on", setting_config_->getValue("analog/sec"));
    parent->analog_checkbox_->setChecked(analog_sec);

    // デジタル秒の 表示 / 非表示
    bool digital_sec = ! strcmp("on", setting_config_->getValue("digital/sec"));
    parent->digital_checkbox_->setChecked(digital_sec);

    // 部品画像のツリーへの設定
    loadImages();

    return true;
  }


  // 画像の読み出し (全更新)
  void loadImages(void) {

    size_t n = sizeof(ImageTags) / sizeof(ImageTags[0]);
    for (size_t i = 0; i < n; ++i) {
      loadImage(ImageTags[i]);
    }

    // 描画アイテムの再構築
    updateItems();
  }


  // 画像の読み出し
  void loadImage(const char* tag) {

    const char* archive_tag = setting_config_->getValue(tag);
    const char* name = archive_config_->getNormalData(archive_tag);

    if (strlen(name) <= 0) {
      return;
    }

    // 現在の画像を破棄し、新規に画像を読み込む
    delete pixmaps_.value(tag, NULL);
    delete icons_.value(tag, NULL);

    QPixmap* pixmap = new QPixmap(name);
    pixmaps_[tag] = pixmap;

    QIcon* icon = new QIcon(*pixmap);
    icons_[tag] = icon;

    // ツリーアイテムの更新
    name_to_item_[tag]->setIcon(1, *icon);
    name_to_item_[tag]->setText(1, name);

    // デジタル画像の場合、数値部品を更新する
    if (! strcmp(tag, "digital/parts")) {
      updateDigitalParts(*pixmap);
    }
  }

  // 数値部品の更新
  void updateDigitalParts(const QPixmap& pixmap) {

    // 文字毎に等間隔で分割して登録する
    const char* tag_names[] = {
      "0", "1", "2", "3", "4", "5", "6", "7", "8", "9",
      " ", ":", "A", "P",
    };

    // 部品の更新
    digital_parts_.clear();
    QSize size = pixmap.size();
    size_t n = sizeof(tag_names) / sizeof(tag_names[0]);
    for (size_t i = 0; i < n; ++i) {
      digital_parts_.push_back(pixmap.copy(size.width() * i / n,
                                           0, size.width() / n, size.height()));
    }

    // 登録した vector のアドレスを確保
    for (size_t i = 0; i < n; ++i) {
      pixmaps_.insert(tag_names[i], &digital_parts_[i]);
    }

    // 部品の座標位置が全て (0, 0) ならば、配置を初期化する
    // !!!
  }


  // 描画アイテムの再構築
  void updateItems(void) {

    // デジタル部品の位置が全て (0, 0) のときに、デフォルトの配置を行う
    if (isDigitalZeroPosition()) {
      updateDefaultDigitalPosition();
    }

    // 全アイテムの除去
    scene_->clear();
    items_.clear();

    // ベース
    QGraphicsItem* parent = NULL;
    parent = registerItem(parent, "base/base_org");

    // デジタル部品
    registerItem(parent, "digital/h1", "0");
    registerItem(parent, "digital/h2", "1");
    registerItem(parent, "digital/m1", "2");
    registerItem(parent, "digital/m2", "3");

    if (! strcmp("on", setting_config_->getValue("digital/sec"))) {
      registerItem(parent, "digital/s1", "4");
      registerItem(parent, "digital/s2", "5");
    }

    registerItem(parent, "digital/cl", ":");
    registerItem(parent, "digital/ap", "A");

    // アナログ部品
    QGraphicsItem* short_hand = registerItem(parent, "analog/hari_h");
    if (same_hands_center_ && short_hand) {
      // 針の中心位置を、短針に揃える
      parts_positions_["analog/hari_m"] = parts_positions_["analog/hari_h"];
      parts_positions_["analog/hari_s"] = parts_positions_["analog/hari_h"];
    }
    registerItem(parent, "analog/hari_m");
    if (! strcmp("on", setting_config_->getValue("analog/sec"))) {
      registerItem(parent, "analog/hari_s");
    }
  }


  // パーツの配置
  QGraphicsItem* registerItem(QGraphicsItem* parent, const char* tag,
                              const char* pixmap_tag = NULL) {

    QGraphicsPixmapItem* item = NULL;
    const char* use_tag = (pixmap_tag == NULL) ? tag : pixmap_tag;

    QPixmap* pixmap = pixmaps_.value(use_tag, NULL);
    if (pixmap) {
      item = new QGraphicsPixmapItem(*pixmap, parent);

      // 指定位置に配置。ただし、アナログ針は中心位置を管理している
      QPoint offset;
      size_t n = strlen("analog");
      if (! strncmp("analog", tag, n)) {
        QSize size = pixmap->size();
        offset = QPoint(size.width() / 2, size.height() / 2);
      }
      item->setPos(parts_positions_[tag] - item->scenePos() - offset);

      item->setTransformationMode(Qt::SmoothTransformation);
      items_.insert(tag, item);
    }
    if (item && (! parent)) {
      scene_->addItem(item);
    }

    return item;
  }


  // デジタル部品の位置が、全て同じかを調べる
  bool isDigitalZeroPosition(void) {

    const char* digital_tags[] = {
      "digital/h1", "digital/h2",
      "digital/m1", "digital/m2",
      "digital/s1", "digital/s2",
      "digital/cl", "digital/ap",
    };
    size_t n = sizeof(digital_tags) / sizeof(digital_tags[0]);
    for (size_t i = 0; i < n; ++i) {
      QPointF position = parts_positions_[digital_tags[i]];
      if ((fabs(position.x()) > 0.1) || (fabs(position.y()) > 0.1)) {
        return false;
      }
    }
    return true;
  }


  void updateDefaultDigitalPosition(void) {

    QPixmap* pixmap = pixmaps_.value("0", NULL);
    if (! pixmap) {
      return;
    }
    size_t width = pixmap->size().width();

    parts_positions_["digital/ap"] = QPoint(0, 0);
    parts_positions_["digital/h1"] = QPoint(width * 1, 0);
    parts_positions_["digital/h2"] = QPoint(width * 2, 0);
    parts_positions_["digital/cl"] = QPoint(width * 3, 0);
    parts_positions_["digital/m1"] = QPoint(width * 4, 0);
    parts_positions_["digital/m2"] = QPoint(width * 5, 0);
    parts_positions_["digital/s1"] = QPoint(width * 7, 0);
    parts_positions_["digital/s2"] = QPoint(width * 8, 0);
  }


  // 名前を付けて保存
  void save(ThemeConfigWindow* parent, const char* name) {

    if (archive_config_->saveAs(name)) {

      // 位置の反映
      savePosition();

      // XML 設定の保存
      setting_config_->save();

      // 変更があるまで、保存を disable にする
      setSaveEnabled(parent, false);

      // 現在のファイル名をタイトルバーに表示する
      setTitleToText(parent, name);
    }
  }


  // ウィンドウタイトルの更新
  void setTitleToText(ThemeConfigWindow* parent, const char* name) {

    QFileInfo fi(name);
    QString fileName = fi.fileName();
    if (! fileName.isEmpty()) {
      parent->setWindowTitle(first_title_ + " -- " + fileName);
    } else {
      parent->setWindowTitle(first_title_);
    }
  }


  // アイテムのタグ文字を探す
  bool findItemTag(const char* tags[], size_t n, const std::string& tag) {

    for (size_t i = 0; i < n; ++i) {

      if (! tag.compare(tags[i])) {
        return true;
      }
    }
    return false;
  }


  // 位置のタグ文字を探す
  // !!! findItemTagString() と共通化できないかな...
  bool findPositionTag(position_tag_t tags[],
                       size_t n, const std::string& tag) {

    for (size_t i = 0; i < n; ++i) {

      if (! tag.compare(tags[i].name)) {
        return true;
      }
    }
    return false;
  }


  void loadPosition(void) {

    // setting_config_ の位置を描画用の位置に反映させる
    size_t n = sizeof(PositionTags) / sizeof(PositionTags[0]);
    for (size_t i = 0; i < n; ++i) {
      const position_tag_t& tag = PositionTags[i];
      int x = atoi(setting_config_->getValue(tag.x_tag));
      int y = atoi(setting_config_->getValue(tag.y_tag));
      parts_positions_[tag.name] = QPoint(x, y);
    }
  }


  void savePosition(void) {

    // 描画用の位置を setting_config_ に反映させる
    size_t n = sizeof(PositionTags) / sizeof(PositionTags[0]);
    for (size_t i = 0; i < n; ++i) {
      const position_tag_t& tag = PositionTags[i];

      QPoint& point = parts_positions_[tag.name];

      QString x = QString("%1").arg(point.x());
      QString y = QString("%1").arg(point.y());

      setting_config_->setValue(tag.x_tag, x.toStdString().c_str());
      setting_config_->setValue(tag.y_tag, y.toStdString().c_str());
    }
  }


  // 位置を更新
  void updatePosition(ThemeConfigWindow* parent,
                      int x, int y, const std::string& tag) {

    // "base/base_org" は移動させない
    if (! tag.compare("base/base_org")) {
      return;
    }

    // 各針の中心を揃えているときは、"analog/hand_X" を全て移動させる
    if (same_hands_center_ &&
        ((! tag.compare("analog/hari_h")) ||
         (! tag.compare("analog/hari_m")) ||
         (! tag.compare("analog/hari_s")))) {
      same_hands_center_ = false;

      updatePosition(parent, x, y, "analog/hari_h");
      updatePosition(parent, x, y, "analog/hari_m");
      updatePosition(parent, x, y, "analog/hari_s");

      same_hands_center_ = true;
      return;
    }

    // アナログ部品をまとめて操作させる
    if (! tag.compare("analog")) {
      bool pre_flag = same_hands_center_;
      same_hands_center_ = false;

      updatePosition(parent, x, y, "analog/hari_h");
      updatePosition(parent, x, y, "analog/hari_m");
      updatePosition(parent, x, y, "analog/hari_s");

      same_hands_center_ = pre_flag;
      return;
    }

    // "digital/parts" タグのときは、"digital/XX" を全て移動させる
    if (! tag.compare("digital/parts")) {
      updatePosition(parent, x, y, "digital/h1");
      updatePosition(parent, x, y, "digital/m1");
      updatePosition(parent, x, y, "digital/s1");
      updatePosition(parent, x, y, "digital/cl");
      updatePosition(parent, x, y, "digital/ap");
    }

    // "digital/X1" のタグのときは、"digital/X2" も移動させる
    if (! tag.compare("digital/h1")) {
      updatePosition(parent, x, y, "digital/h2");
    }
    if (! tag.compare("digital/m1")) {
      updatePosition(parent, x, y, "digital/m2");
    }
    if (! tag.compare("digital/s1")) {
      updatePosition(parent, x, y, "digital/s2");
    }

    PositionMap::iterator p = parts_positions_.find(tag);
    if (p == parts_positions_.end()) {
      // 見つからなければ、戻る
      return;
    }

    QGraphicsItem* item = items_.value(tag, NULL);
    if (! item) {
      // !!! ありえないが、一応
      return;
    }

    // 管理位置の更新
    parts_positions_[tag] += QPoint(x, y);

    // 描画位置の更新
    item->setPos(item->pos() + QPoint(x, y));

    setSaveEnabled(parent, true);
  }


  // リリースファイルの作成
  void release(void) {

    releaseDataFile(*archive_config_);
  }


  // 登録していある名前を変更し、画像を更新する
  void updateItemName(const QString& fileName) {

    const char* archive_tag = setting_config_->getValue(current_tag_.c_str());
    archive_config_->setNormalData(archive_tag, fileName.toStdString().c_str());

    loadImage(current_tag_.c_str());
    updateItems();
  }
};


ThemeConfigWindow::ThemeConfigWindow(const QString& config_file)
  : pimpl(new pImpl) {
  setupUi(this);

  // ツールバーの初期化
  pimpl->new_act_ = new QAction(QIcon(":icons/new_icon"), tr("New"), this);
  pimpl->new_act_->setStatusTip(tr("Create a new config file."));
  connect(pimpl->new_act_, SIGNAL(triggered()), this, SLOT(newFile()));

  pimpl->open_act_ = new QAction(QIcon(":icons/open_icon"), tr("Open"), this);
  pimpl->open_act_->setStatusTip(tr("Open an exist config file."));
  connect(pimpl->open_act_, SIGNAL(triggered()), this, SLOT(openFile()));

  pimpl->save_act_ = new QAction(QIcon(":icons/save_icon"), tr("Save"), this);
  pimpl->save_act_->setStatusTip(tr("Save the config file to disk."));
  connect(pimpl->save_act_, SIGNAL(triggered()), this, SLOT(saveFile()));

  toolbar_->addAction(pimpl->new_act_);
  toolbar_->addAction(pimpl->open_act_);
  toolbar_->addAction(pimpl->save_act_);

  // メニューの書記化
  connect(action_new_, SIGNAL(triggered()), this, SLOT(newFile()));
  connect(action_open_, SIGNAL(triggered()), this, SLOT(openFile()));
  connect(action_save_, SIGNAL(triggered()), this, SLOT(saveFile()));
  connect(action_save_as_, SIGNAL(triggered()), this, SLOT(saveFileAs()));
  connect(action_release_, SIGNAL(triggered()), this, SLOT(releaseData()));
  connect(action_preview_, SIGNAL(triggered()), this, SLOT(previewClock()));
  connect(action_about_, SIGNAL(triggered()), this, SLOT(aboutApplication()));

  // 初期状態では、保存不可、リリース不可にしておく
  pimpl->setActionsEnabled(this, false);

  // 画像の更新 Widget は、ツリーがクリックされるまでは、disable
  pimpl->setUpdateEnabled(this, false);

  // フォームの初期化
  pimpl->initializeForm(this);

  // ツリー構成の取得
  pimpl->treeTagInitialize(this);

  // ファイルが指定されていれば、読み出し
  if (! config_file.isEmpty()) {
    pimpl->loadArchiveFile(this, config_file.toStdString().c_str());
    pimpl->setSaveEnabled(this, false);
  }

  // アナログ画像のサブツリーを開く
  pimpl->expandAnalogNode(this);
}


ThemeConfigWindow::~ThemeConfigWindow(void) {
}


void ThemeConfigWindow::closeEvent(QCloseEvent* event) {

  // 保存の必要があれば、ダイアログを表示して質問する
  // !!! save が enable であれば、保存可能とみなしてよい

  // !!! Qt 本の P66 のあたりを参考のこと

  // !!! とりあえず、終了
  event->accept();
}


// 新規
void ThemeConfigWindow::newFile(void) {

  // 空の名前でファイル読み出しを指示し、設定が空のオブジェクトを作成する
  pimpl->loadArchiveFile(this, "");
}


// 開く
void ThemeConfigWindow::openFile(void) {

  QString fileName =
    QFileDialog::getOpenFileName(this, tr("Open archive list file."), ".",
                                 tr("archive list (*.txt)"));

  if (! fileName.isEmpty()) {
    pimpl->loadArchiveFile(this, fileName.toStdString().c_str());
    pimpl->setSaveEnabled(this, false);
  }
}


// 保存
void ThemeConfigWindow::saveFile(void) {

  if (! pimpl->archive_config_) {
    return;
  }

  const char* archive_name = pimpl->archive_config_->getArchiveFileName();
  if (strlen(archive_name) <= 0) {
    // 名前がなければ、名前を付けて保存
    saveFileAs();

  } else {
    pimpl->save(this, archive_name);
  }
}


// 名前を付けて保存
void ThemeConfigWindow::saveFileAs(void) {

  QString fileName =
    QFileDialog::getSaveFileName(this, tr("Save archive list file."), ".",
                                 tr("archive list (*.txt)"));

  if (! fileName.isEmpty()) {
    pimpl->save(this, fileName.toStdString().c_str());
  }
}


// リリース
void ThemeConfigWindow::releaseData(void) {

  // リリース名、キー値の設定
  const char* name = pimpl->archive_config_->getReleaseName();
  const char* key = pimpl->archive_config_->getRrdaKey();
  pimpl->release_dialog_.setDefaultValue(name, key);

  // ダイアログの表示
  pimpl->release_dialog_.show();
  pimpl->release_dialog_.raise();
}


// リリースの実処理
void ThemeConfigWindow::releaseHandler(const QString& file,
                                       const QString& key) {

  // 登録情報の更新
  pimpl->archive_config_->setReleaseName(file.toStdString().c_str());
  pimpl->archive_config_->setRrdaKey(key.toStdString().c_str());

  pimpl->release();
}


// プレビュー
void ThemeConfigWindow::previewClock(void) {

  // !!!

  // !!! ThemeClockWidget を呼び出して表示
  // !!! 表示をどうやって消すのかは、不明
}


// このアプリケーションについて
void ThemeConfigWindow::aboutApplication(void) {

  // !!!

  // !!! Qt 本を参考にして、適当に実装する

  // !!! これは何か
  // !!! 作者、ライセンス、

  // !!! とか
}


// 針の中心を、揃えるか、の設定
void ThemeConfigWindow::centerCheckHandler(bool checked) {

  pimpl->same_hands_center_ = checked;
  pimpl->updateItems();
  pimpl->setSaveEnabled(this, true);
}


// コロンを点滅させるか、の設定
void ThemeConfigWindow::blinkCheckHandler(bool checked) {

  pimpl->setting_config_->setValue("digital/colon", checked ? "on" : "off");
  pimpl->setSaveEnabled(this, true);
}


// 秒針を表示するか、の設定
void ThemeConfigWindow::analogCheckHandler(bool checked) {

  pimpl->setting_config_->setValue("analog/sec", checked ? "on" : "off");
  pimpl->updateItems();
  pimpl->setSaveEnabled(this, true);
}


// 秒針を表示するか、の設定
void ThemeConfigWindow::digitalCheckHandler(bool checked) {

  pimpl->setting_config_->setValue("digital/sec", checked ? "on" : "off");
  pimpl->updateItems();
  pimpl->setSaveEnabled(this, true);
}


// 画像更新用のイベント処理
void ThemeConfigWindow::imageUploadHandler(void) {

  QList<QTreeWidgetItem*> list = parts_tree_->selectedItems();
  if (list.empty() || (list.size() > 1)) {
    // 選択が１つでなければ、戻る
    // !!! 同時に操作できるようにするあたりは、また今度検討する
    return;
  }

  QTreeWidgetItem* item = list.front();
  ItemToName::iterator p = pimpl->item_to_name_.find(item);
  if (p == pimpl->item_to_name_.end()) {
    // タグが見つからなければ戻る
    return;
  }
  std::string tag = p.value();

  // カーソルのイベントハンドラが、現在のタグに従って処理を行う
  pimpl->current_tag_ = tag;

  size_t n = sizeof(ImageTags) / sizeof(ImageTags[0]);
  bool found = pimpl->findItemTag(ImageTags, n, pimpl->current_tag_);
  if (! found) {
    // 画像の以外ならば、アップロードフォームを無効化
    image_lineedit_->setText("");
    pimpl->setUpdateEnabled(this, false);

  } else {
    // 画像のアップロードフォームを有効化
    image_lineedit_->setText(pimpl->name_to_item_[tag]->text(1));
    pimpl->setUpdateEnabled(this, true);
  }
}


void ThemeConfigWindow::leftPressed(void) {

  pimpl->updatePosition(this, -1, 0, pimpl->current_tag_);
}


void ThemeConfigWindow::rightPressed(void) {

  pimpl->updatePosition(this, +1, 0, pimpl->current_tag_);
}


void ThemeConfigWindow::upPressed(void) {

  pimpl->updatePosition(this, 0, -1, pimpl->current_tag_);
}


void ThemeConfigWindow::downPressed(void) {

  pimpl->updatePosition(this, 0, +1, pimpl->current_tag_);
}



// 部品画像の更新、文字列入力
void ThemeConfigWindow::partsEditHandler(void) {

  pimpl->updateItemName(image_lineedit_->text());
}


// 部品画像の更新、選択
void ThemeConfigWindow::updateHandlelr(void) {

  QString fileName =
    QFileDialog::getOpenFileName(this, tr("Open parts file."), ".",
                                 tr("parts file (*.png)"));

  if (! fileName.isEmpty()) {
    pimpl->updateItemName(fileName.toStdString().c_str());
  }
}


// !!! slot
// !!! ドラッグ & ドロップで画像ファイルを選択できるようにする
// !!! 画像でなければ、実際には登録しない


// 選択中の部品移動を行う
// !!! slot


// 編集中に再生
// !!! slot


// 再生のオフセット時間を変更
// !!!


// !!! Qt Demo の 画像のパズルを参考に、部品配置を変更するあたりを実装する
