/*!
  \file
  \brief RRDA のリソース管理

  gz_header_check(), gunzip_data() は、lm_tmclock/src/lm_rrda.c より抜粋

  \author Satofumi KAMIMURA

  $Id$
*/

#include <fstream>
#include <zlib.h>
#include <vector>
#include "RrdaResource.h"


namespace {
  typedef std::vector<unsigned char> Buffer;
};


struct RrdaResource::pImpl {
  enum {
    Reserved = 0xe0,
    ExtraField = 0x04,
    OriginalName = 0x08,
    Comment = 0x10,
    HeadCrc = 0x02,

    ZlibBufferSize = 1024,          // 解凍時の出力チャンク単位
    ZlibOutputMax = 1024 * 1024 * 5 // 解凍時の1ファイルのサイズ制限(安全策)
  };

  unsigned char* raw_memory_;   // 圧縮ファイルの格納メモリ
  size_t raw_memory_size_;

  Buffer extended_memory_;      // 展開後のリソース格納メモリ
  size_t size_;


  pImpl(std::ifstream& fin, size_t size)
    : raw_memory_(NULL), raw_memory_size_(size), size_(0) {

    if (raw_memory_size_ > 0) {
      raw_memory_ = new unsigned char[raw_memory_size_];
    }

    // ファイル内容の読み出し
    fin.read(reinterpret_cast<char*>(raw_memory_), raw_memory_size_);
  }


  ~pImpl(void) {
    releaseTemporaryMemory();
  }


  // 一時リソースの解放
  void releaseTemporaryMemory(void) {
    if (raw_memory_) {
      delete [] raw_memory_;
      raw_memory_ = NULL;
    }
    raw_memory_size_ = 0;
  }


  // リソースの取得
  unsigned char* get(size_t* size) {
    if (extended_memory_.empty()) {
      decompress(raw_memory_, raw_memory_size_);
    }

    if (size) {
      *size = size_;
    }
    return &extended_memory_[0];
  }


  // リソースの解凍
  bool decompress(const unsigned char* data_first, size_t area_size) {

    const unsigned char* work = gz_header_check(data_first, area_size);
    if (work == NULL) {
      // zip ヘッダがない場合
      work = data_first;
    }
    long zbody_size = area_size - static_cast<long>(work - data_first);
    if (! gunzip_data(work, zbody_size)) {
      return false;
    }

    releaseTemporaryMemory();
    return true;
  }


  // gz データのヘッダーチェックとスキップ
  // zlib の gzio.c 内ローカル関数 check_header() を参考にリライトしたもの
  const unsigned char* gz_header_check(const unsigned char* data, size_t size) {

    const unsigned char* limit = data + size;

    // マジックナンバーのチェック
    if ((data[0] != 0x1f) || (data[1] != 0x8b)) {
      // Error キーが違うかデータが壊れている
      return NULL;
    }
    data += 2;

    // ファイルメソッドのチェック (Z_DEFLATED は zlib.h 内を参照)
    unsigned char flags = data[1];
    if ((data[0] != Z_DEFLATED) || ((flags & Reserved) != 0)) {
      // Error キーが違うかデータが壊れている
      return NULL;
    }

    // time 等の情報もまとめてスキップ
    data += 8;

    // extra フィールドのスキップ
    if ((flags & ExtraField) != 0) {
      unsigned long l =
        (((static_cast<unsigned long>(data[1])) << 8) & ~0xff) |
        static_cast<unsigned long>(data[0]);
      data += (l + 2);
    }

    // オリジナルファイルネームのスキップ
    if ((flags & OriginalName) != 0) {
      while (*data != 0) {
        ++data;
        if (data > limit) {
          return NULL;
        }
      }
      ++data;
    }

    // ファイルコメントのスキップ
    if ((flags & Comment) != 0) {
      while (*data != 0) {
        ++data;
        if (data > limit) {
          return NULL;
        }
      }
      ++data;
    }

    // ヘッダーCRC
    if ((flags & HeadCrc) != 0) {
      data += 2;
    }

    // ポインタ溢れチェック
    if (data > limit) {
      return NULL;
    }

    return data;
  }


  // メモリ上の gz データを解凍し、ポインタを返す
  // ヘッダーは既にスキップされて圧縮データ本体へのポインタが渡されているとする
  bool gunzip_data(const unsigned char* data, long size) {

    // メモリ管理の指定
    z_stream z;
    z.zalloc = Z_NULL;
    z.zfree = Z_NULL;
    z.opaque = Z_NULL;

    // zlib 初期化
    z.next_in = Z_NULL;
    z.avail_in = 0;
    if (inflateInit2(&z, -MAX_WBITS) != Z_OK) {
      return false;
    }

    // 展開結果バッファ
    Buffer decompress_buffer;
    decompress_buffer.resize(ZlibBufferSize);

    // 解凍データ
    z.avail_in = size;
    z.next_in = const_cast<unsigned char*>(data);
    z.avail_out = decompress_buffer.size();
    z.next_out = &decompress_buffer[0];
    int status = Z_OK;

    // 解凍処理
    static Buffer output_buffer;
    size_t total_size = 0;
    unsigned long outptr = 0;
    while ((status != Z_STREAM_END) && (outptr < ZlibOutputMax)) {

      // zlib call
      status = inflate(&z, Z_NO_FLUSH);
      size_t additional_size = ZlibBufferSize - z.avail_out;
      total_size += additional_size;

      // inflate error
      if ((status != Z_OK) && (status != Z_STREAM_END)) {
        return false;
      }
      // 解凍データの追加保持
      if (output_buffer.empty()) {
        output_buffer.resize(ZlibBufferSize);

        memcpy(&output_buffer[0], &decompress_buffer[0], ZlibBufferSize);
        outptr = ZlibBufferSize;
        z.avail_out = ZlibBufferSize;
        z.next_out = &decompress_buffer[0];

      } else {
        output_buffer.resize(total_size);

        memcpy(&output_buffer[0] + outptr,
               &decompress_buffer[0], additional_size);
        outptr += additional_size;
        z.avail_out = ZlibBufferSize;
        z.next_out = &decompress_buffer[0];
      }
    }

    // zlib 後片づけ
    if (inflateEnd(&z) != Z_OK) {
      return false;
    }

    size_ = total_size;
    output_buffer.resize(total_size);
    std::swap(output_buffer, extended_memory_);
    return true;
  }
};


RrdaResource::RrdaResource(std::ifstream& fin, size_t size)
  : pimpl(new pImpl(fin, size)) {
}


RrdaResource::~RrdaResource(void) {
}


// リソースを返す
unsigned char* RrdaResource::get(size_t* size) {
  return pimpl->get(size);
}
