/*!
  \file
  \brief テーマ時計の部品描画

  \author Satofumi KAMIMURA

  $Id$

  \todo 拡大縮小を毎回しているのを、なんとかする
*/

#include <QPainter>
#include <QBitmap>
#include <QTime>
#include <cmath>
#include "ClockPartsDraw.h"
#include "ThemeResource.h"
#include "ThemeClockDraw.h"


namespace {
  static const bool CenterPosition = true; // 画像中心を基準にした描画を指示

  // 位置の取得
  QRect getPosition(const ThemeResource& resource,
                    const std::string& tag_name) {

    return resource.offset.value(tag_name, QRect(0, 0, 0, 0));
  }


  // 画像の描画
  void drawPixmap(QPainter& painter, QRegion& region,
                  const ThemeResource& resource,
                  const std::string& tag_name,
                  const QRect& offset,
                  size_t scaling_percent,
                  qreal rotate_degree = 0,
                  bool center_position = false) {

    QPixmap* pixmap = resource.pixmap.value(tag_name, NULL);
    if (! pixmap) {
      return;
    }
    QMatrix matrix;
    matrix = matrix.rotate(rotate_degree);
    QPixmap rotated_pixmap =
      pixmap->transformed(matrix, Qt::SmoothTransformation);
    if (rotated_pixmap.isNull()) {
      return;
    }

    QSize size = rotated_pixmap.size();
    int scaled_x =
      static_cast<int>(round(size.width() * scaling_percent / 100.0));
    int scaled_y =
      static_cast<int>(round(size.height() * scaling_percent / 100.0));
    QPixmap scaled_pixmap =
      rotated_pixmap.scaled(scaled_x, scaled_y,
                            Qt::IgnoreAspectRatio, Qt::SmoothTransformation);

    int x, y;
    if (center_position) {
      x = static_cast<int>((-round(size.width() / 2.0) + offset.x())
                               * scaling_percent / 100.0);
      y = static_cast<int>((-round(size.height() / 2.0) + offset.y())
                               * scaling_percent / 100.0);
    } else {
      x = static_cast<int>(round(offset.x() * scaling_percent / 100.0));
      y = static_cast<int>(round(offset.y() * scaling_percent / 100.0));
    }
    painter.drawPixmap(QPoint(x, y), scaled_pixmap);

    QRegion parts_region = scaled_pixmap.mask();
    parts_region.translate(x, y);
    region += parts_region;
  }


  // アナログ部分の再描画
  void updateAnalog(QPainter& painter, QRegion& region,
                    const ThemeResource& resource,
                    int hour, int minute, int second, size_t scaling_percent) {

    // 針の角度を再計算
    qreal analog_h_degree = (360.0 * (hour / 12.0)) + (30.0 * (minute / 60.0));
    QRect analog_h_offset = getPosition(resource, "analog/hari_h");
    drawPixmap(painter, region, resource, "analog/hari_h", analog_h_offset,
               scaling_percent, analog_h_degree, CenterPosition);

    qreal analog_m_degree = 360.0 * minute / 60.0;
    QRect analog_m_offset = getPosition(resource, "analog/hari_m");
    drawPixmap(painter, region, resource, "analog/hari_m", analog_m_offset,
               scaling_percent, analog_m_degree, CenterPosition);

    if (! resource.option.value("analog/sec", "").compare("on")) {
      // 秒針は、指定があるときのみ描画する
      qreal analog_s_degree = 360.0 * second / 60.0;
      QRect analog_s_offset = getPosition(resource, "analog/hari_s");
      drawPixmap(painter, region, resource, "analog/hari_s", analog_s_offset,
                 scaling_percent, analog_s_degree, CenterPosition);
    }
  }


  // デジタル部分の再描画
  void updateDigital(QPainter& painter, QRegion& region,
                     const ThemeResource& resource,
                     const QVector<QPoint>& digital_positions,
                     int hour, int minute, int second, size_t scaling_percent) {

    enum { BufferSize = 11 };
    char buffer[BufferSize] = "hh:mm ss x";

    if (! resource.option.value("digital/sec", "").compare("on")) {
      // 秒を描画する
      snprintf(buffer, BufferSize, "%02d:%02d_%02d_%c",
               hour, minute, second, (hour >= 12) ? 'P' : 'A');
    } else {
      // 秒は描画しない
      snprintf(buffer, BufferSize, "%02d:%02d____%c",
               hour, minute, (hour >= 12) ? 'P' : 'A');
    }

    if (! resource.option.value("digital/colon", "").compare("on")) {
      // コロンを点滅させる
      if (second & 0x01) {
        buffer[2] = ' ';
      }
    }

    // 描画処理
    char digital_char[] = "_";
    size_t n = strlen(buffer);
    for (size_t i = 0; i < n; ++i) {
      digital_char[0] = buffer[i];
      QPixmap* parts = resource.pixmap.value(digital_char, NULL);
      if (! parts) {
        continue;
      }

      QSize size = parts->size();
      int x = digital_positions[i].x();
      int y = digital_positions[i].y();
      drawPixmap(painter, region, resource, digital_char,
                 QRect(x, y, size.width(), size.height()), scaling_percent);
    }
  }

  // カレンダー部品の描画
  void updateCalendar(QPainter& painter, QRegion& region,
                      const ThemeResource& resource,
                      const QVector<QPoint>& calendar_positions,
                      int month, int day, int day_of_week,
                      size_t scaling_percent) {

    enum { BufferSize = 5 };
    char buffer[BufferSize] = "mmdd";

    bool calendar_month =
      ! resource.option.value("calendar/month", "").compare("on");
    if (calendar_month) {
      // 月を描画する
      snprintf(buffer, BufferSize, "%02d%02d", month, day);
    } else {
      // 月は描画しない
      snprintf(buffer, BufferSize, "__%02d", day);
    }

    // セパレータの描画
    const char* separator_tags[] = { "MONTH", "DAY" };
    size_t n = sizeof(separator_tags) / sizeof(separator_tags[0]);
    for (int i = 0; i < 2; ++i) {
      if ((i == 0) && (! calendar_month)) {
        // !!! 月を表示しないときは、っていう意味だが、わかりにくい
        // !!! どうにかすべき
        continue;
      }

      const char* tag = separator_tags[i];
      QPixmap* parts = resource.pixmap.value(tag, NULL);
      if (! parts) {
        continue;
      }

      // !!! マジックナンバー 4 をなんとかする
      QSize size = parts->size();
      int x = calendar_positions[4 + i].x();
      int y = calendar_positions[4 + i].y();
      drawPixmap(painter, region, resource, tag,
                 QRect(x, y, size.width(), size.height()), scaling_percent);
    }

    // 曜日の描画
    const char* week_names[] = {
      "SUN", "MON", "TUE", "WED", "THU", "FRI", "SAT", "SUN",
    };
    const char* tag = week_names[day_of_week];
    QPixmap* parts = resource.pixmap.value(tag, NULL);
    if (parts) {
      // !!! マジックナンバー 6 をなんとかする
      QSize size = parts->size();
      int x = calendar_positions[6].x();
      int y = calendar_positions[6].y();
      drawPixmap(painter, region, resource, tag,
                 QRect(x, y, size.width(), size.height()), scaling_percent);
    }

    // 数値の描画処理
    char number_char[] = "Cx";
    n = strlen(buffer);
    for (size_t i = 0; i < n; ++i) {
      number_char[1] = buffer[i];
      QPixmap* parts = resource.pixmap.value(number_char, NULL);
      if (! parts) {
        continue;
      }

      QSize size = parts->size();
      int x = calendar_positions[i].x();
      int y = calendar_positions[i].y();
      drawPixmap(painter, region, resource, number_char,
                 QRect(x, y, size.width(), size.height()), scaling_percent);
    }
  }


  // 追加部品の描画
  void updateAdditional(QPainter& painter, QRegion& region,
                        const ThemeResource& resource,
                        size_t scaling_percent) {

    QRect additional_offset = getPosition(resource, "additional/near");
    drawPixmap(painter, region, resource, "additional/near",
               additional_offset, scaling_percent);

    // !!! いずれ、アニメーション画像を追加
  }
};


// 再描画
void ClockPartsDraw::redraw(QPainter& painter, QRegion& region,
                            const ThemeResource& resource,
                            const QVector<QPoint>& digital_positions,
                            const QVector<QPoint>& calendar_positions,
                            size_t scaling_percent, int sec_offset) {

  painter.setRenderHint(QPainter::SmoothPixmapTransform);

  // ベース画像の描画
  QRect offset = getPosition(resource, "base/base_org");
  drawPixmap(painter, region,
             resource, "base/base_org", offset, scaling_percent);

  // 部品画像の描画
  QTime current_time = QTime::currentTime();
  QDateTime current_date = QDateTime(QDate::currentDate(), current_time);
  QTime time = current_time.addSecs(sec_offset);
  QDate date = (current_date.addSecs(sec_offset)).date();

  updateCalendar(painter, region, resource, calendar_positions,
                 date.month(), date.day(), date.dayOfWeek(), scaling_percent);
  updateDigital(painter, region, resource, digital_positions,
                time.hour(), time.minute(), time.second(), scaling_percent);
  updateAnalog(painter, region, resource,
               time.hour(), time.minute(), time.second(), scaling_percent);
  updateAdditional(painter, region, resource, scaling_percent);
}
