/*!
  \file
  \brief テーマ時計の描画

  \author Satofumi KAMIMURA

  $Id$

  \todo エラーメッセージの処理
  \todo 時刻毎のベース画像切替え機能を実装する
  \todo アニメーション機能を付加する
*/

#include <QLabel>
#include <QVBoxLayout>
#include <QRegion>
#include <QBitmap>
#include <QCloseEvent>
#include <QPainter>
#include "ThemeClockDraw.h"
#include "RrdaEncoder.h"
#include "RrdaResource.h"
#include "SettingConfig.h"
#include "ThemeResource.h"
#include "ClockPartsDraw.h"

#ifdef _MSC_VER
#define snprintf _snprintf
#endif


struct ThemeClockDraw::pImpl {

  static const bool CenterPosition = true; // 画像中心を基準にして描画

  QLabel* message_label_;
  bool theme_loaded_;
  QString theme_file_;
  ThemeResource theme_resource_;
  QRegion region_;
  QWidget* parent_widget_;
  QVector<QPixmap> digital_pixmap_;
  QVector<QPixmap> number_pixmap_;
  QVector<QPixmap> week_pixmap_;
  QVector<QPixmap> separator_pixmap_;
  QVector<QPoint> digital_positions_;
  QVector<QPoint> calendar_positions_;
  size_t scaling_percent_;
  int sec_offset_;


  pImpl(ThemeClockDraw* parent)
    : message_label_(NULL), theme_loaded_(false), parent_widget_(NULL),
      scaling_percent_(100), sec_offset_(0) {

    // テーマファイルのドラッグを促すフォームを作成
    message_label_ =
      new QLabel(QLabel::tr("Drop theme file in this form."), parent);
    message_label_->setAlignment(Qt::AlignCenter);
    QSize size = message_label_->sizeHint();
    parent->resize(QSize(static_cast<int>(size.width() * 1.2),
                         size.height() * 4));
    QVBoxLayout* layout = new QVBoxLayout(parent);
    layout->addWidget(message_label_);
    parent->setLayout(layout);
  }


  // テーマファイルの読み出し
  bool loadThemeFile(const QString& theme_file, const char* key) {

    // リソースの読み出しを行う
    RrdaEncoder rrda_encoder(theme_file.toStdString().c_str(), key);

    std::vector<std::string> resource_files;
    rrda_encoder.getResourceNames(resource_files);

    // "config" がリソースに含まれているかの確認
    if (! isExist("config", resource_files)) {
      // !!! リソースにないよー、と表示して戻る
      // !!!
      // !!! 失敗する可能性としては、
      // !!! キーが違う、テーマファイルでない、壊れている、などか？
      return false;
    }

    // config を読み出してパース
    std::auto_ptr<RrdaResource> config(rrda_encoder.getResource("config"));
    size_t size = 0;
    char* p = reinterpret_cast<char*>(config->get(&size));

    // 設定を読みだし、必要な情報を格納する
    if (! loadResources(p, size, rrda_encoder)) {
      // !!! 「リソースが不足しています」という表示を行う
      // !!! テーマファイルが不完全です。か？ これも意味不明だな...。
      // !!!
      return false;
    }

    theme_loaded_ = true;
    theme_file_ = theme_file;

    return true;
  }


  // リソース名が存在するかを返す
  bool isExist(const char* resource_name,
               const std::vector<std::string>& resources) {

    std::vector<std::string>::const_iterator p =
      std::find(resources.begin(), resources.end(), resource_name);

    return (p == resources.end()) ? false : true;
  }


  // リソースの読み出し
  bool loadResources(char* p, size_t size, RrdaEncoder& rrda_encoder) {

    // 設定ファイルの読み出し
    QByteArray data(p, size);
    QMap<std::string, std::string> tags;
    SettingConfig setting(tags, data);

    // 画像の格納
    ThemeResource resource;
    loadPixmap(resource, "base/base_org", tags, rrda_encoder);
    loadPixmap(resource, "analog/hari_h", tags, rrda_encoder);
    loadPixmap(resource, "analog/hari_m", tags, rrda_encoder);
    loadPixmap(resource, "analog/hari_s", tags, rrda_encoder);
    loadPixmap(resource, "additional/near", tags, rrda_encoder);

    // アナログ部品、追加画像の位置を格納
    loadOffset(resource, "analog/hari_h",
               "analog/h_xpos", "analog/h_ypos", tags);
    loadOffset(resource, "analog/hari_m",
               "analog/m_xpos", "analog/m_ypos", tags);
    loadOffset(resource, "analog/hari_s",
               "analog/s_xpos", "analog/s_ypos", tags);
    loadOffset(resource, "additional/near",
               "additional/ne_xpos", "additional/ne_ypos", tags);

    // 設定の格納
    loadOption(resource, "analog/sec", tags);
    loadOption(resource, "digital/colon", tags);
    loadOption(resource, "digital/sec", tags);
    loadOption(resource, "calendar/month", tags);

    // デジタル文字を、文字毎に格納
    loadDigitalCharacter(resource, tags, rrda_encoder);

    // デジタル文字の描画位置を設定
    loadDigitalPosition(digital_positions_, tags);

    // カレンダー文字、曜日、セパレータを、文字毎に格納
    loadCalendarCharacter(resource, tags, rrda_encoder);

    // カレンダー文字の描画位置を設定
    loadCalendarPosition(calendar_positions_, tags);

    // 描画リソースが何もなければ、エラー
    if (resource.pixmap.empty()) {
      return false;
    }

    std::swap(theme_resource_, resource);

    return true;
  }


  // 画像の読み出し
  bool loadPixmap(ThemeResource& resource, const std::string& tag_name,
                  const QMap<std::string, std::string>& tags,
                  RrdaEncoder& rrda_encoder) {

    // タグ名の RRDA リソースからの取得
    std::string pixmap_tag_name = tags.value(tag_name, "");
    if (pixmap_tag_name.empty()) {
      // !!! 存在しない
      return false;
    }

    // 画像の読み出し
    std::auto_ptr<RrdaResource>
      config(rrda_encoder.getResource(pixmap_tag_name.c_str()));
    size_t size = 0;
    char* p = reinterpret_cast<char*>(config->get(&size));
    if (size == 0) {
      return false;
    }

    // 画像の登録
    QByteArray data(p, size);
    QPixmap* pixmap = new QPixmap;
    pixmap->loadFromData(data);
    resource.pixmap.insert(tag_name, pixmap);

    return true;
  }


  // 対応する画像が既に resouce に登録されていなければならない
  void loadOffset(ThemeResource& resource, const std::string& tag_name,
                  const std::string& x_tag, const std::string& y_tag,
                  const QMap<std::string, std::string>& tags) {

    // 位置の取得と登録
    int x = atoi(tags.value(x_tag, "0").c_str());
    int y = atoi(tags.value(y_tag, "0").c_str());

    // サイズの取得
    QPixmap* pixmap = resource.pixmap.value(tag_name, NULL);
    int w = (pixmap) ? pixmap->width() : 0;
    int h = (pixmap) ? pixmap->height() : 0;

    resource.offset.insert(tag_name, QRect(x, y, w, h));
  }


  // オプションの読み出し
  void loadOption(ThemeResource& resource, const std::string& tag_name,
                  const QMap<std::string, std::string>& tags) {

    // オプションの取得と登録
    std::string tag_value = tags.value(tag_name, "");
    if (tag_value.empty()) {
      // !!! 存在しない
      return;
    }
    resource.option.insert(tag_name, tag_value);
  }


  // デジタル画像の読み出し
  void loadDigitalCharacter(ThemeResource& resource,
                            QMap<std::string, std::string>& tags,
                            RrdaEncoder& rrda_encoder) {

    // 文字毎に等間隔で分割して登録する
    const char* tag_names[] = {
      "0", "1", "2", "3", "4", "5", "6", "7", "8", "9",
      " ", ":", "A", "P",
    };
    size_t n = sizeof(tag_names) / sizeof(tag_names[0]);

    loadCharacterImages(resource, tags, rrda_encoder,
                        digital_pixmap_, "digital/parts", tag_names, n);
  }


  // カレンダー用画像の読み出し
  bool loadCalendarCharacter(ThemeResource& resource,
                             QMap<std::string, std::string>& tags,
                             RrdaEncoder& rrda_encoder) {

    // 数値文字の読み出し
    const char* number_names[] = {
      "C0", "C1", "C2", "C3", "C4", "C5", "C6", "C7", "C8", "C9",
    };
    size_t n = sizeof(number_names) / sizeof(number_names[0]);

    if (! loadCharacterImages(resource, tags, rrda_encoder, number_pixmap_,
                              "calendar/number", number_names, n)) {
      return false;
    }

    // 曜日の読み出し
    const char* week_names[] = {
      "SUN", "MON", "TUE", "WED", "THU", "FRI", "SAT",
    };
    n = sizeof(week_names) / sizeof(week_names[0]);

    if (! loadCharacterImages(resource, tags, rrda_encoder,
                              week_pixmap_, "calendar/week", week_names, n)) {
      return false;
    }

    // 区切り文字の読み出し
    const char* separator_names[] = {
      "MONTH", "DAY",
    };
    n = sizeof(separator_names) / sizeof(separator_names[0]);

    if (! loadCharacterImages(resource, tags, rrda_encoder, separator_pixmap_,
                              "calendar/separator", separator_names, n)) {
      return false;
    }

    return true;
  }


  // 文字画像の読み出し
  bool loadCharacterImages(ThemeResource& resource,
                           QMap<std::string, std::string>& tags,
                           RrdaEncoder& rrda_encoder,
                           QVector<QPixmap>& pixmaps,
                           const char* image_tag,
                           const char* tag_names[], size_t tag_names_size) {

    ThemeResource digital_resource;
    if (! loadPixmap(digital_resource, image_tag, tags, rrda_encoder)) {
      return false;
    }

    QPixmap* pixmap = digital_resource.pixmap.value(image_tag, NULL);
    if (! pixmap) {
      // !!! 失敗はしないはずだが、一応
      return false;
    }
    const QSize size = pixmap->size();

    // 文字毎に等間隔で分割して登録する
    pixmaps.clear();
    size_t each_width = size.width() / tag_names_size;
    size_t each_height = size.height();
    for (size_t i = 0; i < tag_names_size; ++i) {

      // !!! list にしておけば、アドレスが変化しないので、より適切か？
      pixmaps.push_back(pixmap->copy(i * each_width, 0,
                                     each_width, each_height));
    }

    // 登録した vector のアドレスを確保
    for (size_t i = 0; i < tag_names_size; ++i) {
      resource.pixmap.insert(tag_names[i], &pixmaps[i]);
    }

    return true;
  }


  // フォームの初期化
  void initializeWidget(ThemeClockDraw* parent) {

#if 0
    // 透明にしてみる
    //parent->setAutoFillBackground(true);
    QColor color(0, 0, 0, 0);
    parent->setPalette(color);
#endif

    resizeWidget(parent);
  }


  // リサイズ
  void resizeWidget(ThemeClockDraw* parent) {

    QPixmap* base_pixmap = theme_resource_.pixmap.value("base/base_org");
    QSize size = base_pixmap->size();
    QSize scaled_size =
      QSize(static_cast<int>(scaling_percent_ * size.width() / 100.0),
            static_cast<int>(scaling_percent_ * size.height() / 100.0));
    QPixmap scaled_pixmap = base_pixmap->scaled(scaled_size);
    parent->resize(scaled_pixmap.size());
  }


  // 再描画
  void paintEvent(ThemeClockDraw* parent) {

    // 部品の再描画
    QPainter painter(parent);
    region_ = QRegion();
    ClockPartsDraw::redraw(painter, region_, theme_resource_,
                           digital_positions_, calendar_positions_,
                           scaling_percent_, sec_offset_);
    if (parent_widget_) {
      parent_widget_->setMask(region_);
    }
    painter.end();
  }

  // テーマのリソースからの読み出し
  bool loadThemeResouce(const QMap<std::string, QPixmap*>& pixmaps,
                        const QMap<std::string, QRect>& offset,
                        const QVector<QPoint>& digital_positions,
                        const QVector<QPoint>& calendar_positions,
                        const QMap<std::string, std::string>& option) {

    // 画像リソースの更新
    theme_resource_.pixmap = pixmaps;

    // 位置情報の更新
    theme_resource_.offset = offset;
    digital_positions_ = digital_positions;
    calendar_positions_ = calendar_positions;

    // オプション設定の更新
    theme_resource_.option = option;

    theme_loaded_ = true;

    return true;
  }
};


ThemeClockDraw::ThemeClockDraw(QWidget* parent)
  : QWidget(parent), pimpl(new pImpl(this)) {
}


ThemeClockDraw::~ThemeClockDraw(void) {
}


// テーマファイルの読み出し
bool ThemeClockDraw::loadThemeFile(const QString& theme_file,
                                   const char* key) {

  if (! pimpl->loadThemeFile(theme_file, key)) {
    return false;
  }

  // フォーム初期化を行い、ドラッグ用のメッセージを隠す
  pimpl->initializeWidget(this);
  pimpl->message_label_->hide();

  return true;
}


// テーマのリソースからの読み出し
bool ThemeClockDraw::loadThemeResource(const QMap<std::string, QPixmap*>&
                                       pixmaps,
                                       const QMap<std::string, QRect>& offset,
                                       const QVector<QPoint>&
                                       digital_positions,
                                       const QVector<QPoint>&
                                       calendar_positions,
                                       const QMap<std::string, std::string>&
                                       option) {

  if (! pimpl->loadThemeResouce(pixmaps, offset,
                                digital_positions, calendar_positions,
                                option)) {
    return false;
  }

  pimpl->initializeWidget(this);
  pimpl->message_label_->hide();

  return true;
}


// テーマファイル名の取得
const QString ThemeClockDraw::getThemeFile(void) {

  return pimpl->theme_file_;
}


// 再描画
void ThemeClockDraw::redraw(int sec_offset, QWidget* parent) {

  pimpl->sec_offset_ = sec_offset;
  pimpl->parent_widget_ = parent;
  if (pimpl->theme_loaded_) {
    // 描画イベントを発生させる
    update();
  }
}


// 描画イベント
void ThemeClockDraw::paintEvent(QPaintEvent* event) {
  static_cast<void>(event);

  if (pimpl->theme_loaded_) {
    pimpl->paintEvent(this);
  }
}


// クローズイベント
void ThemeClockDraw::closeEvent(QCloseEvent* event) {

  event->accept();
  emit previewClose();
}


// 描画の拡大率を設定
void ThemeClockDraw::updateScalingPercent(size_t percent) {

  pimpl->scaling_percent_ = percent;
  pimpl->resizeWidget(this);
}
