/*!
  \file
  \brief テーマファイル作成用のメインウィンドウ管理

  \author Satofumi KAMIMURA

  $Id$

  \todo 作りなおす機会があれば、ベース画像は特別扱いにしておく
*/

#include <QHeaderView>
#include <QCloseEvent>
#include <QFileDialog>
#include <QGraphicsPixmapItem>
#include <QShortcut>
#include <QMessageBox>
#include <QTimer>
#include <cmath>
#include "ThemeConfigWindow.h"
#include "ArchiveConfig.h"
#include "SettingConfig.h"
#include "ReleaseDialog.h"
#include "ReleaseDataFile.h"
#include "ConvertStdStringPath.h"
#include "ThemeClockDraw.h"
#include "ThemeResource.h"
#include "PartsPlaceScene.h"

using namespace qrk;


namespace {
  typedef QMap<QTreeWidgetItem*, std::string> ItemToName;
  typedef QMap<std::string, QTreeWidgetItem*> NameToItem;

  typedef QMap<std::string, QPixmap*> Pixmaps;
  typedef QMap<std::string, QIcon*> Icons;
  typedef QMap<std::string, QGraphicsItem*> Items;

  typedef QMap<std::string, QRect> AdditionalLinesRect;

  typedef QMap<std::string, QPoint> PositionMap;

  typedef struct {
    const char* name;
    const char* x_tag;
    const char* y_tag;
  } position_tag_t;
};


static const char* ImageTags[] = {
  "base/base_org",
  "analog/hari_h", "analog/hari_m", "analog/hari_s",
  "digital/parts",
  "calendar/number", "calendar/week", "calendar/separator",
  "additional/near",
};


static const position_tag_t PositionTags[] = {
  { "analog/hari_h", "analog/h_xpos", "analog/h_ypos", },
  { "analog/hari_m", "analog/m_xpos", "analog/m_ypos", },
  { "analog/hari_s", "analog/s_xpos", "analog/s_ypos", },

  { "digital/h1", "digital/h1_xpos", "digital/h1_ypos", },
  { "digital/h2", "digital/h2_xpos", "digital/h2_ypos", },
  { "digital/m1", "digital/m1_xpos", "digital/m1_ypos", },
  { "digital/m2", "digital/m2_xpos", "digital/m2_ypos", },
  { "digital/s1", "digital/s1_xpos", "digital/s1_ypos", },
  { "digital/s2", "digital/s2_xpos", "digital/s2_ypos", },
  { "digital/cl", "digital/cl_xpos", "digital/cl_ypos", },
  { "digital/ap", "digital/ap_xpos", "digital/ap_ypos", },

  { "calendar/m1", "calendar/m1_xpos", "calendar/m1_ypos", },
  { "calendar/m2", "calendar/m2_xpos", "calendar/m2_ypos", },
  { "calendar/d1", "calendar/d1_xpos", "calendar/d1_ypos", },
  { "calendar/d2", "calendar/d2_xpos", "calendar/d2_ypos", },
  { "calendar/we", "calendar/we_xpos", "calendar/we_ypos", },
  { "calendar/s1", "calendar/s1_xpos", "calendar/s1_ypos", },
  { "calendar/s2", "calendar/s2_xpos", "calendar/s2_ypos", },

  { "additional/near", "additional/ne_xpos", "additional/ne_ypos", },
};



struct ThemeConfigWindow::pImpl {

  QAction* new_act_;
  QAction* open_act_;
  QAction* save_act_;

  ArchiveConfig* archive_config_;
  SettingConfig* setting_config_;

  bool save_ok_;
  QString first_title_;

  bool same_hands_center_;

  std::string current_tag_;

  ItemToName item_to_name_;
  NameToItem name_to_item_;

  Pixmaps pixmaps_;
  Icons icons_;

  PartsPlaceScene* scene_;
  Items items_;
  AdditionalLinesRect additional_lines_rect_;
  std::vector<QPixmap> digital_parts_;
  std::vector<QPixmap> number_parts_;
  std::vector<QPixmap> week_parts_;
  std::vector<QPixmap> separator_parts_;

  PositionMap parts_positions_;

  ReleaseDialog release_dialog_;
  std::string work_dir_;

  ThemeClockDraw preview_draw_;

  QTimer redraw_timer_;

  QCursor cursor_;

  pImpl(void)
    : new_act_(NULL), open_act_(NULL), save_act_(NULL),
      archive_config_(NULL), setting_config_(NULL), save_ok_(false),
      same_hands_center_(false), scene_(NULL), work_dir_(".") {
  }


  // フォームの初期化
  void initializeForm(ThemeConfigWindow* parent) {

    // アイコンの適用
    parent->setWindowIcon(QIcon(":icons/qtmclock_config_icon"));

    // Window Title の取得
    first_title_ = parent->windowTitle();

    // 描画ウィンドウの設定
    initializeGraphicsView(parent);

    // ツリー表示の幅を調整
    parent->parts_tree_->header()->setResizeMode(0, QHeaderView::Stretch);
    parent->parts_tree_->header()->setResizeMode(1, QHeaderView::Stretch);

    // タブの初期選択
    parent->checkbox_tab_->setCurrentIndex(0);

    // チェックボックス
    connect(parent->center_checkbox_, SIGNAL(clicked(bool)),
            parent, SLOT(centerCheckHandler(bool)));
    connect(parent->blink_checkbox_, SIGNAL(clicked(bool)),
            parent, SLOT(blinkCheckHandler(bool)));
    connect(parent->analog_checkbox_, SIGNAL(clicked(bool)),
            parent, SLOT(analogCheckHandler(bool)));
    connect(parent->digital_checkbox_, SIGNAL(clicked(bool)),
            parent, SLOT(digitalCheckHandler(bool)));
    same_hands_center_ = parent->center_checkbox_->isChecked();
    connect(parent->month_checkbox_, SIGNAL(clicked(bool)),
            parent, SLOT(monthCheckHandler(bool)));

    // メニューイベントの接続
    connect(parent->action_exit_, SIGNAL(triggered()), parent, SLOT(close()));
    connect(parent->action_exit_, SIGNAL(triggered()),
            &release_dialog_, SLOT(close()));

    // カーソルによる、選択画像の移動
    (void) new QShortcut(Qt::Key_Left, parent, SLOT(leftPressed()));
    (void) new QShortcut(Qt::Key_Right, parent, SLOT(rightPressed()));
    (void) new QShortcut(Qt::Key_Up, parent, SLOT(upPressed()));
    (void) new QShortcut(Qt::Key_Down, parent, SLOT(downPressed()));

    // クリックによる部品の選択
    connect(scene_, SIGNAL(clicked(int, int)),
            parent, SLOT(partsClickedHandler(int, int)));

    // 部品画像の更新
    connect(parent->image_lineedit_, SIGNAL(returnPressed()),
            parent, SLOT(partsEditHandler()));
    connect(parent->update_button_, SIGNAL(clicked()),
            parent, SLOT(updateHandlelr()));

    // 部品表示ツリー
    connect(parent->parts_tree_, SIGNAL(itemSelectionChanged()),
            parent, SLOT(imageUploadHandler()));

    // リリース
    connect(&release_dialog_, SIGNAL(release(QString, QString)),
            parent, SLOT(releaseHandler(QString, QString)));

    // プレビュー
    connect(parent->play_button_, SIGNAL(clicked()),
            parent, SLOT(previewClock()));

    // プレビュー
    connect(&redraw_timer_, SIGNAL(timeout()), parent, SLOT(previewRedraw()));
    connect(&preview_draw_, SIGNAL(previewClose()),
            parent, SLOT(previewCloseHandler()));

    // 時刻オフセット
    connect(parent->offset_slider_, SIGNAL(valueChanged(int)),
            parent, SLOT(offsetUpdated(int)));
  }


  // 描画ウィンドウの更新
  void initializeGraphicsView(ThemeConfigWindow* parent) {

    scene_ = new PartsPlaceScene(parent);
    scene_->setBackgroundBrush(QPixmap(":/images/background"));
    parent->parts_view_->setScene(scene_);
  }


  // ツリー情報ポインタの格納
  void treeTagInitialize(ThemeConfigWindow* parent) {

    // ベース画像
    QTreeWidgetItem* base = parent->parts_tree_->itemAt(0, 0);
    registerTreeItem("base/base_org", base);

    // アナログ部品
    QTreeWidgetItem* analog = parent->parts_tree_->itemBelow(base);
    registerTreeItem("analog", analog);

    // 短針の画像
    QTreeWidgetItem* short_hand = analog->child(0);
    registerTreeItem("analog/hari_h", short_hand);

    // 長針の画像
    QTreeWidgetItem* long_hand = analog->child(1);
    registerTreeItem("analog/hari_m", long_hand);

    // 秒針の画像
    QTreeWidgetItem* second_hand = analog->child(2);
    registerTreeItem("analog/hari_s", second_hand);

    // デジタル文字
    QTreeWidgetItem* digital = parent->parts_tree_->itemBelow(analog);
    registerTreeItem("digital/parts", digital);

    // h1
    QTreeWidgetItem* h1 = digital->child(0);
    registerTreeItem("digital/h1", h1);

    // h2
    QTreeWidgetItem* h2 = h1->child(0);
    registerTreeItem("digital/h2", h2);

    // colon
    QTreeWidgetItem* colon = digital->child(1);
    registerTreeItem("digital/cl", colon);

    // m1
    QTreeWidgetItem* m1 = digital->child(2);
    registerTreeItem("digital/m1", m1);

    // m2
    QTreeWidgetItem* m2 = m1->child(0);
    registerTreeItem("digital/m2", m2);

    // s1
    QTreeWidgetItem* s1 = digital->child(3);
    registerTreeItem("digital/s1", s1);

    // s2
    QTreeWidgetItem* s2 = s1->child(0);
    registerTreeItem("digital/s2", s2);

    // am/mp
    QTreeWidgetItem* ap = digital->child(4);
    registerTreeItem("digital/ap", ap);

    // カレンダー
    QTreeWidgetItem* calendar = parent->parts_tree_->itemBelow(digital);
    registerTreeItem("calendar/parts", calendar);

    // カレンダー用の文字画像
    QTreeWidgetItem* number = calendar->child(0);
    registerTreeItem("calendar/number", number);

    QTreeWidgetItem* separator = calendar->child(1);
    registerTreeItem("calendar/separator", separator);

    // m1
    QTreeWidgetItem* calendar_m1 = number->child(0);
    registerTreeItem("calendar/m1", calendar_m1);

    // m2
    QTreeWidgetItem* calendar_m2 = calendar_m1->child(0);
    registerTreeItem("calendar/m2", calendar_m2);

    // s1
    QTreeWidgetItem* calendar_s1 = calendar_m1->child(1);
    registerTreeItem("calendar/s1", calendar_s1);

    // d1
    QTreeWidgetItem* calendar_d1 = number->child(1);
    registerTreeItem("calendar/d1", calendar_d1);

    // d2
    QTreeWidgetItem* calendar_d2 = calendar_d1->child(0);
    registerTreeItem("calendar/d2", calendar_d2);

    // s2
    QTreeWidgetItem* calendar_s2 = calendar_d1->child(1);
    registerTreeItem("calendar/s2", calendar_s2);

    // 曜日
    QTreeWidgetItem* week = calendar->child(2);
    registerTreeItem("calendar/week", week);

    // 追加画像
    QTreeWidgetItem* additional = parent->parts_tree_->itemBelow(calendar);
    registerTreeItem("additional", additional);

    // 手前
    QTreeWidgetItem* near = additional->child(0);
    registerTreeItem("additional/near", near);

  }


  // 部品名とポインタの対応を登録
  void registerTreeItem(const std::string& tag, QTreeWidgetItem* item) {

    item_to_name_.insert(item, tag);
    name_to_item_.insert(tag, item);
  }


  // 部品のノードを開く
  void expandSubTreeNode(ThemeConfigWindow* parent) {

    QTreeWidgetItem* base = parent->parts_tree_->itemAt(0, 0);
    QTreeWidgetItem* analog = parent->parts_tree_->itemBelow(base);
    parent->parts_tree_->expandItem(analog);

#if 0
    // !!! 動作しない理由は不明
    QTreeWidgetItem* digital = parent->parts_tree_->itemBelow(analog);
    QTreeWidgetItem* calendar = parent->parts_tree_->itemBelow(digital);
    parent->parts_tree_->expandItem(calendar);
#endif
  }


  // enable / disable の設定
  void setActionsEnabled(ThemeConfigWindow* parent, bool enable) {

    // 保存
    setSaveEnabled(parent, enable);

    // メニュー、ツールボタン
    parent->action_save_as_->setEnabled(enable);
    parent->action_release_->setEnabled(enable);
    parent->action_preview_->setEnabled(enable);

    // 再生ボタン
    parent->play_button_->setEnabled(enable);
    parent->offset_slider_->setEnabled(enable);

    // 編集用 Widget
    parent->clock_groupbox_->setEnabled(enable);
    parent->calendar_groupbox_->setEnabled(enable);
    parent->parts_tree_->setEnabled(enable);
  }


  // 保存機能の enable / disable 設定
  void setSaveEnabled(ThemeConfigWindow* parent, bool enable) {

    parent->action_save_->setEnabled(enable);
    save_act_->setEnabled(enable);
  }


  // 画像の更新機能の enable / disable
  void setUpdateEnabled(ThemeConfigWindow* parent, bool enable) {

    parent->image_label_->setEnabled(enable);
    parent->image_lineedit_->setEnabled(enable);
    parent->update_button_->setEnabled(enable);
  }


  // 設定ファイルオブジェクトの生成
  void loadArchiveFile(ThemeConfigWindow* parent, const std::string& name) {

    ArchiveConfig* archive_config = NULL;
    if (name.empty()) {
      archive_config = new ArchiveConfig("");
    } else {
      QFileInfo fi(name.c_str());
      std::string absolute_path = fi.absoluteFilePath().toStdString();
      archive_config = new ArchiveConfig(absolute_path);
    }

    std::swap(archive_config, archive_config_);
    delete archive_config;

    work_dir_ = archive_config_->getReleaseDirectoryName();
    const std::string xml_file = archive_config_->getNormalData("config");
    const std::string adjusted_xml_file = xml_file.empty() ? "" : xml_file;
    SettingConfig* setting_config = new SettingConfig(adjusted_xml_file);
    std::swap(setting_config, setting_config_);
    delete setting_config;

    load(parent);
  }


  // 読み出し
  void load(ThemeConfigWindow* parent) {

    // 設定を読み出して反映する
    loadSettings(parent);

    // 保存可能、リリース可能にする
    setActionsEnabled(parent, true);

    // 現在のファイル名をタイトルバーに表示する
    setTitleToText(parent, archive_config_->getArchiveFileName());
  }


  // 設定の読み出し
  bool loadSettings(ThemeConfigWindow* parent) {

    // 部品位置の反映
    loadPosition();

    // 秒針の 表示 / 非表示
    bool analog_sec = ! setting_config_->getValue("analog/sec").compare("on");
    parent->analog_checkbox_->setChecked(analog_sec);

    // デジタル秒の 表示 / 非表示
    bool digital_sec = ! setting_config_->getValue("digital/sec").compare("on");
    parent->digital_checkbox_->setChecked(digital_sec);

    // カレンダー月の 表示 / 非表示
    bool calendar_month =
      ! setting_config_->getValue("calendar/month").compare("on");
    parent->month_checkbox_->setChecked(calendar_month);

    // 部品画像のツリーへの設定
    loadImages();

    return true;
  }


  // 画像の読み出し (全更新)
  void loadImages(void) {

    size_t n = sizeof(ImageTags) / sizeof(ImageTags[0]);
    for (size_t i = 0; i < n; ++i) {
      loadImage(ImageTags[i]);
    }

    // 描画アイテムの再構築
    updateItems();
  }


  // 画像の読み出し
  void loadImage(const char* tag) {

    const std::string archive_tag = setting_config_->getValue(tag);
    const std::string name = archive_config_->getNormalData(archive_tag);
    if (name.empty()) {
      return;
    }

    // 現在の画像を破棄し、新規に画像を読み込む
    delete pixmaps_.value(tag, NULL);
    delete icons_.value(tag, NULL);

    QPixmap* pixmap = new QPixmap(fromStdStringPath(name));
    pixmaps_[tag] = pixmap;

    QIcon* icon = new QIcon(*pixmap);
    icons_[tag] = icon;

    // ツリーアイテムの更新
    name_to_item_[tag]->setIcon(1, *icon);

    size_t dir_length = work_dir_.size();
    const std::string set_name = name.substr(dir_length, std::string::npos);
    name_to_item_[tag]->setText(1, set_name.c_str());

    // デジタル画像の場合、数値部品を更新
    if (! strcmp(tag, "digital/parts")) {
      updateDigitalParts(*pixmap);
    }

    // カレンダー画像の場合、構成部品を更新
    if (! strcmp(tag, "calendar/number")) {
      updateNumberParts(*pixmap);

    } else if (! strcmp(tag, "calendar/separator")) {
      updateSeparatorParts(*pixmap);

    } else if (! strcmp(tag, "calendar/week")) {
      updateWeekParts(*pixmap);
    }
  }

  // 数値部品の更新
  void updateDigitalParts(const QPixmap& pixmap) {

    // 文字毎に等間隔で分割して登録する
    const char* tag_names[] = {
      "0", "1", "2", "3", "4", "5", "6", "7", "8", "9",
      " ", ":", "A", "P",
    };

    size_t n = sizeof(tag_names) / sizeof(tag_names[0]);
    loadCharacterImages(digital_parts_, pixmap, tag_names, n);
  }


  // カレンダー部品(数値)の更新
  void updateNumberParts(const QPixmap& pixmap) {

    // 数値
    const char* number_names[] = {
      "C0", "C1", "C2", "C3", "C4", "C5", "C6", "C7", "C8", "C9",
    };

    size_t n = sizeof(number_names) / sizeof(number_names[0]);
    loadCharacterImages(number_parts_, pixmap, number_names, n);
  }


  // カレンダー部品(週)の更新
  void updateWeekParts(const QPixmap& pixmap) {

    // 曜日
    const char* week_names[] = {
      "SUN", "MON", "TUE", "WED", "THU", "FRI", "SAT",
    };

    size_t n = sizeof(week_names) / sizeof(week_names[0]);
    loadCharacterImages(week_parts_, pixmap, week_names, n);
  }


  // カレンダー部品(セパレータ)の更新
  void updateSeparatorParts(const QPixmap& pixmap) {

    // セパレータ
    const char* separator_names[] = {
      "MONTH", "DAY",
    };

    size_t n = sizeof(separator_names) / sizeof(separator_names[0]);
    loadCharacterImages(separator_parts_, pixmap, separator_names, n);
  }


  // 部分画像の読み出し
  void loadCharacterImages(std::vector<QPixmap>& pixmap_parts,
                           const QPixmap& pixmap, const char* tag_names[],
                           size_t tag_names_size) {
    // 部品の更新
    pixmap_parts.clear();
    QSize size = pixmap.size();
    size_t each_width = size.width() / tag_names_size;
    size_t each_height = size.height();
    for (size_t i = 0; i < tag_names_size; ++i) {
      pixmap_parts.push_back(pixmap.copy(each_width * i, 0,
                                         each_width, each_height));
    }

    // 登録した vector のアドレスを確保
    for (size_t i = 0; i < tag_names_size; ++i) {
      pixmaps_.insert(tag_names[i], &pixmap_parts[i]);
    }
  }


  // 描画アイテムの再構築
  void updateItems(void) {

    if (! setting_config_) {
      // 部品がロードされていなければ、戻る
      return;
    }

    // デジタル部品の位置が全て (0, 0) のときに、デフォルトの配置を行う
    if (isDigitalZeroPosition()) {
      updateDefaultDigitalPosition();
    }

    // カレンダー部品の位置が全て (0, 0) のときに、デフォルトの配置を行う
    if (isCalendarZeroPosition()) {
      updateDefaultCalendarPosition();
    }

    // 全アイテムの除去
    scene_->clear();
    items_.clear();

    // ベース
    QGraphicsItem* parent = NULL;
    parent = registerItem(parent, "base/base_org");

    // カレンダー部品
    registerItem(parent, "calendar/m1", "C0");
    registerItem(parent, "calendar/m2", "C1");
    registerItem(parent, "calendar/d1", "C2");
    registerItem(parent, "calendar/d2", "C3");
    registerItem(parent, "calendar/we", "MON");
    registerItem(parent, "calendar/s1", "MONTH");
    registerItem(parent, "calendar/s2", "DAY");

    // デジタル部品
    registerItem(parent, "digital/h1", "0");
    registerItem(parent, "digital/h2", "1");
    registerItem(parent, "digital/m1", "2");
    registerItem(parent, "digital/m2", "3");

    if (! setting_config_->getValue("digital/sec").compare("on")) {
      registerItem(parent, "digital/s1", "4");
      registerItem(parent, "digital/s2", "5");
    }

    registerItem(parent, "digital/cl", ":");
    registerItem(parent, "digital/ap", "A");

    // アナログ部品
    QGraphicsItem* short_hand = registerItem(parent, "analog/hari_h");
    if (same_hands_center_ && short_hand) {
      // 針の中心位置を、短針に揃える
      parts_positions_["analog/hari_m"] = parts_positions_["analog/hari_h"];
      parts_positions_["analog/hari_s"] = parts_positions_["analog/hari_h"];
    }
    registerItem(parent, "analog/hari_m");
    if (! setting_config_->getValue("analog/sec").compare("on")) {
      registerItem(parent, "analog/hari_s");
    }

    // 追加部品
    registerItem(parent, "additional/near");

    // 補助線
    QPen rect_pen;
    rect_pen.setStyle(Qt::DashLine);

    QPen line_pen;
    line_pen.setStyle(Qt::DotLine);

    for (AdditionalLinesRect::iterator it = additional_lines_rect_.begin();
         it != additional_lines_rect_.end(); ++it) {

      // 枠線を描画
      std::string key = it.key();
      QGraphicsItem* item_parent = items_[key];

      QRect rect = it.value();
      QGraphicsRectItem* rect_item =
        new QGraphicsRectItem(0, 0, rect.width() - 1, rect.height() - 1,
                              item_parent);
      rect_item->setPen(rect_pen);

      // 中心を通る直線を描画
      if (! key.compare(0, 12, "analog/hari_")) {
        int center_x = rect.width() / 2;
        int center_y = rect.height() / 2;
        QGraphicsLineItem* horizontal_line =
          new QGraphicsLineItem(0, center_y, rect.width() - 1, center_y,
                                item_parent);
        horizontal_line->setPen(line_pen);
        QGraphicsLineItem* vertical_line =
          new QGraphicsLineItem(center_x, 0, center_x, rect.height() - 1,
                                item_parent);
        vertical_line->setPen(line_pen);
      }
    }
  }


  // パーツの配置
  QGraphicsItem* registerItem(QGraphicsItem* parent, const char* tag,
                              const char* pixmap_tag = NULL) {

    QGraphicsPixmapItem* item = NULL;
    const char* use_tag = (pixmap_tag == NULL) ? tag : pixmap_tag;

    QPixmap* pixmap = pixmaps_.value(use_tag, NULL);
    if (pixmap) {
      item = new QGraphicsPixmapItem(*pixmap, parent);

      // 指定位置に配置。ただし、アナログ針は中心位置を管理している
      QPoint offset;
      size_t n = strlen("analog");
      if (! strncmp("analog", tag, n)) {
        QSize size = pixmap->size();
        offset = QPoint(size.width() / 2, size.height() / 2);
      }
      item->setPos(parts_positions_[tag] - item->scenePos() - offset);

      item->setTransformationMode(Qt::SmoothTransformation);
      items_.insert(tag, item);
    }
    if (item && (! parent)) {
      scene_->addItem(item);
    }

    return item;
  }


  // デジタル部品の位置が、全て同じかを調べる
  bool isDigitalZeroPosition(void) {

    const char* digital_tags[] = {
      "digital/h1", "digital/h2",
      "digital/m1", "digital/m2",
      "digital/s1", "digital/s2",
      "digital/cl", "digital/ap",
    };
    size_t n = sizeof(digital_tags) / sizeof(digital_tags[0]);
    for (size_t i = 0; i < n; ++i) {

      PositionMap::iterator p = parts_positions_.find(digital_tags[i]);
      if (p == parts_positions_.end()) {
        // タグがなければ、戻る
        return false;
      }

      QPointF position = parts_positions_[digital_tags[i]];
      if ((fabs(position.x()) > 0.1) || (fabs(position.y()) > 0.1)) {
        return false;
      }
    }
    return true;
  }


  // デジタル文字の初期位置を設定
  void updateDefaultDigitalPosition(void) {

    QPixmap* pixmap = pixmaps_.value("0", NULL);
    if (! pixmap) {
      return;
    }
    size_t width = pixmap->size().width();

    parts_positions_["digital/ap"] = QPoint(0, 0);
    parts_positions_["digital/h1"] = QPoint(width * 1, 0);
    parts_positions_["digital/h2"] = QPoint(width * 2, 0);
    parts_positions_["digital/cl"] = QPoint(width * 3, 0);
    parts_positions_["digital/m1"] = QPoint(width * 4, 0);
    parts_positions_["digital/m2"] = QPoint(width * 5, 0);

    parts_positions_["digital/s1"] = QPoint(width * 7, 0);
    parts_positions_["digital/s2"] = QPoint(width * 8, 0);
  }


  // カレンダー部品の位置が、全て同じかを調べる
  bool isCalendarZeroPosition(void) {

    const char* calendar_tags[] = {
      "calendar/m1", "calendar/m2",
      "calendar/d1", "calendar/d2",
      "calendar/s1", "calendar/s2",
      "calendar/we",
    };
    size_t n = sizeof(calendar_tags) / sizeof(calendar_tags[0]);
    for (size_t i = 0; i < n; ++i) {

      PositionMap::iterator p = parts_positions_.find(calendar_tags[i]);
      if (p == parts_positions_.end()) {
        // タグがなければ、戻る
        return false;
      }

      QPointF position = parts_positions_[calendar_tags[i]];
      if ((fabs(position.x()) > 0.1) || (fabs(position.y()) > 0.1)) {
        return false;
      }
    }
    return true;
  }


  // カレンダー文字の初期位置を設定
  void updateDefaultCalendarPosition(void) {

    QPixmap* pixmap = pixmaps_.value("C0", NULL);
    if (! pixmap) {
      return;
    }
    size_t width = pixmap->size().width();

    parts_positions_["calendar/m1"] = QPoint(0, 0);
    parts_positions_["calendar/m2"] = QPoint(width * 1, 0);
    parts_positions_["calendar/s1"] = QPoint(width * 2, 0);
    parts_positions_["calendar/d1"] = QPoint(width * 3, 0);
    parts_positions_["calendar/d2"] = QPoint(width * 4, 0);
    parts_positions_["calendar/s2"] = QPoint(width * 5, 0);
    parts_positions_["calendar/we"] = QPoint(width * 6, 0);
  }


  // 名前を付けて保存
  void save(ThemeConfigWindow* parent, const std::string& name) {

    if (archive_config_->saveAs(name)) {

      // 位置の反映
      savePosition();

      // XML 設定の保存
      setting_config_->save();

      // 変更があるまで、保存を disable にする
      setSaveEnabled(parent, false);

      // 現在のファイル名をタイトルバーに表示する
      setTitleToText(parent, name);
    }
  }


  // ウィンドウタイトルの更新
  void setTitleToText(ThemeConfigWindow* parent, const std::string& name) {

    QFileInfo fi(name.c_str());
    QString fileName = fi.fileName();
    if (! fileName.isEmpty()) {
      parent->setWindowTitle(first_title_ + " -- " + fileName);
    } else {
      parent->setWindowTitle(first_title_);
    }
  }


  // アイテムのタグ文字を探す
  bool findItemTag(const char* tags[],
                   size_t n, const std::string& tag) {

    for (size_t i = 0; i < n; ++i) {

      if (! tag.compare(tags[i])) {
        return true;
      }
    }
    return false;
  }


  // 位置のタグ文字を探す
  bool findPositionTag(position_tag_t tags[],
                       size_t n, const std::string& tag) {

    for (size_t i = 0; i < n; ++i) {

      if (! tag.compare(tags[i].name)) {
        return true;
      }
    }
    return false;
  }


  void loadPosition(void) {

    // setting_config_ の位置を描画用の位置に反映させる
    size_t n = sizeof(PositionTags) / sizeof(PositionTags[0]);
    for (size_t i = 0; i < n; ++i) {
      const position_tag_t& tag = PositionTags[i];
      int x = atoi(setting_config_->getValue(tag.x_tag).c_str());
      int y = atoi(setting_config_->getValue(tag.y_tag).c_str());
      parts_positions_[tag.name] = QPoint(x, y);
    }
  }


  void savePosition(void) {

    // 描画用の位置を setting_config_ に反映させる
    size_t n = sizeof(PositionTags) / sizeof(PositionTags[0]);
    for (size_t i = 0; i < n; ++i) {
      const position_tag_t& tag = PositionTags[i];

      QPoint& point = parts_positions_[tag.name];

      QString x = QString("%1").arg(point.x());
      QString y = QString("%1").arg(point.y());

      setting_config_->setValue(tag.x_tag, x.toStdString().c_str());
      setting_config_->setValue(tag.y_tag, y.toStdString().c_str());
    }
  }


  // 位置を更新
  void updatePosition(ThemeConfigWindow* parent,
                      int x, int y, const std::string& tag) {

    // "base/base_org" は移動させない
    if (! tag.compare("base/base_org")) {
      return;
    }

    // 各針の中心を揃えているときは、"analog/hand_X" を全て移動させる
    if (same_hands_center_ &&
        ((! tag.compare("analog/hari_h")) ||
         (! tag.compare("analog/hari_m")) ||
         (! tag.compare("analog/hari_s")))) {
      same_hands_center_ = false;

      updatePosition(parent, x, y, "analog/hari_h");
      updatePosition(parent, x, y, "analog/hari_m");
      updatePosition(parent, x, y, "analog/hari_s");

      same_hands_center_ = true;
      return;
    }

    // カレンダー部品をまとめて操作させる
    if (! tag.compare("calendar/parts")) {
      updatePosition(parent, x, y, "calendar/m1");
      updatePosition(parent, x, y, "calendar/d1");
      updatePosition(parent, x, y, "calendar/we");
    }

    // 曜日を移動させるための調整
    // !!! 原因を理解したら、修正すべき
    if (! tag.compare("calendar/week")) {
      updatePosition(parent, x, y, "calendar/we");
    }

    // "calendar/X1" タグのときは、"digital/X2", セパレータも移動させる
    if (! tag.compare("calendar/m1")) {
      updatePosition(parent, x, y, "calendar/m2");
      updatePosition(parent, x, y, "calendar/s1");

    } else if (! tag.compare("calendar/d1")) {
      updatePosition(parent, x, y, "calendar/d2");
      updatePosition(parent, x, y, "calendar/s2");
    }

    // "calendar/number" タグのときは、"calendar/mX", "calendar/dX" も移動させる
    if (! tag.compare("calendar/number")) {
      updatePosition(parent, x, y, "calendar/m1");
      updatePosition(parent, x, y, "calendar/d1");
    }

    // "calendar/separator" タグのときは、"calendar/s1", "calendar/s2" も移動させる
    if (! tag.compare("calendar/separator")) {
      updatePosition(parent, x, y, "calendar/s1");
      updatePosition(parent, x, y, "calendar/s2");
    }

    // アナログ部品をまとめて操作させる
    if (! tag.compare("analog")) {
      bool pre_flag = same_hands_center_;
      same_hands_center_ = false;

      updatePosition(parent, x, y, "analog/hari_h");
      updatePosition(parent, x, y, "analog/hari_m");
      updatePosition(parent, x, y, "analog/hari_s");

      same_hands_center_ = pre_flag;
      return;
    }

    // "digital/parts" タグのときは、"digital/XX" を全て移動させる
    if (! tag.compare("digital/parts")) {
      updatePosition(parent, x, y, "digital/h1");
      updatePosition(parent, x, y, "digital/m1");
      updatePosition(parent, x, y, "digital/s1");
      updatePosition(parent, x, y, "digital/cl");
      updatePosition(parent, x, y, "digital/ap");
    }

    // "digital/X1" のタグのときは、"digital/X2" も移動させる
    if (! tag.compare("digital/h1")) {
      updatePosition(parent, x, y, "digital/h2");
    }
    if (! tag.compare("digital/m1")) {
      updatePosition(parent, x, y, "digital/m2");
    }
    if (! tag.compare("digital/s1")) {
      updatePosition(parent, x, y, "digital/s2");
    }

    PositionMap::iterator p = parts_positions_.find(tag);
    if (p == parts_positions_.end()) {
      // 見つからなければ、戻る
      return;
    }

    QGraphicsItem* item = items_.value(tag, NULL);
    if (! item) {
      // !!! ありえないが、一応
      return;
    }

    // 管理位置の更新
    parts_positions_[tag] += QPoint(x, y);

    // 描画位置の更新
    item->setPos(item->pos() + QPoint(x, y));

    setSaveEnabled(parent, true);
  }


  // リリースファイルの作成
  void release(void) {

    releaseDataFile(*archive_config_);
  }


  // 登録してある名前を変更し、画像を更新する
  void updateItemName(ThemeConfigWindow* parent, const QString& fileName) {

    const std::string archive_tag = setting_config_->getValue(current_tag_);
    size_t dir_length = work_dir_.size();
    const std::string set_name =
      fileName.toStdString().substr(dir_length, std::string::npos);
    archive_config_->setNormalData(archive_tag, set_name);

    loadImage(current_tag_.c_str());
    updateItems();
    setSaveEnabled(parent, true);
  }


  // オフセット位置の設定
  void setOffset(QMap<std::string, QRect>& offset,
                 const std::string& tag_name) {

    QPixmap* pixmap = pixmaps_.value(tag_name, NULL);
    if (! pixmap) {
      // !!! 失敗はしないはずだが、一応
      return;
    }

    const QSize size = pixmap->size();
    const QPoint point = parts_positions_[tag_name];
    offset.insert(tag_name, QRect(point.x(), point.y(),
                                  size.width(), size.height()));
  }


  // Prevew の開始
  void previewOpened(ThemeConfigWindow* parent) {

    redraw_timer_.stop();

    // テーマの読み出し
    QVector<QPoint> digital_positions;
    loadDigitalPosition(digital_positions, parts_positions_);

    QVector<QPoint> calendar_positions;
    loadCalendarPosition(calendar_positions, parts_positions_);

    // オフセット位置の生成
    QMap<std::string, QRect> offset;
    setOffset(offset, "analog/hari_h");
    setOffset(offset, "analog/hari_m");
    setOffset(offset, "analog/hari_s");
    setOffset(offset, "additional/near");

    // オプションタグの登録
    QMap<std::string, std::string> option;
    option.insert("analog/sec",
                  parent->analog_checkbox_->isChecked() ? "on" : "off");
    option.insert("digital/sec",
                  parent->digital_checkbox_->isChecked() ? "on" : "off");
    option.insert("digital/colon",
                  parent->blink_checkbox_->isChecked() ? "on" : "off");
    option.insert("calendar/month",
                  parent->month_checkbox_->isChecked() ? "on" : "off");

    preview_draw_.loadThemeResource(pixmaps_, offset ,
                                    digital_positions, calendar_positions,
                                    option);
    setPreviewEnabled(parent, true);

    // ダイアログの表示
    preview_draw_.show();
    preview_draw_.raise();

    redraw_timer_.start(1000);
    redrawPreview(parent);

    // ボタンアイコンの変更
    QIcon stop_icon(":/icons/stop_icon");
    parent->play_button_->setIcon(stop_icon);
  }


  // Preview の終了
  void previewClosed(ThemeConfigWindow* parent) {

    redraw_timer_.stop();
    setPreviewEnabled(parent, false);

    // ボタンアイコンの変更
    QIcon stop_icon(":/icons/play_icon");
    parent->play_button_->setIcon(stop_icon);
  }


  // プレビュー次のフォーム設定
  void setPreviewEnabled(ThemeConfigWindow* parent, bool enable) {

    parent->parts_view_->setEnabled(! enable);
    parent->parts_tree_->setEnabled(! enable);
    parent->center_checkbox_->setEnabled(! enable);

    parent->offset_slider_->setEnabled(enable);
  }


  // プレビューの再描画
  void redrawPreview(ThemeConfigWindow* parent) {

    int sec_offset = parent->offset_slider_->value() * 60;
    preview_draw_.redraw(sec_offset);
  }


  // 補助線の追加
  void drawAdditionalLine(const std::string& selected_tag) {

    // ベース画像は対象外
    if (! selected_tag.compare("base/base_org")) {
      return;
    }

    // 関連部品についても描画を行う
    if (! selected_tag.compare("analog")) {
      drawAdditionalLine("analog/hari_h");
      drawAdditionalLine("analog/hari_m");
      drawAdditionalLine("analog/hari_s");

    } else if (! selected_tag.compare("digital/parts")) {
      drawAdditionalLine("digital/h1");
      drawAdditionalLine("digital/m1");
      drawAdditionalLine("digital/s1");
      drawAdditionalLine("digital/cl");
      drawAdditionalLine("digital/ap");

    } else if (! selected_tag.compare("digital/h1")) {
      drawAdditionalLine("digital/h2");

    } else if (! selected_tag.compare("digital/m1")) {
      drawAdditionalLine("digital/m2");

    } else if (! selected_tag.compare("digital/s1")) {
      drawAdditionalLine("digital/s2");

    } else if (! selected_tag.compare("calendar/parts")) {
      drawAdditionalLine("calendar/number");
      drawAdditionalLine("calendar/we");

    } else if (! selected_tag.compare("calendar/number")) {
      drawAdditionalLine("calendar/m1");
      drawAdditionalLine("calendar/d1");
      drawAdditionalLine("calendar/s1");
      drawAdditionalLine("calendar/s2");

    } else if (! selected_tag.compare("calendar/separator")) {
      drawAdditionalLine("calendar/s1");
      drawAdditionalLine("calendar/s2");

    } else if (! selected_tag.compare("calendar/m1")) {
      drawAdditionalLine("calendar/m2");
      drawAdditionalLine("calendar/s1");

    } else if (! selected_tag.compare("calendar/d1")) {
      drawAdditionalLine("calendar/d2");
      drawAdditionalLine("calendar/s2");
    }

    // 曜日を移動させるための調整
    // !!! 原因を理解したら、修正すべき
    if (! selected_tag.compare("calendar/week")) {
      drawAdditionalLine("calendar/we");
    }


    PositionMap::iterator it_position = parts_positions_.find(selected_tag);
    if (it_position == parts_positions_.end()) {
      return;
    }

    QPoint position = it_position.value();
    QPixmap* pixmap = NULL;
    if (! selected_tag.compare(0, 8, "digital/")) {
      pixmap = pixmaps_["0"];

    } else if (! selected_tag.compare(0, 10, "calendar/s")) {
      pixmap = pixmaps_["MONTH"];

    } else if (! selected_tag.compare("calendar/we")) {
      pixmap = pixmaps_["MON"];

    } else if (! selected_tag.compare(0, 9, "calendar/")) {
      pixmap = pixmaps_["C0"];

    } else {
      pixmap = pixmaps_[selected_tag];
    }
    QSize size = (pixmap) ? pixmap->size() : QSize(0, 0);

    // 針の部品は中心座標を登録してあるのを、画像位置に変更
    if (! selected_tag.compare(0, 12, "analog/hari_")) {
      position.setX(position.x() - (size.width() / 2));
      position.setY(position.y() - (size.height() / 2));
    }

    // 描画する枠線を登録
    additional_lines_rect_.
      insert(selected_tag,
             (QRect(position.x(), position.y(), size.width(), size.height())));
  }


  // 補助線の削除
  void removeAdditionalLine(void) {
    additional_lines_rect_.clear();
  }


  // 選択を解除する
  void clearItemSelection(ThemeConfigWindow* parent) {

    parent->parts_tree_->setCurrentItem(NULL);
    additional_lines_rect_.clear();
    updateItems();
  }


  // チェックボックスの変更を反映する
  void applyCheckChanged(ThemeConfigWindow* parent) {

    setSaveEnabled(parent, true);

    if (redraw_timer_.isActive()) {
      previewOpened(parent);
    }
  }
};


ThemeConfigWindow::ThemeConfigWindow(const QString& config_file)
  : pimpl(new pImpl) {
  setupUi(this);

  // ツールバーの初期化
  pimpl->new_act_ = new QAction(QIcon(":icons/new_icon"), tr("New"), this);
  pimpl->new_act_->setStatusTip(tr("Create a new config file."));
  connect(pimpl->new_act_, SIGNAL(triggered()), this, SLOT(newFile()));

  pimpl->open_act_ = new QAction(QIcon(":icons/open_icon"), tr("Open"), this);
  pimpl->open_act_->setStatusTip(tr("Open an exist config file."));
  connect(pimpl->open_act_, SIGNAL(triggered()), this, SLOT(openFile()));

  pimpl->save_act_ = new QAction(QIcon(":icons/save_icon"), tr("Save"), this);
  pimpl->save_act_->setStatusTip(tr("Save the config file to disk."));
  connect(pimpl->save_act_, SIGNAL(triggered()), this, SLOT(saveFile()));

  toolbar_->addAction(pimpl->new_act_);
  toolbar_->addAction(pimpl->open_act_);
  toolbar_->addAction(pimpl->save_act_);

  // メニューの書記化
  connect(action_new_, SIGNAL(triggered()), this, SLOT(newFile()));
  connect(action_open_, SIGNAL(triggered()), this, SLOT(openFile()));
  connect(action_save_, SIGNAL(triggered()), this, SLOT(saveFile()));
  connect(action_save_as_, SIGNAL(triggered()), this, SLOT(saveFileAs()));
  connect(action_release_, SIGNAL(triggered()), this, SLOT(releaseData()));
  connect(action_preview_, SIGNAL(triggered()), this, SLOT(previewClock()));
  connect(action_about_, SIGNAL(triggered()), this, SLOT(aboutApplication()));

  // 初期状態では、保存不可、リリース不可にしておく
  pimpl->setActionsEnabled(this, false);

  // 画像の更新 Widget は、ツリーがクリックされるまでは、disable
  pimpl->setUpdateEnabled(this, false);

  // フォームの初期化
  pimpl->initializeForm(this);

  // ツリー構成の取得
  pimpl->treeTagInitialize(this);

  // ファイルが指定されていれば、読み出し
  if (! config_file.isEmpty()) {
    pimpl->loadArchiveFile(this, config_file.toStdString());
    pimpl->setSaveEnabled(this, false);
  }

  // サブツリーを開く
  pimpl->expandSubTreeNode(this);
}


ThemeConfigWindow::~ThemeConfigWindow(void) {
}


void ThemeConfigWindow::closeEvent(QCloseEvent* event) {

  // 保存の必要があれば、ダイアログを表示して質問する
  // !!! save が enable であれば、保存可能とみなしてよい

  // !!! Qt 本の P66 のあたりを参考のこと

  // サブウィンドウを閉じる
  pimpl->preview_draw_.close();

  // !!! とりあえず、終了
  event->accept();
}


// 新規
void ThemeConfigWindow::newFile(void) {

  // プレビューを閉じる
  pimpl->
preview_draw_.close();

  // 空の名前でファイル読み出しを指示し、設定が空のオブジェクトを作成する
  pimpl->loadArchiveFile(this, "");
}


// 開く
void ThemeConfigWindow::openFile(void) {

  QString fileName =
    QFileDialog::getOpenFileName(this, tr("Open archive list file."),
				 fromStdStringPath(pimpl->work_dir_.c_str()),
                                 tr("archive list (*.txt)"));

  if (! fileName.isEmpty()) {

    // プレビューを閉じる
    pimpl->preview_draw_.close();

    // 日本語のパスを含む場合への対処
    fileName = toStdStringPath(fileName);

    pimpl->loadArchiveFile(this, fileName.toStdString());
    pimpl->setSaveEnabled(this, false);
  }
}


// 保存
void ThemeConfigWindow::saveFile(void) {

  if (! pimpl->archive_config_) {
    return;
  }

  const std::string archive_name = pimpl->archive_config_->getArchiveFileName();
  if (archive_name.empty()) {
    // 名前がなければ、名前を付けて保存
    saveFileAs();

  } else {
    pimpl->save(this, archive_name);
  }
}


// 名前を付けて保存
void ThemeConfigWindow::saveFileAs(void) {

  QString fileName =
    QFileDialog::getSaveFileName(this, tr("Save archive list file."),
				 fromStdStringPath(pimpl->work_dir_.c_str()),
                                 tr("archive list (*.txt)"));

  if (! fileName.isEmpty()) {
    // 日本語のパスを含む場合への対処
    fileName = toStdStringPath(fileName);

    pimpl->save(this, fileName.toStdString().c_str());
  }
}


// リリース
void ThemeConfigWindow::releaseData(void) {

  // リリース名、キー値の設定
  size_t base_length = pimpl->work_dir_.size();
  const std::string name = pimpl->archive_config_->getReleaseName();
  const std::string key = pimpl->archive_config_->getRrdaKey();
  pimpl->release_dialog_.
    setDefaultValue(name.substr(base_length, std::string::npos), key);

  // ダイアログの表示
  pimpl->release_dialog_.show();
  pimpl->release_dialog_.raise();
}


// リリースの実処理
void ThemeConfigWindow::releaseHandler(const QString& file,
                                       const QString& key) {

  // 現在の値を退避
  const std::string pre_file = pimpl->archive_config_->getReleaseName();
  const std::string pre_key = pimpl->archive_config_->getRrdaKey();

  // 登録情報の更新
  pimpl->archive_config_->setReleaseName(file.toStdString());
  pimpl->archive_config_->setRrdaKey(key.toStdString());

  // 値が変更されていたら、保存可能にする
  if (pre_file.compare(pimpl->archive_config_->getReleaseName()) ||
      pre_key.compare(pimpl->archive_config_->getRrdaKey())) {
    pimpl->setSaveEnabled(this, true);
  }

  pimpl->release();
}


// プレビュー
void ThemeConfigWindow::previewClock(void) {

  if (! pimpl->redraw_timer_.isActive()) {
    pimpl->previewOpened(this);
  } else {
    pimpl->preview_draw_.close();
    pimpl->previewClosed(this);
  }
}


// プレビューの再描画
void ThemeConfigWindow::previewRedraw(void) {

  pimpl->redrawPreview(this);
}


// プレビュー表示オフセットの変更
void ThemeConfigWindow::offsetUpdated(int value) {
  static_cast<void>(value);

  pimpl->redrawPreview(this);
}


// このアプリケーションについて
void ThemeConfigWindow::aboutApplication(void) {

  QMessageBox::about(this, tr("About qtmclock_config"),
                     tr("<h2>Qtmclock Config</h2>"
                        "<p>Theme making tool for Qtmclock.</p>"
                        "<p>Report bugs to "
                        "&lt;satofumi@users.sourceforge.jp&gt;</p>"));
}


// 針の中心を、揃えるか、の設定
void ThemeConfigWindow::centerCheckHandler(bool checked) {

  pimpl->same_hands_center_ = checked;
  pimpl->updateItems();
  pimpl->setSaveEnabled(this, true);
}


// コロンを点滅させるか、の設定
void ThemeConfigWindow::blinkCheckHandler(bool checked) {

  pimpl->setting_config_->setValue("digital/colon", checked ? "on" : "off");
  pimpl->applyCheckChanged(this);
}


// 秒針を表示するか、の設定
void ThemeConfigWindow::analogCheckHandler(bool checked) {

  pimpl->setting_config_->setValue("analog/sec", checked ? "on" : "off");
  pimpl->applyCheckChanged(this);
}


// デジタル文字の秒を表示するか、の設定
void ThemeConfigWindow::digitalCheckHandler(bool checked) {

  pimpl->setting_config_->setValue("digital/sec", checked ? "on" : "off");
  pimpl->applyCheckChanged(this);
}


// カレンダーで月を表示するか、の設定
void ThemeConfigWindow::monthCheckHandler(bool checked) {

  pimpl->setting_config_->setValue("calendar/month", checked ? "on" : "off");
  pimpl->applyCheckChanged(this);
}


// 画像更新用のイベント処理
void ThemeConfigWindow::imageUploadHandler(void) {

  QList<QTreeWidgetItem*> list = parts_tree_->selectedItems();
  if (list.empty() || (list.size() > 1)) {
    // 選択が１つでなければ、戻る
    // !!! 同時に操作できるようにするあたりは、また今度検討する
    return;
  }

  QTreeWidgetItem* item = list.front();
  ItemToName::iterator p = pimpl->item_to_name_.find(item);
  if (p == pimpl->item_to_name_.end()) {
    // タグが見つからなければ戻る
    return;
  }
  std::string tag = p.value();

  // カーソルのイベントハンドラが、現在のタグに従って処理を行えるように登録
  pimpl->current_tag_ = tag;

  // 補助線の再描画
  pimpl->removeAdditionalLine();
  pimpl->drawAdditionalLine(pimpl->current_tag_);
  pimpl->updateItems();

  size_t n = sizeof(ImageTags) / sizeof(ImageTags[0]);
  bool found = pimpl->findItemTag(ImageTags, n, pimpl->current_tag_);
  if (! found) {
    // 画像の以外ならば、アップロードフォームを無効化
    image_lineedit_->setText("");
    pimpl->setUpdateEnabled(this, false);

  } else {
    // 画像のアップロードフォームを有効化
    image_lineedit_->setText(pimpl->name_to_item_[tag]->text(1));
    pimpl->setUpdateEnabled(this, true);
  }
}


void ThemeConfigWindow::leftPressed(void) {

  pimpl->updatePosition(this, -1, 0, pimpl->current_tag_);
}


void ThemeConfigWindow::rightPressed(void) {

  pimpl->updatePosition(this, +1, 0, pimpl->current_tag_);
}


void ThemeConfigWindow::upPressed(void) {

  pimpl->updatePosition(this, 0, -1, pimpl->current_tag_);
}


void ThemeConfigWindow::downPressed(void) {

  pimpl->updatePosition(this, 0, +1, pimpl->current_tag_);
}



// 部品画像の更新、文字列入力
void ThemeConfigWindow::partsEditHandler(void) {

  pimpl->updateItemName(this, image_lineedit_->text());
}


// 部品画像の更新、選択
void ThemeConfigWindow::updateHandlelr(void) {

  QString fileName =
    QFileDialog::getOpenFileName(this, tr("Open parts file."),
				 fromStdStringPath(pimpl->work_dir_.c_str()),
                                 tr("parts file (*.png)"));

  if (! fileName.isEmpty()) {
    // 日本語のパスを含む場合への対処
    fileName = toStdStringPath(fileName);

    pimpl->updateItemName(this, fileName);
  }
}


// プレビューのクローズイベント処理
void ThemeConfigWindow::previewCloseHandler(void) {

  pimpl->previewClosed(this);
}


// クリックによる部品の選択処理
void ThemeConfigWindow::partsClickedHandler(int x, int y) {

  QGraphicsItem* clicked_item = pimpl->scene_->itemAt(x, y);
  if (! clicked_item) {
    // 背景がクリックされた場合は、選択をクリアする
    pimpl->clearItemSelection(this);
    return;
  }

  // クリックされた部品を検索し、その部品を選択する
  std::string found_tag;
  for (Items::iterator it = pimpl->items_.begin();
       it != pimpl->items_.end(); ++it) {
    if (it.value() == clicked_item) {
      found_tag = it.key();
      break;
    }
  }
  if (found_tag.empty() || (! found_tag.compare("base/base_org"))) {
    // 登録外のタグ、またはベース画像ならば、現在の選択を解除する
    pimpl->clearItemSelection(this);
    return;
  }

  // 指定された項目のツリーを開く
  QTreeWidgetItem* tree_item = pimpl->name_to_item_.value(found_tag, NULL);
  if (tree_item) {
    parts_tree_->setCurrentItem(tree_item);
  }
}


// !!! ドラッグ & ドロップで画像ファイルを変更できるようにする
// !!! 画像でなければ、実際には登録しない
// !!! slot


// !!! Qt Demo の 画像のパズルを参考に、部品配置を変更するあたりを実装する
