/*!
  \file
  \brief 部品描画ウィジット

  \author Satofumi KAMIMURA

  $Id$
*/

#include <QPainter>
#include <QPaintEvent>
#include "PartsDrawWidget.h"
#include "PartsConfig.h"


struct PartsDrawWidget::pImpl {

  PartsConfig parts_;
  QRect back_rect_;

  void paint(QPaintEvent* event, PartsDrawWidget* parent) {

    std::vector<QString>& tags = parts_.tags;
    std::vector<QSize> tags_size;

    QPainter painter(parent);

    painter.fillRect(event->rect(), QBrush(QColor(Qt::white)));

    // フォント、スタイルの適用
    QFont font = QFont(parts_.font_name.c_str());
    QFont::Style style = PartsConfig::fontStyle(parts_.font_style_name);
    QFont::Weight weight = PartsConfig::fontWeight(parts_.font_style_name);
    font.setWeight(weight);
    font.setStyle(style);
    font.setPixelSize(parts_.font_size);
    painter.setFont(font);

    QColor back_color = PartsConfig::encodeColor(parts_.back_color);
    painter.setBackground(QBrush(back_color));
    painter.setBackgroundMode(Qt::OpaqueMode);

    // 描画タグの最大長を求める
    QFontMetrics font_metrics(font);
    size_t max_width = 0;
    size_t max_height = 0;
    for (std::vector<QString>::iterator it = tags.begin();
         it != tags.end(); ++it) {
      size_t width = font_metrics.width(*it);
      size_t height = font_metrics.height();
      tags_size.push_back(QSize(width, height));

      // !!! qmax を使うべきかも
      if (width > max_width) {
        max_width = width;
      }
      if (height > max_height) {
        max_height = height;
      }
    }

    // 背景の描画
    enum {
      DrawOffset_X = 10,
      DrawOffset_Y = 10,
    };
    size_t offset = parts_.offset;
    back_rect_ = QRect(DrawOffset_X, DrawOffset_Y - 1,
                       (offset + max_width) * tags.size(), max_height + 2);
    painter.fillRect(back_rect_, QBrush(back_color));

    // フォントの描画
    int descent = font_metrics.descent();
    QColor fore_color = PartsConfig::encodeColor(parts_.fore_color);
    painter.setPen(QPen(fore_color));
    size_t index = 0;
    for (std::vector<QString>::iterator it = tags.begin();
         it != tags.end(); ++it, ++index) {
      size_t width = tags_size[index].width();
      size_t x = (max_width + offset) * index;
      painter.drawText((offset / 2) + DrawOffset_X
                       + (max_width / 2) - (width / 2) + x,
                       DrawOffset_Y + max_height - descent, *it);
    }
  }

  bool outputPixmap(const QString& file_name, PartsDrawWidget* parent) {
    if ((back_rect_.width() == 0) || (back_rect_.height() == 0)) {
      // 画像が有効でない
      return false;
    }

    // ファイル書きだし
    QPixmap image = QPixmap::grabWidget(parent, back_rect_);
    return image.save(file_name, "PNG");
  }
};


PartsDrawWidget::PartsDrawWidget(QWidget* parent)
  : QWidget(parent), pimpl(new pImpl) {
}


PartsDrawWidget::~PartsDrawWidget(void) {
}


void PartsDrawWidget::paintEvent(QPaintEvent* event) {

  pimpl->paint(event, this);
}


void PartsDrawWidget::updatePreview(const PartsConfig& parts) {

  pimpl->parts_ = parts;
  update();
}


bool PartsDrawWidget::outputPixmap(const QString& file_name) {

  return pimpl->outputPixmap(file_name, this);
}
