/*!
  \file
  \brief 部品生成ツールのウィンドウ管理

  \author Satofumi KAMIMURA

  $Id: PartsGeneratorWindow.cpp 263 2009-01-30 14:48:10Z satofumi $

  \todo コンポーネントの変更を、PartsConfig に反映させる
*/

#include "PartsGeneratorWindow.h"
#include "PartsDrawWidget.h"
#include "GenerateConfigHandler.h"
#include "PartsConfig.h"
#include "ConvertStdStringPath.h"
#include <QSettings>
#include <QCloseEvent>
#include <QMessageBox>
#include <QFontDatabase>
#include <QButtonGroup>
#include <QFontComboBox>
#include <QColorDialog>
#include <QFileDialog>
#include <QFileInfo>

using namespace qrk;
using namespace std;


struct PartsGeneratorWindow::pImpl
{
  QAction* new_act_;
  QAction* open_act_;
  QAction* save_act_;
  PartsDrawWidget* parts_draw_;
  QButtonGroup* button_group_;
  QComboBox* font_combobox_;
  GenerateConfigHandler parts_configs_;
  string work_file_;
  QString first_title_;

  pImpl(const QString& config_file, PartsGeneratorWindow* parent)
    : parts_draw_(NULL)
  {
    if (config_file.isEmpty()) {
      // デフォルト設定で初期化
      initializeConfig(parent);

    } else {
      // 指定ファイルの内容を読み出し
      if (loadConfigFile(config_file.toStdString().c_str(), parent)) {
        QFileInfo fi(config_file);
        work_file_ = fi.absoluteFilePath().toStdString();
      }
    }
  }


  void initializeForm(PartsGeneratorWindow* parent)
  {
    // Window Title の取得
    first_title_ = parent->windowTitle();

    // アイコンの適用
    parent->setWindowIcon(QIcon(":icons/parts_generator_icon"));

    // フォントツリーの作成
    initializeFontTree(parent);

    // 画像描画ウィジットの配置
    parts_draw_ = new PartsDrawWidget(parent->centralwidget);
    parent->scrollarea_->setWidget(parts_draw_);

    // フォントの選択変更イベント
    // !!! isChanged() で、表示中のフォントが変更されるようにする
    // !!! この処理は、Undo したいかも

    // メニューイベント
    connect(parent->action_quit_, SIGNAL(triggered()), parent, SLOT(close()));
    connect(parent->action_new_, SIGNAL(triggered()), parent, SLOT(newFile()));
    connect(parent->action_open_, SIGNAL(triggered()),
            parent, SLOT(openFile()));
    connect(parent->action_save_, SIGNAL(triggered()),
            parent, SLOT(saveFile()));
    connect(parent->action_save_as_, SIGNAL(triggered()),
            parent, SLOT(saveAsFile()));

    // 画像ボタンのイベントで設定の表示を変更し、
    // 設定の変更で、描画ウィジットの画像を更新する
    // !!! って感じのイベントを設定

    // ボタンをグループ化
    button_group_ = new QButtonGroup(parent);
    button_group_->addButton(parent->digital_button_, 0);
    button_group_->addButton(parent->number_button_, 1);
    button_group_->addButton(parent->week_button_, 2);
    button_group_->addButton(parent->separator_button_, 3);

    connect(button_group_, SIGNAL(buttonClicked(int)),
            parent, SLOT(update(void)));

    // 最後に押されていたボタンを押下させる
    // !!! とりあえず、最初のにしておく。デフォルト設定で

    // !!! 不要ならば、除去する
    parent->update();

    // 出力イベント処理
    connect(parent->output_button_, SIGNAL(clicked()),
            parent, SLOT(outputPixmap()));

    // フォーム操作イベント
    connect(parent->font_size_spinbox_, SIGNAL(valueChanged(int)),
            parent, SLOT(fontSizeChangedHandler(int)));
    connect(parent->offset_spinbox_, SIGNAL(valueChanged(int)),
            parent, SLOT(offsetChangedHandler(int)));
    connect(parent->fore_color_button_, SIGNAL(clicked()),
            parent, SLOT(fontColorClickedHandler()));
    connect(parent->back_color_button_, SIGNAL(clicked()),
            parent, SLOT(backgroundColorClickedHandler()));

    connect(parent->font_tree_, SIGNAL(itemSelectionChanged()),
            parent, SLOT(fontChangedHandler()));
    connect(parent->font_tree_, SIGNAL(itemClicked(QTreeWidgetItem*,int)),
            parent, SLOT(fontChangedHandler()));

    // ファイル名が有効ならば、保存不可にしておく
    if (! work_file_.empty()) {
      setSaveEnabled(false);
    }
  }


  // 設定をフォームに反映
  void loadPartsConfig(const PartsConfig& parts, PartsGeneratorWindow* parent)
  {
    parent->output_label_->setText((parts.type + ".png").c_str());

    // !!! フォントを変更する
    parent->font_size_spinbox_->setValue(parts.font_size);
    updateColorButton(parent->fore_color_button_, parts.fore_color);
    updateColorButton(parent->back_color_button_, parts.back_color);
    parent->offset_spinbox_->setValue(parts.offset);
  }


  void updateColorButton(QToolButton* button, const string& color_str)
  {
    QColor fore_color = PartsConfig::encodeColor(color_str);
    QPixmap fore_pixmap(32, 32);
    fore_pixmap.fill(fore_color);
    button->setIcon(QIcon(fore_pixmap));
  }


  // プレビュー画像の描画
  void drawPreviewPixmap(const PartsConfig& parts)
  {
    parts_draw_->updatePreview(parts);
  }


  // フォントツリーの作成
  void initializeFontTree(PartsGeneratorWindow* parent)
  {
    QFontDatabase database;
    foreach (QString family, database.families()) {
      const QStringList styles = database.styles(family);
      if (styles.isEmpty()) {
        continue;
      }

      QTreeWidgetItem* family_item = new QTreeWidgetItem(parent->font_tree_);
      family_item->setText(0, family);

      foreach (QString style, styles) {
        QTreeWidgetItem* style_item = new QTreeWidgetItem(family_item);
        style_item->setText(0, style);
        style_item->setData(0, Qt::UserRole,
                            QVariant(database.weight(family, style)));
        style_item->setData(0, Qt::UserRole + 1,
                            QVariant(database.italic(family, style)));
      }
    }
  }


  // 設定の読み出し
  void readSettings(PartsGeneratorWindow* parent)
  {
    QSettings settings("Hyakuren Soft LTD.", "parts_generator");
    parent->restoreGeometry(settings.value("geometry").toByteArray());
  }


  // 設定の保存
  void writeSettings(PartsGeneratorWindow* parent)
  {
    QSettings settings("Hyakuren Soft LTD.", "parts_generator");
    settings.setValue("geometry", parent->saveGeometry());
  }


  // デフォルト設定で初期化
  void initializeConfig(PartsGeneratorWindow* parent)
  {
    parts_configs_.initialize();
    work_file_ = "";
    setTitle("", parent);
  }


  // 設定ファイルの読み込み
  bool loadConfigFile(const char* file, PartsGeneratorWindow* parent)
  {
    bool ret = parts_configs_.load(file);
    if (ret) {
      // タイトルの更新
      QFileInfo fi(file);
      setTitle(fi.fileName(), parent);
    }
    return ret;
  }


  PartsConfig& getCurrentPartsConfig(void)
  {
    int index = button_group_->checkedId();
    return parts_configs_.getPartsConfig(index);
  }


  void setSaveEnabled(bool enable)
  {
    save_act_->setEnabled(enable);
  }


  bool getCurrentSaveEnable()
  {
    return save_act_->isEnabled();
  }


  void setTitle(const QString& title, PartsGeneratorWindow* parent)
  {
    if (! title.isEmpty()) {
      parent->setWindowTitle(first_title_ + " -- " + title);
    } else {
      parent->setWindowTitle(first_title_);
    }
  }
};


PartsGeneratorWindow::PartsGeneratorWindow(const QString& config_file)
  : QMainWindow(), pimpl(new pImpl(config_file, this))
{
  setupUi(this);
  pimpl->readSettings(this);

  // 以下の初期化は、pImpl 内で行うと tr() が使えないため、ここで行っている

  // ツールバーの初期化
  pimpl->new_act_ = new QAction(QIcon(":icons/new_icon"), tr("New"), this);
  pimpl->new_act_->setStatusTip(tr("Create a new config file."));
  connect(pimpl->new_act_, SIGNAL(triggered()), this, SLOT(newFile()));

  pimpl->open_act_ = new QAction(QIcon(":icons/open_icon"), tr("Open"), this);
  pimpl->open_act_->setStatusTip(tr("Open an exist config file."));
  connect(pimpl->open_act_, SIGNAL(triggered()), this, SLOT(openFile()));

  pimpl->save_act_ = new QAction(QIcon(":icons/save_icon"), tr("Save"), this);
  pimpl->save_act_->setStatusTip(tr("Save the config file to disk."));
  connect(pimpl->save_act_, SIGNAL(triggered()), this, SLOT(saveFile()));

  toolbar_->addAction(pimpl->new_act_);
  toolbar_->addAction(pimpl->open_act_);
  toolbar_->addAction(pimpl->save_act_);

  // フォントツリーのタイトル設定
  font_tree_->setColumnCount(1);
  font_tree_->setHeaderLabels(QStringList() << tr("Font"));

  // フォームの初期化
  pimpl->initializeForm(this);

  // メニューの初期化
  connect(action_about_, SIGNAL(triggered()), this, SLOT(aboutApplication()));
}


PartsGeneratorWindow::~PartsGeneratorWindow(void)
{
  pimpl->writeSettings(this);
}


// 終了
void PartsGeneratorWindow::closeEvent(QCloseEvent* event)
{
  event->accept();
}


// 新規作成
void PartsGeneratorWindow::newFile(void)
{
  pimpl->initializeConfig(this);
  update();
}


// 読み出し
void PartsGeneratorWindow::openFile(void)
{
  QString fileName =
    QFileDialog::getOpenFileName(this, tr("Open parts config file."),
				 toQtStringPath(pimpl->work_file_.c_str()),
                                 tr("config file (*.txt)"));

  if (! fileName.isEmpty()) {
    string std_path = toStdStringPath(fileName);
    if (pimpl->loadConfigFile(std_path.c_str(), this)) {
      update();
    }
  }
}


// 保存
void PartsGeneratorWindow::saveFile(void)
{
  if (! pimpl->parts_configs_.save()) {
    // 保存できなければ、ファイル名の入力を促した上で保存する
    saveAsFile();
  } else {
    pimpl->setSaveEnabled(false);
  }
}


bool PartsGeneratorWindow::saveAsFile(void)
{
  fprintf(stderr, "saveAsFile()\n");

  // ファイル名の入力を促し、保存を試みる
  QString fileName =
    QFileDialog::getSaveFileName(this, tr("Save parts config file."),
                                 toQtStringPath(pimpl->work_file_.c_str()),
                                 tr("config file (*.txt)"));

  if (! fileName.isEmpty()) {
    // 日本語のパスを含む場合への対処
    string std_path = toStdStringPath(fileName);
    if (pimpl->parts_configs_.saveAs(std_path.c_str())) {
      pimpl->setSaveEnabled(false);
      QFileInfo fi(std_path.c_str());
      pimpl->setTitle(fi.fileName(), this);
      return true;
    }
  }

  return false;
}


// このアプリケーションについて
void PartsGeneratorWindow::aboutApplication(void)
{
  QMessageBox::about(this, tr("About parts_generator"),
                     tr("<h2>Parts generator</h2>"
                        "<p>Theme making sub tool for Qtmclock.</p>"
                        "<p>Report bugs to "
                        "&lt;satofumi@users.sourceforge.jp&gt;</p>"));
}


// フォーム設定とプレビュー画像の更新
void PartsGeneratorWindow::update(void)
{
  int index = pimpl->button_group_->checkedId();
  if (index >= static_cast<int>(pimpl->parts_configs_.size())) {
    return;
  }

  bool current_enable = pimpl->getCurrentSaveEnable();
  const PartsConfig& parts = pimpl->parts_configs_.getPartsConfig(index);
  pimpl->loadPartsConfig(parts, this);
  pimpl->drawPreviewPixmap(parts);
  pimpl->setSaveEnabled(current_enable);
}


// 画像ファイルの出力
void PartsGeneratorWindow::outputPixmap(void)
{
  QFileInfo fi(pimpl->work_file_.c_str());
  QString output_file =
    fi.absoluteDir().absolutePath() + "/" + output_label_->text();
  fprintf(stderr, "%s\n", output_file.toStdString().c_str());
  pimpl->parts_draw_->outputPixmap(output_file);
}


// フォントサイズの変更
void PartsGeneratorWindow::fontSizeChangedHandler(int value)
{
  PartsConfig& parts = pimpl->getCurrentPartsConfig();
  parts.font_size = value;

  pimpl->drawPreviewPixmap(parts);
  pimpl->setSaveEnabled(true);
}


void PartsGeneratorWindow::offsetChangedHandler(int value)
{
  PartsConfig& parts = pimpl->getCurrentPartsConfig();
  parts.offset = value;

  pimpl->drawPreviewPixmap(parts);
  pimpl->setSaveEnabled(true);
}


void PartsGeneratorWindow::fontColorClickedHandler(void)
{
  PartsConfig& parts = pimpl->getCurrentPartsConfig();
  QColor fore_color = PartsConfig::encodeColor(parts.fore_color);
  QColor color = QColorDialog::getColor(fore_color, this);
  if (! color.isValid()) {
    // キャンセルされた
    return;
  }

  parts.fore_color =
    PartsConfig::decodeColor(color.red(), color.green(), color.blue());

  pimpl->updateColorButton(fore_color_button_, parts.fore_color);
  pimpl->drawPreviewPixmap(parts);
  pimpl->setSaveEnabled(true);
}


void PartsGeneratorWindow::backgroundColorClickedHandler(void)
{
  // !!! このあたりは、上と同じ。くくるべき
  PartsConfig& parts = pimpl->getCurrentPartsConfig();
  QColor back_color = PartsConfig::encodeColor(parts.back_color);
  QColor color = QColorDialog::getColor(back_color, this);
  if (! color.isValid()) {
    // キャンセルされた
    return;
  }

  parts.back_color =
    PartsConfig::decodeColor(color.red(), color.green(), color.blue());

  pimpl->updateColorButton(back_color_button_, parts.back_color);
  pimpl->drawPreviewPixmap(parts);
  pimpl->setSaveEnabled(true);
}


// フォントの変更
void PartsGeneratorWindow::fontChangedHandler(void)
{
  QTreeWidgetItem* item = font_tree_->currentItem();
  QTreeWidgetItem* parent = item->parent();

  // 選択されたフォント名、スタイル名の取得
  int column = font_tree_->currentColumn();
  string font_name =
    ((parent) ? parent->text(column) : item->text(column)).toStdString();
  string style_name = (parent) ? item->text(column).toStdString() : "";

  PartsConfig& parts = pimpl->getCurrentPartsConfig();
  parts.font_name = font_name;
  parts.font_style_name = style_name;

  pimpl->drawPreviewPixmap(parts);
  pimpl->setSaveEnabled(true);
}
