#
# $Id: Ipmsg::Host.pm,v 0.10 2004/02/06 Mikio Fukushima Exp Mikio Fukushima $
#

package Ipmsg::Host;

use Ipmsg::Common;
use strict;

use vars qw( $VERSION $THISNAME
	@ISA @EXPORT @EXPORT_OK
);

require Exporter;
require AutoLoader;

@ISA	= qw( Exporter AutoLoader Ipmsg::HostList Ipmsg::Common );
@EXPORT	= qw();

# --------------------
# Set global variables in this module.
#

	$VERSION	= 0.09;
	$THISNAME	= "Ipmsg Host module Ver.$VERSION";

	use constant Code		=> 0;
	use constant PeerAddr	=> 1;
	use constant PeerPort	=> 2;

	use constant PROTO_VER	=> 3;
	use constant User		=> 4;
	use constant Host		=> 5;
	use constant Command	=> 6;
	use constant NickName	=> 7;
	use constant Group		=> 8;

	use constant Agent			=> 9;
	use constant AbsenceHead	=> 10;
	use constant AbsenceStr		=> 11;
	use constant AbsenceStrNot	=> 12;

	use constant ABSENCEOPT		=> 13;
	use constant SERVEROPT		=> 14;
	use constant DIALUPOPT		=> 15;
	use constant FILEATTACHOPT	=> 16;
	use constant ENCRYPTOPT		=> 17;

	use constant Pubkey			=> 18;
	use constant Exponent		=> 19;
	use constant EncryptPower	=> 20;

	use constant RSA			=> 21;
	use constant RC2			=> 22;
	use constant BLOWFISH		=> 23;
	use constant SIGN_MD5		=> 24;

	use constant UpdateTime		=> 25;

	my %_index = (
		Code			=> Code,

		PeerAddr		=> PeerAddr,
		PeerPort		=> PeerPort,
		PROTO_VER		=> PROTO_VER,
		User			=> User,
		Host			=> Host,
		Command			=> Command,
		NickName		=> NickName,
		Group			=> Group,

		Agent			=> Agent,
		AbsenceHead		=> AbsenceHead,
		AbsenceStr		=> AbsenceStr,
		AbsenceStrNot	=> AbsenceStrNot,

		ABSENCEOPT		=> ABSENCEOPT,
		SERVEROPT		=> SERVEROPT,
		DIALUPOPT		=> DIALUPOPT,
		FILEATTACHOPT	=> FILEATTACHOPT,
		ENCRYPTOPT		=> ENCRYPTOPT,

		Pubkey			=> Pubkey,
		Exponent		=> Exponent,
		EncryptPower	=> EncryptPower,

		RSA				=> RSA,
		RC2				=> RC2,
		BLOWFISH		=> BLOWFISH,
		SIGN_MD5		=> SIGN_MD5,

		UpdateTime		=> UpdateTime,
	);

# --------------------
# Category: Constructor

# --------------------
# Name: new

sub new
{
	my $class = shift;
	   $class = ref( $class ) || $class;

	my %arg = @_;
	my $self = [];

	# --------------------
	# initialize

	@{ $self }[ values %_index ] = @arg{ keys %_index };

	# --------------------
	# set default

	$self->[ PeerAddr ]		= '0.0.0.0'						if( ! defined $self->[ PeerAddr ]	|| $self->[ PeerAddr ]	eq "" );
	$self->[ PeerPort ]		= $class->SUPER::default_port	if( ! defined $self->[ PeerPort ]	|| $self->[ PeerPort ]	eq "" );

	$self->[ PROTO_VER ]	= $class->SUPER::proto_ver	if( ! defined $self->[ PROTO_VER ]	|| $self->[ PROTO_VER ]	eq "" );
	$self->[ User ]			= $class->delim_dummy		if( ! defined $self->[ User ]		|| $self->[ User ]		eq "" );
	$self->[ Host ]			= $class->delim_dummy		if( ! defined $self->[ Host ]		|| $self->[ Host ]		eq "" );
	$self->[ Command ]		= 0							if( ! defined $self->[ Command ]	|| $self->[ Command ]	eq "" );
	$self->[ NickName ]		= $class->delim_dummy		if( ! defined $self->[ NickName ]	|| $self->[ NickName ]	eq "" );
	$self->[ Group ] 		= $class->delim_dummy		if( ! defined $self->[ Group ]		|| $self->[ Group ]		eq "" );

	$self->[ Agent ] 		= '' if( ! defined $self->[ Agent ] );
	$self->[ AbsenceHead ]	= '' if( ! defined $self->[ AbsenceHead ] );
	$self->[ AbsenceStr ]	= '' if( ! defined $self->[ AbsenceStr ] );
	$self->[ AbsenceStrNot ]= '' if( ! defined $self->[ AbsenceStrNot ] );

	$self->[ Pubkey ] 		= '' if( ! defined $self->[ Pubkey ] );
	$self->[ Exponent ] 	= '';
	$self->[ EncryptPower ]	= '';

	$self->[ RSA ] 			= '';
	$self->[ RC2 ] 			= '';
	$self->[ BLOWFISH ] 	= '';
	$self->[ SIGN_MD5 ] 	= '';

	$self->[ ABSENCEOPT ]	= 0;
	$self->[ SERVEROPT ]	= 0;
	$self->[ DIALUPOPT ]	= 0;
	$self->[ FILEATTACHOPT ]= 0;
	$self->[ ENCRYPTOPT ]	= 0;

	$self->[ UpdateTime ]	= time();

	$self->[ Code ]			= 1;

	# --------------------
	# bless

	bless $self, $class;

	# --------------------
	# command init

	$self->_lengthcheck;
	$self->command( $self->[ Command ] );
	$self->_encryptpower_init;

	# --------------------
	# return

	return $self;
}

# --------------------
# Category: Access methods

# --------------------
# Name: Base information

# --------------------
# Sub1: addr

sub addr
{
	my ( $self, $options ) = @_;

	# --------------------
	# update nickname name

	if( defined $options ){

		$self->[ PeerAddr ] = $options;
		$self->_updatetime;

		return $self;
	}

	# --------------------
	# return

	return $self->[ PeerAddr ];
}

# --------------------
# Sub1: port

sub port
{
	my ( $self, $options ) = @_;

	# --------------------
	# update nickname name

	if( defined $options ){

		$self->[ PeerPort ] = $options;
		$self->_updatetime;

		return $self;
	}

	# --------------------
	# return

	return $self->[ PeerPort ];
}

# --------------------
# Sub1: proto_ver

sub proto_ver
{
	my ( $self, $options ) = @_;

	# --------------------
	# update nickname name

	if( defined $options ){

		$self->[ PROTO_VER ] = $options;
		$self->_updatetime;

		return $self;
	}

	# --------------------
	# return

	return $self->[ PROTO_VER ];
}

# --------------------
# Sub1: user

sub user
{
	my ( $self, $options ) = @_;

	# --------------------
	# update nickname name

	if( defined $options ){

		$self->[ User ] = $options;
		$self->_updatetime;
		$self->_lengthcheck;

		return $self;
	}

	# --------------------
	# return

	return $self->[ User ];
}

# --------------------
# Sub1: host

sub host
{
	my ( $self, $options ) = @_;

	# --------------------
	# update nickname name

	if( defined $options ){

		$self->[ Host ] = $options;
		$self->_updatetime;
		$self->_lengthcheck;

		return $self;
	}

	# --------------------
	# return

	return $self->[ Host ];
}

# --------------------
# Sub1: command

sub command
{
	my ( $self, $options ) = @_;

	# --------------------
	# substitution

	if( defined $options ){ 

		$self->[ Command ] = $options;

		# --------------------
		# options init

		$self->_command_init->_updatetime;

		return $self;

	}else{

		return $self->[ Command ];

	}

}

# --------------------
# Sub2: _command_init

sub _command_init
{
	my ( $self ) = shift;

	# --------------------
	# init

	$self->[ ABSENCEOPT ]	= $self->get_opt( $self->[ Command ] ) & $self->ipmsg_cmd( 'IPMSG_ABSENCEOPT' );	# 256
	$self->[ SERVEROPT ]	= $self->get_opt( $self->[ Command ] ) & $self->ipmsg_cmd( 'IPMSG_SERVEROPT' );		# 512
	$self->[ DIALUPOPT ]	= $self->get_opt( $self->[ Command ] ) & $self->ipmsg_cmd( 'IPMSG_DIALUPOPT' );		# 63336
	$self->[ FILEATTACHOPT ]= $self->get_opt( $self->[ Command ] ) & $self->ipmsg_cmd( 'IPMSG_FILEATTACHOPT' );	# 2097152
	$self->[ ENCRYPTOPT ]	= $self->get_opt( $self->[ Command ] ) & $self->ipmsg_cmd( 'IPMSG_ENCRYPTOPT' );	# 4194304

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub1: group

sub group
{
	my ( $self, $options ) = @_;

	# --------------------
	# update nickname name

	if( defined $options ){

		$self->[ Group ] = $options;
		$self->_updatetime;
		$self->_lengthcheck;

		return $self;
	}

	# --------------------
	# return

	return $self->[ Group ];
}

# --------------------
# Sub1: nickname

sub nickname
{
	my ( $self, $options ) = @_;

	# --------------------
	# update nickname name

	if( defined $options ){

		$self->[ NickName ] = $options;
		$self->_updatetime;
		$self->_lengthcheck;

		return $self;
	}

	# --------------------
	# return

	return $self->[ NickName ];
}

# --------------------
# Name: Other information

# --------------------
# Sub1: agent

sub agent
{
	my ( $self, $options ) = @_;

	# --------------------
	# update agent name

	if( defined $options ){

		$self->[ Agent ] = $options;
		$self->_updatetime;
		$self->_lengthcheck;

		return $self;
	}

	# --------------------
	# return

	return $self->[ Agent ];
}

# --------------------
# Sub1: absencehead

sub absencehead
{
	my ( $self, $options ) = @_;

	# --------------------
	# update agent name

	if( defined $options ){

		$self->[ AbsenceHead ] = $options;
		$self->_updatetime;
		$self->_lengthcheck;

		return $self;
	}

	# --------------------
	# return

	return $self->[ AbsenceHead ];
}

# --------------------
# Sub1: absencestr

sub absencestr
{
	my ( $self, $options ) = @_;

	# --------------------
	# update agent name

	if( defined $options ){

		$self->[ AbsenceStr ] = $options;
		$self->_updatetime;
		$self->_lengthcheck;

		return $self;
	}

	# --------------------
	# return

	return $self->[ AbsenceStr ];
}

# --------------------
# Sub1: absencestrnot

sub absencestrnot
{
	my ( $self, $options ) = @_;

	# --------------------
	# update agent name

	if( defined $options ){

		$self->[ AbsenceStrNot ] = $options;
		$self->_updatetime;
		$self->_lengthcheck;

		return $self;
	}

	# --------------------
	# return

	return $self->[ AbsenceStrNot ];
}

# --------------------
# Name: Ź沽Ϣ

# --------------------
# Sub1: pubkey

sub pubkey
{
	my ( $self, $options ) = @_;

	# --------------------
	# update pubkey name

	if( defined $options ){

		my ( $EncryptPower, $Exponent, $key );

		( $EncryptPower, $key )	= split( /:/, $options );
		( $Exponent, $key )		= split( /-/, $key );

		#print "POWER => ", $EncryptPower,"\n";
		#print "EXP   => ", $Exponent,"\n";
		#print "KEY   => ", $key,"\n";

		$self->[ EncryptPower ]	= $EncryptPower;
		$self->[ Pubkey ]		= $key;
		$self->[ Exponent ]		= $Exponent;

		$self->_encryptpower_init;
		$self->_updatetime;

		# --------------------
		# return

		return $self;
	}

	# --------------------
	# return

	return ( $self->[ EncryptPower ], $self->[ Exponent ], $self->[ Pubkey ],);
}

# --------------------
# Sub2: _encryptpower_init

sub _encryptpower_init
{
	my ( $self ) = shift;

	# --------------------
	# init

	$self->[ RSA ]		= 512	if hex( $self->[ EncryptPower ] ) & $self->ipmsg_cmd( 'IPMSG_RSA_512' );
	$self->[ RSA ]		= 1024	if hex( $self->[ EncryptPower ] ) & $self->ipmsg_cmd( 'IPMSG_RSA_1024' );
	$self->[ RSA ]		= 2048	if hex( $self->[ EncryptPower ] ) & $self->ipmsg_cmd( 'IPMSG_RSA_2048' );

	$self->[ RC2 ]		= 40	if hex( $self->[ EncryptPower ] ) & $self->ipmsg_cmd( 'IPMSG_RC2_40' );
	$self->[ RC2 ]		= 128	if hex( $self->[ EncryptPower ] ) & $self->ipmsg_cmd( 'IPMSG_RC2_128' );
	$self->[ RC2 ]		= 256	if hex( $self->[ EncryptPower ] ) & $self->ipmsg_cmd( 'IPMSG_RC2_256' );

	$self->[ BLOWFISH ]	= 128	if hex( $self->[ EncryptPower ] ) & $self->ipmsg_cmd( 'IPMSG_BLOWFISH_128' );
	$self->[ BLOWFISH ]	= 256	if hex( $self->[ EncryptPower ] ) & $self->ipmsg_cmd( 'IPMSG_BLOWFISH_256' );

	$self->[ SIGN_MD5 ]	= 1		if hex( $self->[ EncryptPower ] ) & $self->ipmsg_cmd( 'IPMSG_SIGN_MD5' );

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub1: rsa

sub rsa
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ RSA ];
}

# --------------------
# Sub1: rc2

sub rc2
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ RC2 ];
}

# --------------------
# Sub1: blowfish

sub blowfish
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ BLOWFISH ];
}

# --------------------
# Sub1: md5

sub md5
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ SIGN_MD5 ];
}

# --------------------
# Name: Status flags
#
#	ABSENCEOPT
#	SERVEROPT
#	DIALUPOPT
#	FILEATTACHOPT
#	ENCRYPTOPT

# --------------------
# Sub1: absence

sub absence
{
	my ( $self, $options ) = @_;

	# --------------------
	# return

	if( ! defined $options ){ return $self->[ ABSENCEOPT ]; }

	# --------------------
	# toggle
	#
	# enable

	if( $options and ! $self->[ ABSENCEOPT ] ){

		$self->command( $self->command + $self->ipmsg_cmd( 'IPMSG_ABSENCEOPT' ) );
		$self->_updatetime;

	# disable

	}elsif( $options and $self->[ ABSENCEOPT ] ){

		$self->command( $self->command - $self->ipmsg_cmd( 'IPMSG_ABSENCEOPT' ) );
		$self->_updatetime;

	}

	# --------------------
	# return

	return $self->[ ABSENCEOPT ];
}

# --------------------
# Sub1: server

sub server
{
	my ( $self, $options ) = @_;

	# --------------------
	# return

	if( ! defined $options ){ return $self->[ SERVEROPT ]; }

	# --------------------
	# toggle
	#
	# enable

	if( $options and ! $self->[ SERVEROPT ] ){

		$self->command( $self->command + $self->ipmsg_cmd( 'IPMSG_SERVEROPT' ) );
		$self->_updatetime;

	# disable

	}elsif( $options and $self->[ SERVEROPT ] ){

		$self->command( $self->command - $self->ipmsg_cmd( 'IPMSG_SERVEROPT' ) );
		$self->_updatetime;

	}

	# --------------------
	# return

	return $self->[ SERVEROPT ];
}

# --------------------
# Sub1: dialup

sub dialup
{
	my ( $self, $options ) = @_;

	# --------------------
	# return

	if( ! defined $options ){ return $self->[ DIALUPOPT ]; }

	# --------------------
	# toggle
	#
	# enable

	if( $options and ! $self->[ DIALUPOPT ] ){

		$self->command( $self->command + $self->ipmsg_cmd( 'IPMSG_DIALUPOPT' ) );
		$self->_updatetime;

	# disable

	}elsif( $options and $self->[ DIALUPOPT ] ){

		$self->command( $self->command - $self->ipmsg_cmd( 'IPMSG_DIALUPOPT' ) );
		$self->_updatetime;

	}

	# --------------------
	# return

	return $self->[ DIALUPOPT ];
}

# --------------------
# Sub1: fileattach

sub fileattach
{
	my ( $self, $options ) = @_;

	# --------------------
	# return

	if( ! defined $options ){ return $self->[ FILEATTACHOPT ]; }

	# --------------------
	# toggle
	#
	# enable

	if( $options and ! $self->[ FILEATTACHOPT ] ){

		$self->command( $self->command + $self->ipmsg_cmd( 'IPMSG_FILEATTACHOPT' ) );
		$self->_updatetime;

	# disable

	}elsif( $options and $self->[ FILEATTACHOPT ] ){

		$self->command( $self->command - $self->ipmsg_cmd( 'IPMSG_FILEATTACHOPT' ) );
		$self->_updatetime;

	}

	# --------------------
	# return

	return $self->[ FILEATTACHOPT ];
}

# --------------------
# Sub1: encrypt

sub encrypt
{
	my ( $self, $options ) = @_;

	# --------------------
	# return

	if( ! defined $options ){ return $self->[ ENCRYPTOPT ]; }

	# --------------------
	# toggle
	#
	# enable

	if( $options and ! $self->[ ENCRYPTOPT ] ){

		$self->command( $self->command + $self->ipmsg_cmd( 'IPMSG_ENCRYPTOPT' ) );
		$self->_updatetime;

	# disable

	}elsif( $options and $self->[ ENCRYPTOPT ] ){

		$self->command( $self->command - $self->ipmsg_cmd( 'IPMSG_ENCRYPTOPT' ) );
		$self->_updatetime;

	}

	# --------------------
	# return

	return $self->[ ENCRYPTOPT ];
}


# --------------------
# Name: updatetime

sub updatetime
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ UpdateTime ];
}

# --------------------
# Name: _updatetime

sub _updatetime
{
	my $self = shift;

	# --------------------
	# update

	$self->[ UpdateTime ] = time();

	# --------------------
	# return

	return $self;
}

# --------------------
# Name: _lengthcheck

sub _lengthcheck
{
	my $self = shift;

	# --------------------
	# chop over $MAX_NAMEBUF( 50bytes ) strings.

	$self->[ User ]			= substr( $self->[ User ],			0, $self->SUPER::namebuf );
	$self->[ Host ]			= substr( $self->[ Host ],			0, $self->SUPER::namebuf );
	$self->[ Group ]		= substr( $self->[ Group ],			0, $self->SUPER::namebuf );
	$self->[ NickName ]		= substr( $self->[ NickName ],		0, $self->SUPER::namebuf );
	$self->[ Agent ]		= substr( $self->[ Agent ],			0, $self->SUPER::namebuf );
	$self->[ AbsenceHead ]	= substr( $self->[ AbsenceHead ],	0, $self->SUPER::namebuf );
	$self->[ AbsenceStr ]	= substr( $self->[ AbsenceStr ],	0, $self->SUPER::namebuf );
	$self->[ AbsenceStrNot ]= substr( $self->[ AbsenceStrNot ],	0, $self->SUPER::namebuf );

	# --------------------
	# return

	return $self;
}

# --------------------
# Category: Methods

# --------------------
# Name: as_hostlist
#
# user1:host1:65536:192.168.0.0:30985:NickName1:Group1:

sub as_hostlist
{
	my ( $self, %options ) = @_;

	# --------------------
	# check requirements

	$options{ Port }		= 0	if ! defined $options{ Port };
	$options{ DropCommand }	= 0	if ! defined $options{ DropCommand };

	# --------------------
	# convert port number endian

	my $port;
	if( $options{ Port } ){
		$port = $self->port_L2B( $self->[ PeerPort ] );
	}else{
		$port = $self->[ PeerPort ];
	}

	# --------------------
	# set IPMSG Command and option

	my $command;
	if( $options{ DropCommand } ){
		$command = $self->get_opt( $self->[ Command ] ),
	}else{
		$command = $self->[ Command ];
	}

	# --------------------
	# make hostlist

	my $tmp = join( $self->delim_hostlist,
		$self->[ User ],
		$self->[ Host ],
		$command,
		$self->[ PeerAddr ],
		$port,
		$self->[ NickName ],
	);

	if( $self->[ ABSENCEOPT ] )	{ $tmp .= $self->[ AbsenceHead ]; }
	if( $self->[ Group ] )		{ $tmp .= $self->delim_hostlist . $self->[ Group ]; }

	# --------------------
	# return

	return $tmp;
}

# --------------------
# Name: as_string
#
# user1:host1:65536:192.168.0.0:30985:NickName1:Group1:

sub as_string
{
	my ( $self, %options ) = @_;

	# --------------------
	# return

	return $self->SUPER::as_string( $self->as_hostlist( %options ) );
}

# --------------------
# Category: Setting object

# --------------------
# Name: put_value

sub put_value
{
	my $self= shift;
	my %arg	= @_;

	# --------------------
	# check requirements

	return undef if ! %arg;

	while( my ( $name, $value ) = each %arg ){
		$self->[ $_index{ $name } ] = $value;
	}

	$self->_updatetime;

	# --------------------
	# return

	return $self;
}

# --------------------
# Name: get_value

sub get_value
{
	my ( $self, $arg ) = @_;

	# --------------------
	# return

	return ( wantarray ? sort keys %_index : $self->[ $_index{ $arg } ] );
}

# --------------------
# Name: display

sub display
{
	my $self = shift;

	# --------------------
	# print

	foreach my $i ( sort keys %_index ){
		print "$i=>",$self->[ $_index{ $i } ],"\n";
	}

}

# --------------------
# Name: code

sub code
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ Code ];
}

# --------------------
# Name: _index

sub _index
{
	my ( $self, $index ) = @_;

	# --------------------
	# return

	return $_index{ $index };
}

1;

