#
# $Id: Ipmsg::HostList.pm,v 0.10 2004/02/09 Mikio Fukushima Exp Mikio Fukushima $
#

package Ipmsg::HostList;
use Ipmsg::Host;

use strict;

use vars qw( $VERSION $THISNAME
	@ISA @EXPORT @EXPORT_OK
);

require Exporter;
require AutoLoader;

@ISA	= qw( Exporter AutoLoader Ipmsg::Common );
@EXPORT	= qw();

# --------------------
# Set global variables in this module.
#

	$VERSION	= 0.09;
	$THISNAME	= "Ipmsg HostList module Ver.$VERSION";

	use constant Code		=> 0;

	use constant UID		=> 1;
	use constant EXT		=> 2;

	use constant HostList	=> 3;
	use constant UpdateTime	=> 4;

	use constant StaticPort	=> 5;

	my %_index = (
		Code	=> Code,

		UID			=> UID,
		EXT			=> EXT,

		HostList	=> HostList,
		UpdateTime	=> UpdateTime,

		StaticPort	=> StaticPort,

	);

# --------------------
# Category: Constructor

# --------------------
# Name: new

sub new
{
	my $class = shift;
	   $class = ref( $class ) || $class;

	my %arg = @_;
	my $self = [];

	# --------------------
	# initialize

	@{ $self }[ values %_index ] = @arg{ keys %_index };

	# --------------------
	# set default

	# --------------------
	# set default values on properties

	$self->[ UID ]		||= '';
	$self->[ EXT ]		||= '';

	# --------------------
	# initialize list and variables

	$self->[ HostList ]		= {};
	$self->[ UpdateTime ]	= time();
	$self->[ Code ]			= 1;

	# --------------------
	# bless

	bless $self, $class;

	# --------------------
	# command init

	return $self;
}

# --------------------
# Name: flush

sub flush
{
	my ( $self ) = @_;

	# --------------------
	# initialize lists and counters.

	$self->[ HostList ]		= {};
	$self->[ Code ]			= 1;
	$self->_updatetime;

	# --------------------
	# return

	return 1;
}

# --------------------
# Category: Access methods

# --------------------
# Name: uid

sub uid
{
	my $self = shift;

	# --------------------
	# return

	@_	? $self->[ UID ] = shift
		: $self->[ UID ];
}

# --------------------
# Name: ext

sub ext
{
	my $self = shift;

	# --------------------
	# return

	@_	? $self->[ UID ] = shift
		: $self->[ UID ];
}

# --------------------
# Name: total

sub total
{
	my ( $self, $group ) = @_;
	return ( scalar $self->get( Group => $group ) );
}

# --------------------
# Name: staticport

sub staticport
{
	my $self = shift;

	# --------------------
	# return

	@_	? $self->[ StaticPort ] = shift
		: $self->[ StaticPort ];
}

# --------------------
# Name: updatetime

sub updatetime
{
	my $self = shift;
	return $self->[ UpdateTime ];
}

# --------------------
# Name: _updatetime

sub _updatetime
{
	my $self = shift;

	# --------------------
	# update

	$self->[ UpdateTime ] = time();

	# --------------------
	# return

	return $self;
}

# --------------------
# Category: Methods

# --------------------
# Name: add

sub add
{
	my ( $self, %options ) = @_;

	# --------------------
	# check requirements

	return undef if ! defined $options{ PeerAddr };
	return undef if ! defined $options{ PeerPort };

	$options{ PeerPort } = $self->staticport if $self->staticport;

	# --------------------
	# add host

	my $host = Ipmsg::Host->new( %options );

	$self->[ HostList ]{ "$options{ PeerAddr }:$options{ PeerPort }" } = $host;

	$self->_updatetime;

	# --------------------
	# return

	return $host;
}

# --------------------
# Name: get

sub get
{
	my ( $self, %options ) = @_;

	# --------------------
	# check required

	if( ! defined $options{ NoGroup } ){ $options{ NoGroup } = 1; }

	# --------------------
	# called by scalar context with PeerAddr and PeerPort, return Host.

	if( $options{ PeerAddr } && $options{ PeerPort } ){

		return $self->[ HostList ]{ $options{ PeerAddr } . ":" . $options{ PeerPort } };

	# --------------------
	# called by list context ( with Group ), return all matching object.

	}elsif( $options{ Group } ){

		my @tmp;
		foreach my $i ( values %{ $self->[ HostList ] } ){

			if( $i->group eq $options{ Group } || ( $options{ NoGroup } && $i->group eq '' ) ){
				push @tmp, $i;
			}

		}

		return ( wantarray ? @tmp : scalar @tmp );

	# --------------------
	# called by scalar context, return number of Host.

	}elsif( ! wantarray ){

		return scalar keys %{ $self->[ HostList ] };

	# --------------------
	# called by list context, return all object.

	}elsif( wantarray ){

		return values %{ $self->[ HostList ] };

	}

}

# --------------------
# Name: del

sub del
{
	my ( $self, %options ) = @_;

	# --------------------
	# check requirements

	return undef if ! defined $options{ PeerAddr };
	return undef if ! defined $options{ PeerPort };

	$self->_updatetime;

	# --------------------
	# del host, return

	return delete $self->[ HostList ]{ "$options{ PeerAddr }:$options{ PeerPort }" };
}

# --------------------
# Name: get_addrlist

sub get_addrlist
{
	my $self = shift;

	# --------------------
	# make address list

	my @tmp;
	foreach my $i ( values %{ $self->[ HostList ] } ){ push @tmp, $i->addr, $i->port; }

	# --------------------
	# set code

	$self->[ Code ] = scalar @tmp / 2;

	# --------------------
	# return

	return ( wantarray ? @tmp : scalar @tmp / 2 );
}

# --------------------
# Name: get_grouplist

sub get_grouplist
{
	my $self = shift;

	# --------------------
	# make grouplist

	my %tmp;
	foreach my $i ( values %{ $self->[ HostList ] } ){ $tmp{ $i->group } = 1; }

	# --------------------
	# delete 0 length groupname

	delete $tmp{ "" };

	# --------------------
	# set code

	$self->[ Code ] = scalar keys %tmp;

	# --------------------
	# return

	return ( wantarray ? sort keys %tmp : scalar keys %tmp );
}

# --------------------
# Name: get_hostlist
#
# user1:host1:65536:192.168.0.0:30985:NickName1:Group1:

sub get_hostlist
{
	my ( $self, %options ) = @_;

	# --------------------
	# check requirements

	return undef if ! $self->get;

	$options{ Count }		= 1				if ! defined $options{ Count };
	$options{ Count }		= 1				if $options{ Count } <= 0;
	$options{ Length }		= $self->udpbuf	if ! defined $options{ Length };
	$options{ DropCommand }	= 0				if ! defined $options{ DropCommand };

	# --------------------
	# make hostlist

	my $tmp			= '';
	my @tmp			= $self->get( Group => $options{ Group } );

	my $totalhost	= 0;
	my $nexthost	= 0;
	my $continue	= 0;

	for( my $i = $options{ Count } - 1; $i <= $#tmp; $i++ ){

		# --------------------
		# over Length?

		if( length( $tmp ) + length( $tmp[ $i ]->as_hostlist ) <= $options{ Length } ){

			$tmp .= $tmp[ $i ]->as_hostlist( Port => 1, DropCommand => $options{ DropCommand } ) . $self->delim_hostlist;

		}else{

			$continue = 1;
			last;

		}

		# --------------------
		# count up totalhost

		$totalhost++;

	}

	# --------------------
	# set counter

	if( $continue ){ $nexthost = $totalhost + $options{ Count }; }

	# --------------------
	# set code

	$self->[ Code ] = $totalhost;

	# --------------------
	# return

	return ( $nexthost, $totalhost, $tmp );
}

# --------------------
# Name: add_hostlist
#
# user1:host1:65536:192.168.0.0:30985:NickName1:Group1:

sub add_hostlist
{
	my ( $self, $list, %options ) = @_;

	# --------------------
	# check requirement

	if( ! defined $list ){ return undef; }

	$options{ DropCommand }	= 0				if ! defined $options{ DropCommand };

	print "DL => $options{ DropCommand }\n";

	# --------------------
	# add host

	my	@hostlist = split( $self->delim_hostlist, $list );

	my	$NextNumber		=  shift @hostlist;
	my	$NumberOfHosts	=  shift @hostlist;
		$NextNumber		=~ s/\s//g;
		$NumberOfHosts	=~ s/\s//g;

	my @tmp;

	for( my $i = 0; $i <= $#hostlist; $i = $i + 7 ){

		# --------------------
		# set IPMSG Command and option

		my $command;
		if( $options{ DropCommand } ){
			$command = $self->get_opt( $hostlist[ $i + 2 ] ),
		}else{
			$command = $hostlist[ $i + 2 ];
		}

		push @tmp, $self->add(
			User		=> $hostlist[ $i ],
			Host		=> $hostlist[ $i + 1 ],
			Command		=> $command,
			PeerAddr	=> $hostlist[ $i + 3 ],
			PeerPort	=> $self->port_L2B( $hostlist[ $i + 4 ] ),
			NickName	=> $hostlist[ $i + 5 ],
			Group		=> $hostlist[ $i + 6 ],
		);

	}

	# --------------------
	# set code

	$self->[ Code ] = $NextNumber;

	# --------------------
	# return

	return ( wantarray ? @tmp : ( $#hostlist + 1 ) / 7 );
}

# --------------------
# Name: make_hostlistformat
#

sub make_hostlistformat
{
	my ( $self, $nexthost, $totalhost, $list ) = @_;

	# --------------------
	# check requirement

	if( ! defined $nexthost or ! defined $totalhost or ! defined $list ){ return undef; }

	# --------------------
	# return

	return sprintf( "%5d%s%5d%s%s",
		$nexthost ? $nexthost : 0,
		$self->delim_hostlist,
		$totalhost,
		$self->delim_hostlist,
		$list
	);

}

# --------------------
# Category: Setting object

# --------------------
# Name: put_value

sub put_value
{
	my $self= shift;
	my %arg	= @_;

	# --------------------
	# check requirements

	return undef if ! %arg;

	while( my ( $name, $value ) = each ( %arg ) ){
		$self->[ $_index{ $name } ] = $value;
	}

	# --------------------
	# return

	return $self;
}

# --------------------
# Name: get_value

sub get_value
{
	my ( $self, $arg ) = @_;

	# --------------------
	# return

	return ( wantarray ? sort keys %_index : $self->[ $_index{ $arg } ] );
}

# --------------------
# Name: display

sub display
{
	my $self = shift;

	# --------------------
	# print

	foreach my $i ( sort keys %_index ){
		print "$i=>",$self->[ $_index{ $i } ],"\n";
	}

}

# --------------------
# Name: code

sub code
{
	my $self = shift;

	# --------------------
	# return

	@_	? $self->[ Code ] = shift
		: $self->[ Code ];
}

# --------------------
# Name: _index

sub _index
{
	my ( $self, $index ) = @_;

	# --------------------
	# return

	return $_index{ $index };
}

1;

