// QASOutput.h
// 2008/12/12

#pragma once

namespace QAX {

// QASVorbisOutputImpl
class QASVorbisOutputImpl : public QDecoderOutput {

	IStream* m_pStream;

	QVorbisDecoder* m_Decoder;

	BYTE*  m_Buffer;
	SIZE_T m_BufferSize;

	const BYTE* m_BufferStart;
	const BYTE* m_BufferEnd;

	UINT64 m_Position;

	QASHeader m_Header;

	BYTE*  m_MetaData;
	SIZE_T m_MetaSize;

	const BYTE* m_Setup;
	const BYTE* m_SetupEnd;

	const BYTE* m_IndexStart;
	const BYTE* m_IndexEnd;

	UINT64 m_Current;

	UINT32 m_PendingSamples;

	const INT16* m_SampleBuffer;

	UINT32 m_SampleUnit;

	BYTE m_PageFlags;

	bool m_EndOfStream;

	UINT32* m_Lens;
	UINT32  m_LensSize;

	const UINT32* m_PosLens;
	const UINT32* m_EndLens;

	IndexArray m_Index;

	UINT32 m_DecodeOffset;

	enum {
		LEN_SIZE = 32,

		INIT_BUFFER_SIZE = 0x1000
	};

	enum {
		FLAGS_START = 0x01,
		FLAGS_END   = 0x02
	};

private:

	QASVorbisOutputImpl() :
		m_pStream(0),
		m_Decoder(0),
		m_Buffer(0),
		m_BufferSize(0),
		m_BufferStart(0),
		m_BufferEnd(0),
		m_Position(0),
		m_MetaData(0),
		m_MetaSize(0),
		m_Setup(0),
		m_SetupEnd(0),
		m_IndexStart(0),
		m_IndexEnd(0),
		m_Current(0),
		m_PendingSamples(0),
		m_SampleBuffer(0),
		m_SampleUnit(0),
		m_PageFlags(0),
		m_EndOfStream(false),
		m_DecodeOffset(0),
		m_Lens(0),
		m_LensSize(0),
		m_PosLens(0),
		m_EndLens(0)
	{
		ZeroMemory(&m_Header, sizeof(m_Header));
	}

public:

	static QDecoderOutput* Create(
		QDecoderFactory* pFactory,
		IStream*         pStream)
	{
		QASVorbisOutputImpl* impl = new QASVorbisOutputImpl();

		try {
			impl->Setup(pStream);

			impl->SetupDecoder(pFactory);

			impl->Prepare(pStream);

		} catch (...) {
			delete impl;
			throw;
		}

		return impl;
	}

private:

	void Setup(IStream* p)
	{
		SetupHeader(p);

		SetupMetaData(p);
	}

	void SetupHeader(IStream* pStream)
	{
		ULONG cb = 0;
		HRESULT hRslt = pStream->Read(
			&m_Header,
			sizeof(m_Header),
			&cb);
		if (FAILED(hRslt)) {
			COMError::Throw("IStream::Read", hRslt);
		}

		if (cb != sizeof(m_Header)) {
			FormatError::Throw("QASHeader.EOS");
		}

		if (memcmp(m_Header.Signature, "QAS1", 4) != 0) {
			FormatError::Throw("QASHeader.Signature");
		}

		if (memcmp(m_Header.Format, "vorbis", 6) != 0) {
			FormatError::Throw("QASHeader.Format");
		}
	}

	void SetupMetaData(IStream* pStream)
	{
		m_MetaData = static_cast<BYTE*>(malloc(m_Header.MetaDataSize));
		if (m_MetaData == 0) {
			DecoderError::Throw("OutOfMemory");
		}

		m_MetaSize = m_Header.MetaDataSize;

		LARGE_INTEGER pos;
		ULARGE_INTEGER newPos;

		pos.QuadPart = m_Header.MetaDataPos;
		HRESULT hRslt = pStream->Seek(
			pos,
			STREAM_SEEK_SET,
			&newPos);
		if (FAILED(hRslt)) {
			COMError::Throw("IStream::Seek", hRslt);
		}

		ULONG cb = 0;
		hRslt = pStream->Read(
			m_MetaData,
			m_MetaSize,
			&cb);
		if (FAILED(hRslt)) {
			COMError::Throw("IStream::Read", hRslt);
		}

		if (cb != m_MetaSize) {
			FormatError::Throw("MetaData.EOS");
		}

		const BYTE* index = m_MetaData + m_Header.MetaDataIndex;
		if (index < m_MetaData || index >= m_MetaData + m_MetaSize) {
			FormatError::Throw("MetaData.Index");
		}

		const BYTE* p   = index;
		const BYTE* end = m_MetaData + m_MetaSize;

		UINT32 count;
		p += DecodeVNumber(p, end, &count);

		if (count != 2) {
			FormatError::Throw("MetaData.Count");
		}

		UINT32 setup_len;
		p += DecodeVNumber(p, end, &setup_len);

		UINT32 index_len;
		p += DecodeVNumber(p, end, &index_len);

		m_Setup      = m_MetaData;
		m_SetupEnd   = m_MetaData + setup_len;
		m_IndexStart = m_SetupEnd;
		m_IndexEnd   = m_IndexStart + index_len;

		if (m_SetupEnd > m_MetaData + m_MetaSize) {
			FormatError::Throw("MetaData.Setup.Size");
		}

		if (m_IndexEnd > m_MetaData + m_MetaSize) {
			FormatError::Throw("MetaData.Index.Size");
		}
	}

	void SetupDecoder(
		QDecoderFactory* pFactory)
	{
		SetupTable::SetupEntry setup = SetupTable::GetSetup(
			m_Setup, m_SetupEnd);

		BYTE key[20];
		CopyMemory(key, setup.Hash, 20);

		QVorbisDecoder* decoder = pFactory->CreateVorbisDecoder(
			key,
			setup.Id,
			setup.IdSize,
			setup.Setup,
			setup.SetupSize);

		m_Decoder = decoder;
	}

	void Prepare(IStream* pStream)
	{
		m_pStream = pStream;

		m_pStream->AddRef();

		LARGE_INTEGER pos;
		ULARGE_INTEGER newPos;

		pos.QuadPart = m_Header.ContentPos;
		HRESULT hRslt = m_pStream->Seek(
			pos,
			STREAM_SEEK_SET,
			&newPos);
		if (FAILED(hRslt)) {
			COMError::Throw("IStream::Seek", hRslt);
		}

		m_Position = 0;

		m_Buffer = static_cast<BYTE*>(malloc(INIT_BUFFER_SIZE));
		if (m_Buffer == 0) {
			DecoderError::Throw("OutOfMemory");
		}

		m_BufferSize = INIT_BUFFER_SIZE;

		m_BufferStart = m_Buffer;
		m_BufferEnd   = m_Buffer;

		m_Current = 0;

		m_PendingSamples = 0;
		m_SampleBuffer   = 0;

		QVorbisDecoderSetup_t* setup = m_Decoder->GetSetup();

		m_SampleUnit = QV_GetDecoderSetupChannels(setup);

		m_PageFlags = 0;

		m_EndOfStream = false;

		m_DecodeOffset = 0;

		m_Lens = static_cast<UINT32*>(malloc(LEN_SIZE * sizeof(UINT32)));
		if (m_Lens == 0) {
			DecoderError::Throw("OutOfMemory");
		}

		m_LensSize = LEN_SIZE;

		m_PosLens = 0;
		m_EndLens = 0;
	}

public:

	~QASVorbisOutputImpl()
	{
		if (m_pStream != 0) {
			m_pStream->Release();
			m_pStream = 0;
		}

		if (m_Buffer != 0) {
			free(m_Buffer);
			m_Buffer = 0;
		}

		if (m_MetaData != 0) {
			free(m_MetaData);
			m_MetaData = 0;
		}

		if (m_Lens != 0) {
			free(m_Lens);
			m_Lens = 0;
		}

		if (m_Decoder != 0) {
			delete m_Decoder;
			m_Decoder = 0;
		}
	}

	/* */

	virtual void STDCALL Release()
	{
		delete this;
	}

	/* QDecoderOutput */

	virtual QDecoderSpec STDCALL GetDecoderSpec()
	{
		QDecoderSpec spec = { 0 };

		QVorbisDecoderSetup_t* setup = m_Decoder->GetSetup();

		spec.Channels     = QV_GetDecoderSetupChannels(setup);
		spec.SamplingRate = QV_GetDecoderSetupSamplingRate(setup);

		spec.Samples = m_Header.Samples;

		return spec;
	}

	virtual UINT32 STDCALL Decode(
		VOID*  buffer,
		UINT32 samples)
	{
		UINT32 output = 0;

		while (!m_EndOfStream) {
			if (m_PendingSamples > 0) {
				UINT32 count = samples;
				if (count > m_PendingSamples) {
					count = m_PendingSamples;
				}

				CopyMemory(
					buffer,
					m_SampleBuffer,
					count * m_SampleUnit * sizeof(INT16));

				m_SampleBuffer   += count * m_SampleUnit;
				m_PendingSamples -= count;

				m_Current += count;

				output = count;
				break;
			}

			if (m_PosLens >= m_EndLens) {
				ParsePageHeader();
			}

			if (m_PosLens < m_EndLens) {
				DecodePacket();
			}
		}

		return output;
	}

	virtual void STDCALL Seek(
		UINT64 sample)
	{
		if (m_Index.IsEmpty()) {
			m_Index.ParseIndex(
				m_IndexStart,
				m_IndexEnd - m_IndexStart,
				m_Header.Samples);
		}

		const IndexEntry& e = m_Index.SeekIndex(sample);

		m_Position = e.Position;

		LARGE_INTEGER pos;
		ULARGE_INTEGER newPos;

		pos.QuadPart = m_Header.ContentPos + m_Position;
		HRESULT hRslt = m_pStream->Seek(
			pos,
			STREAM_SEEK_SET,
			&newPos);
		if (FAILED(hRslt)) {
			COMError::Throw("IStream::Seek", hRslt);
		}

		m_BufferStart = m_Buffer;
		m_BufferEnd   = m_Buffer;

		ParsePageHeader();

		SeekInPage(sample, e);
	}

	virtual std::string STDCALL QueryProperty(
		const char* name)
	{
		DecoderError::Throw("NotImplemented");
	}

	/* */

private:

	bool ReadStream(UINT32 needed)
	{
		SIZE_T cb0 = (m_Buffer + m_BufferSize) - m_BufferStart;
		if (needed > cb0) {
			return false;
		}

		SIZE_T cb1 = (m_Buffer + m_BufferSize) - m_BufferEnd;

		ULONG cb = 0;
		HRESULT hRslt = m_pStream->Read(
			(m_Buffer + m_BufferSize) - cb1,
			ULONG(cb1),
			&cb);
		if (FAILED(hRslt)) {
			COMError::Throw("IStream::Read", hRslt);
		}

		m_BufferEnd += cb;

		return true;
	}

	void ReadBuffer(UINT32 needed)
	{
		SIZE_T cb = m_BufferEnd - m_BufferStart;
		if (cb >= needed) {
			return;
		}

		if (ReadStream(needed)) {
			return;
		}

		if (m_BufferStart > m_Buffer) {
			MoveMemory(
				m_Buffer,
				m_BufferStart,
				cb);

			m_BufferStart = m_Buffer;
			m_BufferEnd   = m_BufferStart + cb;
		}

		if (ReadStream(needed)) {
			return;
		}

		UINT32 new_size = ((needed + INIT_BUFFER_SIZE - 1) / INIT_BUFFER_SIZE) * INIT_BUFFER_SIZE;

		BYTE* new_buf = static_cast<BYTE*>(realloc(
			m_Buffer,
			new_size));
		if (new_buf == 0) {
			DecoderError::Throw("OutOfMemory");
		}

		m_Buffer     = new_buf;
		m_BufferSize = new_size;

		m_BufferStart = m_Buffer;
		m_BufferEnd   = m_BufferStart + cb;

		if (!ReadStream(needed)) {
			DecoderError::Throw("Unexpected.ReadBuffer");
		}
	}

	void ParsePageHeader()
	{
		if (m_Position >= m_Header.ContentSize) {
			m_EndOfStream = true;
			return;
		}

		ReadBuffer(INIT_BUFFER_SIZE);

		const BYTE* p = m_BufferStart;

		m_PageFlags = *(p++);

		UINT32 lens;
		p += DecodeVNumber(p, m_BufferEnd, &lens);

		if (lens > m_LensSize) {
			UINT32* new_len = static_cast<UINT32*>(realloc(
				m_Lens,
				lens * sizeof(UINT32)));
			if (new_len == 0) {
				DecoderError::Throw("OutOfMemory");
			}

			m_Lens     = new_len;
			m_LensSize = lens;
		}

		for (UINT32 i = 0; i < lens; i++) {
			p += DecodeVNumber(p, m_BufferEnd, m_Lens + i);
		}

		m_PosLens = m_Lens;
		m_EndLens = m_Lens + lens;

		SIZE_T start = p - m_BufferStart;

		SIZE_T size = start;
		for (UINT32 i = 0; i < lens; i++) {
			size += m_Lens[i];
		}

		ReadBuffer(size + sizeof(UINT32));

		if (m_BufferStart + size + sizeof(UINT32) > m_BufferEnd) {
			FormatError::Throw("Page.Size.OutOfRange");
		}

		CRC32 crc32;
		UINT32 crc = crc32.Generate(m_BufferStart, size);

		UINT32 val;
		CopyMemory(&val, m_BufferStart + size, sizeof(UINT32));

		if (val != crc) {
			FormatError::Throw("Page.CRC.Error");
		}

		m_BufferStart += start;
		m_Position    += start;
	}

	/* */

	void DecodePacket()
	{
		bool bLastPacket = (m_PosLens == (m_EndLens - 1));

		QVorbisDecoder_t* decoder = m_Decoder->GetDecoder();

		QV_Output_t output = { 0 };
		if (!QV_DecodeFrame(
			decoder,
			m_BufferStart,
			m_PosLens[0],
			&output)) {
			DecoderError::Throw("QV_DecodeFrame");
		}

		UINT32 samples = output.Length;
		if (samples > 0) {
			QV_Convert_t convert = { 0 };
			if (!QV_ConvertFrame(
				decoder,
				&output,
				&convert)) {
				DecoderError::Throw("QV_ConvertFrame");
			}

			if (m_DecodeOffset > samples) {
				FormatError::Throw("DecoderOffset.Invalid");
			}

			m_SampleBuffer   = convert.Sample + m_DecodeOffset * m_SampleUnit;
			m_PendingSamples = samples - m_DecodeOffset;

			m_DecodeOffset = 0;

			if ((m_PageFlags & FLAGS_END) != 0 && bLastPacket) {
				UINT64 rest = m_Header.Samples - m_Current;
				if (rest > samples) {
					DecoderError::Throw("Invalid.Samples");
				}
				m_PendingSamples = UINT32(rest);
			}
		}

		m_BufferStart += m_PosLens[0];
		m_Position    += m_PosLens[0];
		m_PosLens     += 1;

		if (bLastPacket) {
			m_BufferStart += sizeof(UINT32); /* CRC32 */
			m_Position    += sizeof(UINT32);
		}
	}

	/* */

	void SeekInPage(
		UINT64            pos,
		const IndexEntry& e)
	{
		QVorbisDecoder_t* decoder = m_Decoder->GetDecoder();

		QV_ResetDecoder(decoder);
		QV_ResetDecoderChecker(decoder);

		m_PendingSamples = 0;

		m_EndOfStream = false;

		const UINT32* p   = m_PosLens;
		const UINT32* end = m_EndLens;

		if (p >= end) {
			FormatError::Throw("Page.Invalid");
		}

		const BYTE* frame = m_BufferStart;
		SIZE_T      size  = p[0];

		INT32 samples = 0;
		if (!QV_CheckDecoderChecker(
			decoder,
			frame,
			size,
			&samples)) {
			DecoderError::Throw("QV_CheckDecoderChecker");
		}
		if (samples != 0) {
			FormatError::Throw("Check.Invalid");
		}

		const BYTE* n_frame = frame + size;

		UINT64 c_pos = e.Sample + e.Offset;

		if (pos < c_pos) {
			FormatError::Throw("Page.Position.Invalid");
		}

		while (p < end - 1) {
			SIZE_T n_size = p[1];

			if (!QV_CheckDecoderChecker(
				decoder,
				n_frame,
				n_size,
				&samples)) {
				DecoderError::Throw("QV_CheckDecoderChecker");
			}

			if (pos < c_pos + samples) {
				break;
			}

			c_pos += samples;

			frame = n_frame;

			n_frame += n_size;

			p += 1;
		}

		SIZE_T offs = frame - m_BufferStart;

		m_BufferStart += offs;
		m_Position    += offs;

		m_PosLens = p;

		m_Current = pos;

		m_DecodeOffset = UINT32(pos - c_pos);
	}

	/* */

}; // QASVorbisOutputImpl

} // namespace QAX

