/* VorbisDecoder.c */
/* 2008/11/06      */

#include "StdAfx.h"

#include "VorbisDecoder.h"

#include "FrameDecoder.h"

#include "SamplesChecker.h"

#include "FrameConverter.h"

/* */

static MemoryPool_t s_pool;

/* */

static void Unintialize(void)
{
	TransformDecoder_Uninit();
	TransformIMDCT_Uninit();

	MemoryPool_Release(&s_pool);
}

/* */

BOOL g_Enable_SSE2 = FALSE;

static void CPU_Check()
{
	INT32 info[4] = { 0 };

	__cpuid(info, 1);

	if ((info[3] & (1 << 26)) != 0) {
		g_Enable_SSE2 = TRUE;
	}
}

/* */

BOOL QV_Initialize(void)
{
	MemoryPool_Init(&s_pool);

	CPU_Check();

	TransformDecoder_Init(&s_pool);
	TransformIMDCT_Init(&s_pool);

	atexit(Unintialize);

	return TRUE;
}

BOOL QV_SetEnableSSE2(BOOL b)
{
	BOOL e = g_Enable_SSE2;

	g_Enable_SSE2 = b;

	return e;
}

/* */

/* QVorbisDecoderSetup */
struct QVorbisDecoderSetup {

	DecoderContext_t Context;

	MemoryPool_t Pool;

	DecoderSetup_t Setup;

}; /* QVorbisDecoderSetup */

/* */

QVorbisDecoderSetup_t* QV_CreateDecoderSetup(void)
{
	QVorbisDecoderSetup_t* setup;

	setup = (QVorbisDecoderSetup_t*)malloc(sizeof(QVorbisDecoderSetup_t));
	if (setup == NULL) {
		return NULL;
	}

	memset(setup, 0, sizeof(QVorbisDecoderSetup_t));

	MemoryPool_Init(&(setup->Pool));

	setup->Context.Pool  = &(setup->Pool);
	setup->Context.Error = NULL;

	return setup;
}

void QV_ReleaseDecoderSetup(QVorbisDecoderSetup_t* setup)
{
	if (setup != NULL) {
		MemoryPool_Release(&(setup->Pool));
		free(setup);
	}
}

BOOL QV_SetupDecoderSetup(
	QVorbisDecoderSetup_t* setup,
	const VOID*            id_packet,
	SIZE_T                 id_size,
	const VOID*            setup_packet,
	SIZE_T                 setup_size)
{
	BOOL result = FALSE;

	DecoderContext_t* ctx = &(setup->Context);

	SIZE_T cb = setup_size;
	VOID*  pv;

	VOID* top = MemoryPool_EnterBackyard(&(setup->Pool));

	BitDecoder_t* d = (BitDecoder_t*)MemoryPool_AllocateBackyard(&(setup->Pool), sizeof(BitDecoder_t));
	if (d == NULL) {
		THROW0(ctx, "QV_SetupDecoderSetup", "OutOfMemory")
	}

	if (cb < id_size) {
		cb = id_size;
	}

	cb = ((cb + 3) / 4) * 4;

	pv = MemoryPool_AllocateBackyard(&(setup->Pool), cb);
	if (pv == NULL) {
		THROW0(ctx, "QV_SetupDecoderSetup", "OutOfMemory")
	}

	memcpy(pv, id_packet, id_size);
	BitDecoder_Init(d, pv, id_size);

	if (!DecoderSetup_Id(
		ctx,
		d,
		&(setup->Setup))) {
		goto ErrorEnd;
	}

	memcpy(pv, setup_packet, setup_size);
	BitDecoder_Init(d, pv, setup_size);

	if (!DecoderSetup_Setup(
		ctx,
		d,
		&(setup->Setup))) {
		goto ErrorEnd;
	}

	result = TRUE;

ErrorEnd:

	MemoryPool_LeaveBackyard(&(setup->Pool), top);

	return result;
}

/* */

INT32 QV_GetDecoderSetupChannels(
	QVorbisDecoderSetup_t* setup)
{
	DecoderSetup_t* p = &(setup->Setup);
	return p->Channels;
}

INT32 QV_GetDecoderSetupSamplingRate(
	QVorbisDecoderSetup_t* setup)
{
	DecoderSetup_t* p = &(setup->Setup);
	return p->SampleRate;
}

/* */

/* QVorbisDecoder */
struct QVorbisDecoder {

	DecoderContext_t Context;

	MemoryPool_t Pool;

	FrameDecoder_t Frame;

	BitDecoder_t* Decoder;

	VOID*  Buffer;
	SIZE_T Size;

	SamplesChecker_t Checker;

	INT16* Sample;
	INT32  Samples;

	ConvertFrame_LE16_t Convert;

}; /* QVorbisDecoder */

QVorbisDecoder_t* QV_CreateDecoder(void)
{
	QVorbisDecoder_t* d;

	d = (QVorbisDecoder_t*)malloc(sizeof(QVorbisDecoder_t));
	if (d == NULL) {
		return NULL;
	}

	memset(d, 0, sizeof(QVorbisDecoder_t));

	MemoryPool_Init(&(d->Pool));

	d->Context.Pool  = &(d->Pool);
	d->Context.Error = NULL;

	d->Convert = NULL;

	return d;
}

void QV_ReleaseDecoder(QVorbisDecoder_t* d)
{
	if (d != NULL) {
		MemoryPool_Release(&(d->Pool));
		free(d);
	}
}

BOOL QV_SetupDecoder(
	QVorbisDecoder_t*      d,
	QVorbisDecoderSetup_t* setup,
	SIZE_T                 size)
{
	extern BOOL g_Enable_SSE2;

	BOOL result = FALSE;

	DecoderContext_t* ctx = &(d->Context);

	d->Decoder = (BitDecoder_t*)MemoryPool_Allocate(&(d->Pool), sizeof(BitDecoder_t));
	if (d->Decoder == NULL) {
		THROW0(ctx, "QV_SetupDecoder", "OutOfMemory")
	}

	d->Buffer = MemoryPool_Allocate(&(d->Pool), size);
	if (d->Buffer == NULL) {
		THROW0(ctx, "QV_SetupDecoder", "OutOfMemory")
	}

	d->Size = size;

	if (!FrameDecoder_Setup(
		ctx,
		&(setup->Setup),
		&(d->Frame))) {
		goto ErrorEnd;
	}

	if (!SamplesChecker_Setup(
		&(setup->Setup),
		&(d->Checker))) {
		goto ErrorEnd;
	}

	d->Samples = setup->Setup.Channels * setup->Setup.BlockSize1;
	d->Sample  = (INT16*)MemoryPool_Allocate(&(d->Pool), d->Samples * sizeof(INT16));
	if (d->Sample == NULL) {
		THROW0(ctx, "QV_SetupDecoder", "OutOfMemory")
	}

	if (g_Enable_SSE2) {
		d->Convert = ConvertFrame_LE16_SSE2;
	} else {
		d->Convert = ConvertFrame_LE16;
	}

	result = TRUE;

ErrorEnd:

	return result;
}

BOOL QV_DecodeFrame(
	QVorbisDecoder_t* d,
	const VOID*       packet,
	SIZE_T            size,
	QV_Output_t*      output)
{
	BOOL result = FALSE;

	DecoderContext_t* ctx = &(d->Context);

	BitDecoder_t*   decoder = d->Decoder;
	FrameDecoder_t* frame   = &(d->Frame);

	output->Channels  = 0;
	output->Length    = 0;
	output->Output[0] = NULL;
	output->Output[1] = NULL;

	if (size > d->Size) {
		THROW0(ctx, "QV_DecodeFrame", "NotEnoughBufferSize")
	}

	memcpy(d->Buffer, packet, size);
	BitDecoder_Init(decoder, d->Buffer, size);

	if (!FrameDecoder_Decode(
		ctx,
		decoder,
		frame)) {
		goto ErrorEnd;
	}

	output->Channels  = frame->Setup->Channels;
	output->Length    = frame->OutputLength;
	if (frame->OutputLength > 0) {
		INT32 i;
		for (i = 0; i < output->Channels; i++) {
			output->Output[i] = frame->Output[i];
		}
	}

	result = TRUE;

ErrorEnd:

	return result;
}

void QV_ResetDecoder(
	QVorbisDecoder_t* d)
{
	FrameDecoder_t* frame = &(d->Frame);
	FrameDecoder_Reset(frame);
}

/* */

BOOL QV_ConvertFrame(
	QVorbisDecoder_t*  d,
	const QV_Output_t* output,
	QV_Convert_t*      convert)
{
	FrameDecoder_t* frame = &(d->Frame);

	INT32 samples = d->Convert(
		frame->Setup->Channels,
		output,
		d->Sample);
	if (samples < 0) {
		convert->Sample  = NULL;
		convert->Samples = 0;
		return FALSE;
	}

	convert->Sample  = d->Sample;
	convert->Samples = samples;

	return TRUE;
}

/* */

void QV_ResetDecoderChecker(
	QVorbisDecoder_t* d)
{
	SamplesChecker_Reset(&(d->Checker));
}

BOOL QV_CheckDecoderChecker(
	QVorbisDecoder_t* d,
	const VOID*       packet,
	SIZE_T            size,
	INT32*            samples)
{
	return SamplesChecker_CheckFrame(
		&(d->Checker),
		packet,
		size,
		samples);
}

/* */

void QV_SetDecoderTestType(
	QVorbisDecoder_t* d,
	INT32             type)
{
#ifdef TEST_FRAME_DECODER
	FrameDecoder_t* frame = &(d->Frame);
	frame->TestType = type;
#endif
}

void QV_TestDecoder(
	QVorbisDecoder_t* d,
	QV_Output_t*      output)
{
#ifdef TEST_FRAME_DECODER
	FrameDecoder_t* frame = &(d->Frame);

	output->Length    = 0;
	output->Output[0] = NULL;
	output->Output[1] = NULL;

	output->Channels = frame->Setup->Channels;

	if (frame->TestType != QVH_PCM) {
		output->Length = frame->TestLength;
		if (frame->TestLength > 0) {
			INT32 i;
			for (i = 0; i < output->Channels; i++) {
				output->Output[i] = frame->Test[i];
			}
		}
	} else {
		output->Length = frame->OutputLength;
		if (frame->OutputLength > 0) {
			INT32 i;
			for (i = 0; i < output->Channels; i++) {
				output->Output[i] = frame->Output[i];
			}
		}
	}
#else
	output->Channels = 0;
	output->Length   = 0;
	output->Output[0] = NULL;
	output->Output[1] = NULL;
#endif
}

/* */

struct QVorbisSamplesChecker {

	SamplesChecker_t c;

};

QVorbisSamplesChecker_t* QV_CreateSamplesChecker(void)
{
	QVorbisSamplesChecker_t* t = (QVorbisSamplesChecker_t*)malloc(sizeof(QVorbisSamplesChecker_t));
	if (t == NULL) {
		return NULL;
	}

	memset(t, 0, sizeof(QVorbisSamplesChecker_t));

	return t;
}

void QV_ReleaseSamplesChecker(QVorbisSamplesChecker_t* t)
{
	if (t != NULL) {
		free(t);
	}
}

BOOL QV_SetupSamplesChecker(
	QVorbisSamplesChecker_t* t,
	QVorbisDecoderSetup_t*   setup)
{
	return SamplesChecker_Setup(
		&(setup->Setup),
		&(t->c));
}

void QV_ResetSamplesChecker(QVorbisSamplesChecker_t* t)
{
	SamplesChecker_Reset(&(t->c));
}

BOOL QV_CheckSamplesChecker(
	QVorbisSamplesChecker_t* t,
	const VOID*              packet,
	SIZE_T                   size,
	INT32*                   samples)
{
	return SamplesChecker_CheckFrame(
		&(t->c),
		packet,
		size,
		samples);
}

/* */

