// WaveWriter.cs
// 2008/10/23

using System;
using System.IO;
using System.Runtime.InteropServices;

namespace Audio {

// WaveHeader
[StructLayout(LayoutKind.Sequential)]
public struct WaveHeader {

	public UInt32 RIFF;
	public UInt32 RIFFSize;

	public UInt32 WAVE;

	public UInt32 fmt_;
	public UInt32 fmt_Size;

	public UInt16 FormatTag;
	public UInt16 Channels;
	public UInt32 SamplesPerSec;
	public UInt32 AvgBytesPerSec;
	public UInt16 BlockAlign;
	public UInt16 BitsPerSample;

	public UInt32 data;
	public UInt32 dataSize;

} // WaveHeader

// WaveWriter
public class WaveWriter : IDisposable {

	private Int32 _channels;
	private Int32 _samplesPerSec;

	private Int32 _length;

	private FileStream _s;

	public WaveWriter(Int32 c, Int32 s)
	{
		_channels      = c;
		_samplesPerSec = s;

		_length = 0;
	}

	public void Dispose()
	{
		if (_s != null) {
			_s.Close();
			_s = null;
		}
	}

	public void Create(String path)
	{
		_s = File.Create(path);

		Int32 len = Marshal.SizeOf(typeof(WaveHeader));
		Byte[] buf = new Byte[len];
		_s.Write(buf, 0, buf.Length);
	}

	public void Write(Single[][] data, Int32 length)
	{
		Byte[] buffer = new Byte[_channels * length * 2];

		Int32 p = 0;
		for (Int32 i = 0; i < length; i++) {
			for (Int32 ch = 0; ch < _channels; ch++) {
				Single s = data[ch][i];

				Int32 s0 = (Int32)(s * 32768.0);
				if (s0 < -32768) {
					s0 = -32768;
				} else if (s0 > 32767) {
					s0 = 32767;
				}

				buffer[p+0] = (Byte)( s0       & 0xff);
				buffer[p+1] = (Byte)((s0 >> 8) & 0xff);

				p += 2;
			}
		}

		_s.Write(buffer, 0, buffer.Length);

		_length += buffer.Length;
	}

	public void Finish()
	{
		WaveHeader header;

		header.RIFF     = Chunk("RIFF");
		header.RIFFSize = (UInt32)(4 + 8 + 0x10 + 8 + _length);

		header.WAVE = Chunk("WAVE");

		header.fmt_     = Chunk("fmt ");
		header.fmt_Size = 0x10;

		header.FormatTag      = 1; /* PCM */
		header.Channels       = (UInt16)_channels;
		header.SamplesPerSec  = (UInt32)_samplesPerSec;
		header.AvgBytesPerSec = (UInt32)(_samplesPerSec * _channels * 2);
		header.BlockAlign     = (UInt16)(_channels * 2);
		header.BitsPerSample  = 16;

		header.data     = Chunk("data");
		header.dataSize = (UInt32)_length;

		Int32 len = Marshal.SizeOf(typeof(WaveHeader));
		Byte[] by = new Byte[len];

		var gh = GCHandle.Alloc(by, GCHandleType.Pinned);
		try {
			IntPtr p = gh.AddrOfPinnedObject();
			Marshal.StructureToPtr(header, p, false);
		} finally {
			gh.Free();
		}

		_s.Seek(0, SeekOrigin.Begin);
		_s.Write(by, 0, by.Length);
		_s.Close();

		_s = null;
	}

	private UInt32 Chunk(String id)
	{
		UInt32 v = 0;
		v |=  (UInt32)id[0];
		v |= ((UInt32)id[1] <<  8);
		v |= ((UInt32)id[2] << 16);
		v |= ((UInt32)id[3] << 24);
		return v;
	}

} // WaveWriter

} // namespace Audio

