// Main.cs
// 2008/11/18

using System;
using System.IO;

// MainApp
static class MainApp {

	static void Main()
	{
		try {
			QVorbis.Module.Initialize();

			Run();

			Environment.ExitCode = 0;

		} catch (Exception e) {
			Console.Error.WriteLine("Error: " + e);

			Environment.ExitCode = -1;
		}

#if DEBUG
		Console.Error.WriteLine("DONE.");
		Console.ReadLine();
#endif
	}

	static void Run()
	{
		using (var tool = new Tool()) {
			tool.Run();
		}
	}

} // MainApp

/* */

// Tool
public class Tool : IDisposable {

	private String _output;
	private String _property;
	private String _source;

	private String _ovkPath;

	private Boolean _qax;

	public Tool()
	{
		_output   = String.Empty;
		_property = String.Empty;
		_source   = String.Empty;

		_ovkPath = String.Empty;

		_qax = true;
	}

	public void Dispose()
	{
	}

	public void Run()
	{
		if (!ParseArgs()) {
			Console.WriteLine("QAXMaker [Output File]");
			Console.WriteLine(" -p [Property File]");
			Console.WriteLine(" -s [Source Folder]");
			Console.WriteLine("");
			Console.WriteLine("QAXMaker -qas [Source Folder]");
			return;
		}

		DoProcess();
	}

	private bool ParseArgs()
	{
		String[] args = Environment.GetCommandLineArgs();

		bool p = false;
		bool s = false;

		bool ovk = false;
		bool qas = false;

		for (Int32 i = 1; i < args.Length; i++) {
			String arg = args[i];

			if (p) {
				_property = arg;
				p = false;
				continue;
			}

			if (s) {
				_source = arg;
				s = false;
				continue;
			}

			if (ovk) {
				_ovkPath = arg;
				ovk = false;
				continue;
			}

			if (qas) {
				_source = arg;
				qas = false;
				continue;
			}

			if (arg == "-p" || arg == "/p") {
				p = true;
				continue;
			}

			if (arg == "-s" || arg == "/s") {
				s = true;
				continue;
			}

			if (arg == "-ovk" || arg == "/ovk") {
				ovk = true;
				continue;
			}

			if (arg == "-qas" || arg == "/qas") {
				qas  = true;
				_qax = false;
				continue;
			}

			_output = arg;
		}

		if (_qax && _output == String.Empty) {
			return false;
		}

		return true;
	}

	private void DoProcess()
	{
		if (_source == String.Empty) {
			_source = Directory.GetCurrentDirectory();
		}

		if (_qax) {
			DoProcessQAX();
		} else {
			DoProcessQAS();
		}
	}

	/* */

	private void DoProcessQAX()
	{
		var bag = new QAX.PropertyBag();
		if (_property != String.Empty) {
			using (var r = File.OpenText(_property)) {
				bag.Read(r, _property);
			}
		}

		using (var w = File.Create(_output)) {
			var qw = new QAX.QAXWriter(w);

			qw.SetupPropertyBag(bag);

			qw.Prepare();

			AddVorbis(qw);

			if (_ovkPath != String.Empty) {
				AddOvkFile(qw);
			}

			qw.Finish();
		}
	}

	private void AddVorbis(QAX.QAXWriter qw)
	{
		String[] files = Directory.GetFiles(_source, "*.ogg");
		foreach (String file in files) {
			String fname = Path.GetFileName(file);
			String iname = Path.GetFileNameWithoutExtension(file);

			Console.WriteLine(fname);

			using (var r = File.OpenRead(file)) {
				qw.AddVorbisContent(
					iname,
					r);
			}
		}
	}

	private void AddOvkFile(QAX.QAXWriter qw)
	{
		using (var r = File.OpenRead(_ovkPath)) {
			var ovk = new VA.OVKReader(r);

			ovk.ReadHeader();

			Int32 count = ovk.Count;
			for (Int32 i = 0; i < count; i++) {
				String name = ovk.GetName(i);

				Console.WriteLine(name);

				using (var s = ovk.CreateStream(i)) {
					qw.AddVorbisContent(
						name,
						s);
				}
			}
		}
	}

	/* */

	private void DoProcessQAS()
	{
		String[] files = Directory.GetFiles(_source, "*.ogg");
		foreach (String file in files) {
			String fname = Path.GetFileName(file);
			String iname = Path.GetFileNameWithoutExtension(file);

			Console.WriteLine(fname);

			String path = iname + ".qas";

			using (var w = File.Create(path)) {
				var qw = new QAX.QASWriter(w);

				qw.Prepare();

				using (var r = File.OpenRead(file)) {
					qw.AddVorbisContent(r);
				}

				qw.Finish();
			}
		}
	}

	/* */

} // Tool

/* */

