// QASWriter.cs
// 2008/12/11

using System;
using System.IO;
using System.Text;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace QAX {

// QASHeader
[StructLayout(LayoutKind.Sequential)]
public struct QASHeader {

	[MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
	public Byte[] Signature;

	[MarshalAs(UnmanagedType.ByValArray, SizeConst = 12)]
	public Byte[] Format;

	public UInt64 Samples;
	public UInt64 Reserved;

	public UInt64 ContentPos;
	public UInt64 ContentSize;

	public UInt64 MetaDataPos;
	public UInt32 MetaDataSize;
	public UInt32 MetaDataIndex;

} // QASHeader

// QASWriter
public class QASWriter {

	private Stream _s;

	private QASHeader _header;

	private Int64 _position;

	private const Int32 ZERO_PAGE =  0x100;

	private const Int32 PAGE_UNIT = 0x1000;

	private const Byte F_START = 0x01;
	private const Byte F_END   = 0x02;

	public QASWriter(Stream s)
	{
		_s = s;

		_header = new QASHeader();

		_header.Signature = new Byte[ 4];
		_header.Format    = new Byte[12];

		_position = 0;
	}

	public void Prepare()
	{
		_position = _s.Position;

		Byte[] by = new Byte[ZERO_PAGE];
		_s.Write(by, 0, by.Length);
	}

	public void AddVorbisContent(
		Stream s)
	{
		var r = new QOgg.Checker(s);

		r.ReadSetup();

		Byte[] id    = r.Id;
		Byte[] setup = r.Setup;

		using (var vsetup = new QVorbis.Setup()) {
			vsetup.Create(id, setup);

			using (var vchecker = new QVorbis.Checker()) {
				vchecker.Setup(vsetup);

				var w = new QAXContentWriter(_s);

				UInt64 last_gpos = 0;

				UInt64 pos = 0;

				while (r.ReadPage()) {
					Byte[][] packets = r.Packets;

					UInt64 gpos  = r.GranulePosition;
					UInt32 flags = r.Flags;

					Int32 duration = 0;
					Int32 offset   = 0;

					for (Int32 i = 0; i < packets.Length; i++) {
						Byte[] packet = packets[i];

						Int32 samples = vchecker.Check(packet);
						if (i == 0) {
							offset = samples;
						}

						duration += samples;
					}

					Byte cflags = 0;
					if (pos == 0) {
						cflags |= F_START;
					}

					if ((flags & QOgg.Header.HT_EOS) != 0) {
						cflags |= F_END;
					}

					w.Write(
						packets,
						cflags,
						duration,
						offset);

					if ((flags & QOgg.Header.HT_EOS) != 0) {
						last_gpos = gpos;
					}

					pos += (UInt64)duration;
				}

				if (last_gpos == 0) {
					last_gpos = pos;
				}

				UInt64 csize = w.Finish(ZERO_PAGE);

				Byte[] msetup = SetupPool.MakeSetup(id, setup);
				Byte[] mindex = w.GetIndex();

				var mchunk = MakeMetaDataChunk(msetup, mindex);

				_s.Write(mchunk.Payload, 0, mchunk.Payload.Length);

				ToBytes(_header.Format, "vorbis");

				_header.Samples      = last_gpos;

				_header.ContentPos   = ZERO_PAGE;
				_header.ContentSize  = w.Size;

				_header.MetaDataPos   = ZERO_PAGE + csize;
				_header.MetaDataSize  = (UInt32)mchunk.Payload.Length;
				_header.MetaDataIndex = (UInt32)mchunk.Index;
			}
		}
	}

	public void Finish()
	{
		ToBytes(_header.Signature, "QAS1");

		Int32 len = Marshal.SizeOf(typeof(QASHeader));
		Byte[] by = new Byte[len];

		var gh = GCHandle.Alloc(by, GCHandleType.Pinned);
		try {
			IntPtr p = gh.AddrOfPinnedObject();
			Marshal.StructureToPtr(_header, p, false);
		} finally {
			gh.Free();
		}

		_s.Seek(_position, SeekOrigin.Begin);
		_s.Write(by, 0, by.Length);
	}

	private static void ToBytes(Byte[] buf, String text)
	{
		for (Int32 i = 0; i < text.Length; i++) {
			Char ch = text[i];
			buf[i] = (Byte)ch;
		}
	}

	private static Chunk MakeMetaDataChunk(Byte[] setup, Byte[] index)
	{
		var chunk = new Chunk();

		var ms = new MemoryStream();

		Int32 idx = 0;

		using (var w = new BinaryWriter(ms)) {
			Int32[] lens = new Int32[2];

			lens[0] = setup.Length;
			w.Write(setup);
			idx += lens[0];

			lens[1] = index.Length;
			w.Write(index);
			idx += lens[1];

			Utils.WriteLengths(w, lens);
		}

		chunk.Payload = ms.ToArray();
		chunk.Index   = idx;

		return chunk;
	}

} // QASWriter

} // namespace QAX

