// Checker.h
// 2008/11/18

#pragma once

using namespace System;

namespace QVorbis {

// Checker
public ref class Checker : public IDisposable {

	QVorbisSamplesChecker_t* m_checker;

public:

	Checker() : m_checker(0)
	{
		m_checker = QV_CreateSamplesChecker();
		if (m_checker == 0) {
			throw gcnew OutOfMemoryException();
		}
	}

	~Checker()
	{
		if (m_checker != 0) {
			QV_ReleaseSamplesChecker(m_checker);
			m_checker = 0;
		}
	}

	void Setup(
		Setup^ setup)
	{
		if (!QV_SetupSamplesChecker(
			m_checker,
			setup->DecoderSetup)) {
			throw gcnew Error("QV_SetupSamplesChecker");
		}
	}

	Int32 Check(
		array<Byte>^ packet)
	{
		Int32 samples = 0;

		pin_ptr<Byte> p = &(packet[0]);
		if (!QV_CheckSamplesChecker(
			m_checker,
			p,
			packet->Length,
			&samples)) {
			throw gcnew Error("QV_CheckSamplesChecker");
		}

		return samples;
	}

}; // Checker

// CRC32
public ref class CRC32 {

	static array<UInt32>^ CRC_TABLE;

	UInt32 m_crc;

private:

	static void MakeTable()
	{
		if (CRC_TABLE != nullptr) {
			return;
		}

		CRC_TABLE = gcnew array<UInt32>(0x100);

		for (UInt32 i = 0; i < 0x100; i++) {
			UInt32 r = i;
			for (UInt32 j = 0; j < 8; j++) {
				if ((r & 1) != 0) {
					r = (r >> 1) ^ 0xedb88320;
				} else {
					r >>= 1;
				}
			}
			CRC_TABLE[i] = r;
		}
	}

public:

	CRC32() : m_crc(0xffffffff)
	{
		MakeTable();
	}

	property UInt32 CRC {
		UInt32 get()
		{
			return m_crc ^ 0xffffffff;
		}
	}

	UInt32 Generate(
		array<Byte>^ buffer)
	{
		UInt32 crc = m_crc;

		for (Int32 i = 0; i < buffer->Length; i++) {
			crc = CRC_TABLE[(crc & 0xff) ^ buffer[i]] ^ (crc >> 8);
		}

		m_crc = crc;

		return m_crc ^ 0xffffffff;
	}

}; // CRC32

} // namespace QVorbis

