#include <cstdio>
#include <cerrno>

extern "C" {
#include <netdb.h>
#include <unistd.h>
}

#include "udp_socket.h"

udp_socket::udp_socket(const std::string& host, const int port)
  :socket_fd(udp_socket::BAD_FD)
{
  try {
    string func;

    memset(&(this -> addr), 0, sizeof(this -> addr));
    if ((this -> socket_fd = ::socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
      throw func = "socket";
    }

    sockaddr_in my_addr;
    memset(&my_addr, 0, sizeof(my_addr));
    my_addr.sin_family      = AF_INET;
    my_addr.sin_addr.s_addr = htonl(INADDR_ANY);
    my_addr.sin_port        = htons(0);

    if (::bind(this->socket_fd, 
	       (const sockaddr *)(&my_addr),
	       sizeof(my_addr))  ==  -1) {
      throw func = "bind";
    }

    hostent* hp;
    if (!(hp = ::gethostbyname(host.c_str()))) {
      throw func = "gethostbyname";
    }
    this -> addr.sin_family = AF_INET;
    memcpy(&(this -> addr.sin_addr.s_addr), hp -> h_addr , hp -> h_length);
    this -> addr.sin_port = htons(port);

  } catch (string& e) {
    perror(e.c_str());
    this -> socket_fd = udp_socket::BAD_FD;
    this -> close();
    return;
  }
}

udp_socket::~udp_socket()
{
  this -> close();
}

int udp_socket::fd()
{
  return this->socket_fd;
}

int udp_socket::send(const std::string& str)
{
  if (this -> socket_fd == udp_socket::BAD_FD) {
    return -1;
  }

  int n = ::sendto(this -> socket_fd , str.c_str() , str.length() , 0 ,
		   (struct sockaddr *)(&(this -> addr)) ,
		   sizeof(this -> addr));

  if (static_cast<size_t>(n) == str.length()) {
    return n;
  } else {
    if (n == -1) {
      perror("sendto");
    }
    return -1;
  }
}

int udp_socket::recv(std::string& str)
{
  if (this -> socket_fd == udp_socket::BAD_FD) {
    return -1;
  }

  static char buf[udp_socket::BUF_SIZE];
  sockaddr_in from;
  socklen_t from_size = sizeof from;

  int n = ::recvfrom(this -> socket_fd, buf, udp_socket::BUF_SIZE,
		     0 , (sockaddr *) &from, &from_size);

  if (n >= 0) {
    this->addr.sin_port = from.sin_port;
    buf[n] = '\0';
    str = buf;
  } else {
    if(n == -1 && errno == EWOULDBLOCK){
      buf[0] = '\0' ;
      n = 0;
    }
  }

  return n;
}

int udp_socket::recv(char buf[], int size)
{
  if (this -> socket_fd == udp_socket::BAD_FD) {
    return -1;
  }

  sockaddr_in from;
  socklen_t from_size = sizeof from;

  int n = ::recvfrom(this -> socket_fd, buf, size,
		     0 , (sockaddr *) &from, &from_size);

  if (n >= 0) {
    this->addr.sin_port = from.sin_port;
    buf[n] = '\0';
  } else {
    if(n == -1 && errno == EWOULDBLOCK){
      buf[0] = '\0' ;
      n = 0;
    }
  }

  return n;
}

int udp_socket::close()
{
  if (!(this -> socket_fd == udp_socket::BAD_FD)) {
    int ret = ::close(this -> socket_fd); 
    if (ret == -1) {
      perror("close");
    }

    this -> socket_fd = udp_socket::BAD_FD;

    return ret;
  }
  return 0;
}



