# = Log4r Logger
# 
#--
# Ruby version 1.8
#
# == Authors
# * Yomei Komiya
# 
# == Copyright
# 2008 the original author or authors.
#
# == License
# Apache License 2.0
# 
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
# 
#  http://www.apache.org/licenses/LICENSE-2.0
# 
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#++
# == Version
# SVN: $Id: log4r_logger.rb 77 2008-10-12 12:07:49Z whitestar $
#
# == Since
# File available since Release 0.8.0

require 'log4r/logevent'

require 'commons/ruby/exception'
require 'commons/ruby/log4r'
require 'commons/logging/log'

module Commons
  module Logging
    module Impl

      class Log4rLogger
        include Commons::Logging::Log
        
        FQCN = self.name
        
        # The level of the caller that raises a log event.
        # ... (2)target_log_event -> (1)Log#log_method -> (0)Logger#log
        CALLER_LEVEL = 2
        
        LOG_PROPAGATED = true
        
        
        def initialize(param)
          super()
          
          case param
            when String  # logger name
              @name = param
              @logger = get_logger()
            when Log4r::Logger  # logger object, this case for use with a log4r factory.
              if param == nil
                raise ArgumentError,
                  'Warning - nil logger in constructor; possible log4r misconfiguration.'
              end
              @name = param.name
              @logger = param
            else
              raise ArgumentError,
                'Warning - invalid parameter; logger name or logger object is expected.'
          end
        end
        
        
        def get_logger
          if @logger == nil
            @logger = Log4r::Logger.get_logger(@name)
          end
          
          return @logger
        end
        
        
        def create_log_event(level, tracer, message)
          logger = get_logger
          tracercall = (logger.trace ? tracer : nil)
          return Log4r::LogEvent.new(level, logger, tracercall, message)
        end
        protected :create_log_event
        
        
        def log(level, message, exception = nil)
          message = build_message(message, exception)
          
          case level
            when TRACE, DEBUG
              get_logger.debug(
                create_log_event(Log4r::DEBUG, caller(CALLER_LEVEL), message),
                LOG_PROPAGATED)
            when INFO 
              get_logger.info(
                create_log_event(Log4r::INFO, caller(CALLER_LEVEL), message),
                LOG_PROPAGATED)
            when WARN
              get_logger.warn(
                create_log_event(Log4r::WARN, caller(CALLER_LEVEL), message),
                LOG_PROPAGATED)
            when ERROR
              get_logger.error(
                create_log_event(Log4r::ERROR, caller(CALLER_LEVEL), message),
                LOG_PROPAGATED)
            when FATAL
              get_logger.fatal(
                create_log_event(Log4r::FATAL, caller(CALLER_LEVEL), message),
                LOG_PROPAGATED)
            else
              # do nothing.
          end
        end
        
        
        def build_message(message, exception)
          if message == nil
            message = '<nil>'
          end
          
          if exception != nil
            return message.to_s \
            + ' <' + exception.class.name + ': ' + exception.message.to_s + '; ' \
            + exception.stack_trace_string + '>';
          else
            return message.to_s
          end
        end
        protected :build_message
        
        
        def enabled_for?(level)
          case level
            when TRACE, DEBUG
              return get_logger.debug?
            when INFO
              return get_logger.info?
            when WARN
              return get_logger.warn?
            when ERROR
              return get_logger.error?
            when FATAL
              return get_logger.fatal?
            else
              return false
          end
        end
      end

    end
  end
end
