# = Log Factory Default Implementation
# 
#--
# Ruby version 1.8
#
# == Authors
# * Yomei Komiya
# 
# == Copyright
# 2008 the original author or authors.
#
# == License
# Apache License 2.0
# 
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
# 
#  http://www.apache.org/licenses/LICENSE-2.0
# 
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#++
# == Version
# SVN: $Id: log_factory_impl.rb 77 2008-10-12 12:07:49Z whitestar $
#
# == Since
# File available since Release 0.8.0

require 'commons'
require 'commons/io/filename_utils'
require 'commons/lang/class_loader'
require 'commons/logging/log'
require 'commons/logging/log_configuration_error'
require 'commons/logging/log_factory'

module Commons
  module Logging
    module Impl

      class LogFactoryImpl < Commons::Logging::LogFactory
        # The property name used to identify the LOG_MODULE inclusion class name.
        LOG_PROPERTY = 'commons.logging.Log'
        
        # The fully qualified class name of the log module.
        LOG_MODULE = 'Commons::Logging::Log'
        
        # The log factory setter name.
        LOG_METHOD_NAME = 'set_log_factory'
        
        LOG4R_LOGGER_CLASS = 'Commons::Logging::Impl::Log4rLogger'
        
        SIMPLELOG_LOGGER_CLASS = 'Commons::Logging::Impl::SimpleLog'
        
        
        attr_accessor :attributes, :instances, :log_class, :log_method
        
        protected :attributes, :instances, :log_class, :log_method
        
        
        def initialize
          super()
          
          @attributes = Hash.new
          
          # Already instantiated logger objects. keyed by name.
          @instances = Hash.new
          
          # The fully qualified class name of LOG_MODULE inclusion.
          @log_class_name = nil
          
          # The inclusion class object of the LOG_MODULE
          @log_class = nil
          
          # The one-argument LOG_METHOD_NAME method of the selected
          # LOG_MODULE inclusion class, if it exists.
          @log_method = nil          
        end
        
        
        def get_attribute(name)
          if @attributes.has_key?(name)
            return @attributes[name]
          else
            return nil
          end
        end
        
        
        def get_attribute_names
          return @attributes.keys
        end
        
        
        def get_instance(name)
          unless @instances.has_key?(name)
            @instances[name] = new_instance(name)
          end
          
          return @instances[name]
        end
        
        
        def release
          @instances.clear
        end
        
        
        def remove_attribute(name)
          @attributes.delete(name)
        end
        
        
        def set_attribute(name, value)
          if value == nil
            @attributes.delete(name)
          else
            @attributes[name] = value
          end
        end
        
        
        # Gets Log module inclusion class name.
        # throws Commons::Logging::LogConfigurationError
        def get_log_class_name
          unless @log_class_name == nil
            return @log_class_name
          end
          
          # 1. Explicitly defined
          @log_class_name = get_attribute(LOG_PROPERTY)
          
          # 2. Environmental variable
          if @log_class_name == nil
            env = ENV[LOG_PROPERTY]
            unless env == nil
              @log_class_name = env
            end
          end
          # 3. Log4r
          if @log_class_name == nil && log4r_available?
            @log_class_name = LOG4R_LOGGER_CLASS
          end
          # 4. SimpleLog 
          if @log_class_name == nil
            @log_class_name = SIMPLELOG_LOGGER_CLASS
          end
          
          init_log_configuration(@log_class_name)
          
          return @log_class_name
        end
        protected :get_log_class_name
        
        
        # Initializes logging framework configuration.
        def init_log_configuration(log_class_name)
          if log_class_name == LOG4R_LOGGER_CLASS
            log4r_conf = get_attribute(Commons::LOG4R_CONF_PROPERTY)
            unless log4r_conf == nil
              unless Commons::IO::FilenameUtils.absolute?(log4r_conf)
                log4r_conf = Commons::Lang::ClassLoader.get_resource(log4r_conf)
              end
              unless log4r_conf == nil
                Commons::ENV[Commons::LOG4R_CONF_PROPERTY] = log4r_conf
              else
                raise Commons::Logging::LogConfigurationError, 
                  'Log4r configuration file not found, property key: ' \
                  + Commons::LOG4R_CONF_PROPERTY
              end
            end
          end
        end
        protected :init_log_configuration
        
        
        # Obtains LOG_MODULE inclusion's class.
        # throws Commons::Logging::LogConfigurationError
        def get_log_class
          unless @log_class == nil
            return @log_class
          end
          
          begin
            log_class_name = get_log_class_name
            
            Commons::Lang::ClassLoader.load(log_class_name)
            @log_class = eval(log_class_name)
            if @log_class == nil
              raise Commons::Logging::LogConfigurationError, 
                'No suitable Log implementation for ' + log_class_name
            end
            unless @log_class.include?(eval(LOG_MODULE))
              raise Commons::Logging::LogConfigurationError(
                'Class ' . log_class_name + ' does not include \'' \
                + LOG_MODULE + '\'.')
            end
          rescue Commons::Logging::LogConfigurationError
            raise
          rescue Exception => e
            raise Commons::Logging::LogConfigurationError, e.message
          end
          
          begin
            @log_method = @log_class.instance_method(LOG_METHOD_NAME)
          rescue Exception => e
            @log_method = nil
          end
      
          return @log_class;
        end
        protected :get_log_class
        
        
        # Tests whether Log4R is available or not.
        def log4r_available?
          # check the resource existence.
          begin
            require 'log4r'
            return true
          rescue LoadError
            return false
          end
        end
        protected :log4r_available?
        
        
        # Creates and returns LOG_MODULE instance
        # throws Commons::Logging::LogConfigurationError
        def new_instance(name)
          instance = nil
          begin
            # Commons::Logging::Log implementation must have one-argument constructor.
            instance = get_log_class.new(name)
            unless @log_method == nil
              @log_method.bind(instance).call(self)
            end
            return instance
          rescue Commons::Logging::LogConfigurationError
            raise
          rescue Exception => e
            raise Commons::Logging::LogConfigurationError, e.message
          end
        end
        protected :new_instance
      end

    end
  end
end
