# = Simple Log
# 
#--
# Ruby version 1.8
#
# == Authors
# * Yomei Komiya
# 
# == Copyright
# 2008 the original author or authors.
#
# == License
# Apache License 2.0
# 
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
# 
#  http://www.apache.org/licenses/LICENSE-2.0
# 
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#++
# == Version
# SVN: $Id: simple_log.rb 77 2008-10-12 12:07:49Z whitestar $
#
# == Since
# File available since Release 0.8.0

require 'commons/ruby/exception'

require 'commons/lang/system_utils'
require 'commons/logging/log'
require 'commons/logging/impl/no_op_log'
require 'commons/util/properties'

module Commons
  module Logging
    module Impl
      
      class SimpleLog < NoOpLog
        include Commons::Logging::Log
        
        SYSTEM_PREFIX = 'commons.logging.simplelog.'
        
        DEFAULT_DATE_TIME_FORMAT = '%Y/%m/%d %H:%M:%S %Z'
        
        LOG_LEVEL_ALL    = 0
        
        LOG_LEVEL_TRACE  = 1
        
        LOG_LEVEL_DEBUG  = 2
        
        LOG_LEVEL_INFO   = 3
        
        LOG_LEVEL_WARN   = 4
        
        LOG_LEVEL_ERROR  = 5
        
        LOG_LEVEL_FATAL  = 6
        
        LOG_LEVEL_OFF    = 7
        
        @@simple_log_props = nil
        
        @@show_log_name = false
        
        @@show_short_name = true
        
        @@show_date_time = false
        
        @@date_time_format = DEFAULT_DATE_TIME_FORMAT
        
        
        def self.get_string_property(name, default = nil)
          prop = nil;
          prop = ENV[name]
          prop = prop == nil ? @@simple_log_props.get_property(name) : prop
          return prop == nil ? default : prop
        end
        
        
        def self.get_boolean_property(name, default)
          prop = get_string_property(name)
          prop.upcase! unless prop == nil
          return prop == nil ? default : prop == 'TRUE'
        end
        class << self
          private :get_boolean_property
        end
      
  
        def self.static_initialize
          @@simple_log_props = Commons::Util::Properties.new
          
          # Add props from the resource simplelog.properties
          props_path = Commons::Lang::ClassLoader.get_resource('simplelog.properties')
          File.open(props_path, 'r') {|file|
            @@simple_log_props.load(file)
          }
          
          @@show_log_name = get_boolean_property(
            SYSTEM_PREFIX + 'show_log_name', @@show_log_name)
          @@show_short_name = get_boolean_property(
            SYSTEM_PREFIX + 'show_short_name', @@show_short_name)
          @@show_date_time = get_boolean_property(
            SYSTEM_PREFIX + 'show_date_time', @@show_date_time)
        end
        class << self
          protected :static_initialize
        end
        self.static_initialize
        
        
        attr_accessor :stderr, :log_name, :current_log_level
        
        protected :stderr, :log_name, :current_log_level
        
        
        def initialize(name = nil)
          super(name)
          
          @stderr = false
            
          @log_name = name
          
          @current_log_level
          
          @short_log_name = nil
          
          # Set log level from properties
          lvl = self.class.get_string_property(SYSTEM_PREFIX + 'log.' + @log_name)
          i = name == nil ? -1 : name.rindex('::') 
          while lvl == nil && i > -1
            name = name[0 ... i]
            lvl = self.class.get_string_property(SYSTEM_PREFIX + 'log.' + name)
            i = name.rindex('::')
          end
          
          if lvl == nil
            lvl = self.class.get_string_property(SYSTEM_PREFIX + 'default_log')
          end
          
          lvl = lvl.upcase
          case lvl
            when 'ALL'   then set_level(LOG_LEVEL_ALL)
            when 'TRACE' then set_level(LOG_LEVEL_TRACE)
            when 'DEBUG' then set_level(LOG_LEVEL_DEBUG)
            when 'INFO'  then set_level(LOG_LEVEL_INFO)
            when 'WARN'  then set_level(LOG_LEVEL_WARN)
            when 'ERROR' then set_level(LOG_LEVEL_ERROR)
            when 'FATAL' then set_level(LOG_LEVEL_FATAL)
            when 'OFF'   then set_level(LOG_LEVEL_OFF)
            else              set_level(LOG_LEVEL_INFO)
          end
        end
        
        
        def set_level(current_log_level)
          @current_log_level = current_log_level
        end
        
        
        def get_level
          return @current_log_level
        end
        
        
        def log(level, message, exception = nil)
          if message == nil
            message = '<null>'
          end
          
          buf = ''
          
          if @@show_date_time
            now = Time.now
            buf.concat(
              now.strftime(@@date_time_format) + ' +' + now.usec.to_s)
            buf.concat(' ')
          end
          
          case level
            when TRACE then buf.concat('[TRACE] ')
            when DEBUG then buf.concat('[DEBUG] ')
            when INFO  then buf.concat('[INFO]  ')
            when WARN  then buf.concat('[WARN]  ')
            when ERROR then buf.concat('[ERROR] ')
            when FATAL then buf.concat('[FATAL] ')
          end
          
          if @@show_short_name
            if @short_log_name == nil
              # Cut all but the last component of the name
              delim_rindex = @log_name.rindex('::')
              if delim_rindex != nil
                @short_log_name = @log_name[delim_rindex + 2 .. -1]
              end
            end
            buf.concat(@short_log_name).concat(' - ')
          elsif @@show_log_name
            buf.concat(@log_name).concat(' - ')
          end
          
          buf.concat(message.to_s)
          
          # Append stack trace if not null
          if exception != nil
            buf.concat(
              ' <' + exception.class.name + ': ' + exception.message.to_s + '; ' \
              + exception.stack_trace_string + '> ')
          end
          
          write(buf);   
        end
        
        
        def write(buffer)
          $stderr.write(buffer.to_s + Commons::Lang::SystemUtils.line_separator)
        end
        protected :write
        
        
        def enabled_for?(level)
          case level
            when TRACE
              level = LOG_LEVEL_TRACE
            when DEBUG
              level = LOG_LEVEL_DEBUG
            when INFO
              level = LOG_LEVEL_INFO
            when WARN
              level = LOG_LEVEL_WARN
            when ERROR
              level = LOG_LEVEL_ERROR
            when FATAL
              level = LOG_LEVEL_FATAL
            else
              level = LOG_LEVEL_OFF
          end
          
          return (level >= @current_log_level)
        end
      end
      
    end
  end
end
