# = Duration Format Utilities Test Case
#
# Ruby version 1.8
#
# == Authors
# * Yomei Komiya
# 
# == Copyright
# 2008 the original author or authors.
#
# == License
# Apache License 2.0
# 
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
# 
#  http://www.apache.org/licenses/LICENSE-2.0
# 
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# == Version
# SVN: $Id: duration_format_utils_test.rb 68 2008-10-05 05:08:21Z whitestar $
#
# == Since
# File available since Release 0.8.0

require 'commons/ruby/test/unit'

require 'commons/lang/time/duration_format_utils'

module Commons
  module Lang
    module Time
      
      class DurationFormatUtilsTest < Test::Unit::TestCase
        Token = DurationFormatUtils::Token

        
        def setup
          $KCODE = 'UTF8'
        end
        
        
        def teardown
          $KCODE = 'NONE'
        end

        
        def test_initialize
          # no test.
        end
        
        
        def test_format_duration_hms()
          expected = '9:36:39.504'
          actual = DurationFormatUtils.format_duration_hms(34599504)
          assert_equal(expected, actual)
        end
        
        
        def test_format_duration()
          format = 'd HH:mm:ss.SSS'
          expected = '13870 09:36:39.504'
          actual = DurationFormatUtils.format_duration(
            1198402599504, format, true)
          assert_equal(expected, actual)
          
          format = 'd日 HH時間mm分ss.SSS秒'
          expected = '13870日 09時間36分39.504秒'
          actual = DurationFormatUtils.format_duration(
            1198402599504, format, true)
          assert_equal(expected, actual)
        end
        
        
        def test_format
          tokens = [
            Token.new(DurationFormatUtils::TOKEN_y, 4),
            Token.new('/', 1),
            Token.new(DurationFormatUtils::TOKEN_M, 2),
            Token.new('/', 1),
            Token.new(DurationFormatUtils::TOKEN_d, 2),
            Token.new(' ', 1),
            Token.new(DurationFormatUtils::TOKEN_H, 2),
            Token.new(':', 1),
            Token.new(DurationFormatUtils::TOKEN_m, 2),
            Token.new(':', 1),
            Token.new(DurationFormatUtils::TOKEN_s, 2),
            Token.new('.', 1),
            Token.new(DurationFormatUtils::TOKEN_S, 3)
          ]
          
          years = 2005
          months = 3
          days = 4
          hours = 5
          minutes = 5
          seconds = 9
          milliseconds = 25
          pad_with_zeros = true
          
          actual = DurationFormatUtils.format(
            tokens,
            years, months, days, hours, minutes, seconds, milliseconds,
            pad_with_zeros)
          assert_equal('2005/03/04 05:05:09.025', actual)
          
          milliseconds = 125
          pad_with_zeros = false
          actual = DurationFormatUtils.format(
            tokens,
            years, months, days, hours, minutes, seconds, milliseconds,
            pad_with_zeros)
          assert_equal('2005/3/4 5:5:9.125', actual)
        end
        
        
        def test_lexx
          actual_tokens = DurationFormatUtils.lexx('H:mm:ss.SSS')
          expected_tokens = [
            Token.new(DurationFormatUtils::TOKEN_H, 1),
            Token.new(':', 1),
            Token.new(DurationFormatUtils::TOKEN_m, 2),
            Token.new(':', 1),
            Token.new(DurationFormatUtils::TOKEN_s, 2),
            Token.new('.', 1),
            Token.new(DurationFormatUtils::TOKEN_S, 3)
          ]
          assert_equal(expected_tokens, actual_tokens)

          # Multibyte character format
          actual_tokens = DurationFormatUtils.lexx('H時間mm分ss.SSS秒')
          expected_tokens = [
            Token.new(DurationFormatUtils::TOKEN_H, 1),
            Token.new('時間', 1),
            Token.new(DurationFormatUtils::TOKEN_m, 2),
            Token.new('分', 1),
            Token.new(DurationFormatUtils::TOKEN_s, 2),
            Token.new('.', 1),
            Token.new(DurationFormatUtils::TOKEN_S, 3),
            Token.new('秒', 1)
          ]
          assert_equal(expected_tokens, actual_tokens)
        end
      end # DurationFormatUtilsTest
      
    end
  end
end
      
module Commons
  module Lang
    module Time
      
      class DurationFormatUtils
      
        class TokenTest < Test::Unit::TestCase
          def setup
          end
          
          
          def teardown
          end
        
          
          def test_contains_token_with_value?(tokens, value)
            tokens = [
              Token.new('val1'),
              Token.new('val2'),
              Token.new('val3')
            ]
            
            assert_true(Token.contains_token_with_value?(tokens, 'val1'))
            assert_true(Token.contains_token_with_value?(tokens, 'val2'))
            assert_true(Token.contains_token_with_value?(tokens, 'val3'))
            assert_false(Token.contains_token_with_value?(tokens, 'noentry'))
          end
        
        
          def test_initialize(value)
            expected_value = 'dummy'
            expected_count = 5
            
            actualToken = Token.new(expected_value, expected_count)
            assert_equal(expected_value, actualToken.value)
            assert_equal(expected_count, actualToken.count)
            
            actualToken = Token.new(expected_value)
            assert_equal(expected_value, actualToken.value)
            assert_equal(1, actualToken.count)
          end
          
          
          def test_increment
            expected_value = 'dummy'
            expected_count = 5
            
            actualToken = Token.new(expected_value, expected_count)
            actualToken.increment
            assert_equal(expected_value, actualToken.value)
            assert_equal(expected_count + 1, actualToken.count)
          end
          
          
          # test ==
          def test_equal_operator
            # no test. see eql? test.
          end
          
          
          def test_eql?
            token = Token.new('TEST')
            different = Object.new
            assert_false(token.eql?(different))
            
            # different count
            other = Token.new('TEST', 5)
            assert_false(token.eql?(other))
            
            # value's different class
            one = Token.new(Object.new, 2)
            other = Token.new(100, 2)
            assert_false(one.eql?(other))
            
            # value's same class
            value = Object.new
            one = Token.new(value, 2)
            other = Token.new(value, 2)
            assert_true(one.eql?(other))
          end
          
          
          def test_hash
            expected_value = 'dummy'
            actual_token = Token.new(expected_value)
            assert_equal(expected_value.hash, actual_token.value.hash)
          end
          
          
          def test_to_s
            actual_token = Token.new(DurationFormatUtils::TOKEN_y, 4)
            assert_equal('yyyy', actual_token.to_s)
          end
          
          
          def test_to_str
            actual_token = Token.new(DurationFormatUtils::TOKEN_y, 4)
            assert_equal('yyyy', actual_token.to_str)
          end
        end # TokenTest
        
      end # DurationFormatUtils
      
    end
  end
end
