// -*-c++-*-

/*!
  \file vector_2d.h
  \brief 2d vector class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_GEOM_VECTOR2D_H
#define RCSC_GEOM_VECTOR2D_H

#include <iostream>
#include <cmath>
#include <functional>

#include <rcsc/geom/angle_deg.h>

namespace rcsc {

//! 2D point vector class
class Vector2D {
    // : public boost:;addable< Vector2D >
    // , public boost::subtractable< Vector2D >
    // , public multipliable2< Vector2D, double >
    // , public dividable2< Vector2D, double >

public:

    //! constant error value for XY.
    static const double ERROR_VALUE;

    //! used only to create invalid Vector2D instance.
    enum InvalidType {
        INVALID
    };

    enum XYorPole {
        XY,
        POLE,
    };

    double x; //!< X-coordinate
    double y; //!< Y-coordinate

    //! default constructor.
    Vector2D()
        : x( 0.0 )
        , y( 0.0 )
      { }
    //! create Vector from XY value directly.
    Vector2D( const double & xx,
              const double & yy )
        : x( xx )
        , y( yy )
      { }
    //! create Vector from POLAR value.
    Vector2D( int,
              const double & radius,
              const AngleDeg & dir )
        : x( radius * dir.cos() )
        , y( radius * dir.sin() )
      { }
    //! create invalid Vector.
    explicit
    Vector2D( const InvalidType v )
        : x( ERROR_VALUE )
        , y( ERROR_VALUE )
      { }

    //! check if this is validate.
    inline
    bool valid() const
      {
          return ( ( x != ERROR_VALUE ) && ( y != ERROR_VALUE ) );
      }
    //! assign XY value directly.
    inline
    const
    Vector2D & assign( const double & xx,
                       const double & yy )
      {
          x = xx;
          y = yy;
          return *this;
      }
    //! assign XY value from POLAR value.
    inline
    const
    Vector2D & assign( int,
                       const double & radius,
                       const AngleDeg & dir )
      {
          x = radius * dir.cos();
          y = radius * dir.sin();
          return *this;
      }
    //! assign invalid value.
    inline
    const
    void assign( const InvalidType v )
      {
          x = y = ERROR_VALUE;
      }
    //! get the squared length of vector.
    inline
    double r2() const
      {
          return x * x + y * y;
      }
    //! get the length of vector.
    inline
    double r() const
      {
          return std::sqrt( r2() );
      }
    //! get the angle of vector.
    inline
    AngleDeg th() const
      {
          return AngleDeg( AngleDeg::atan2_deg( y, x ) );
      }
    //! get new Vector that XY are absolute value.
    inline
    Vector2D abs() const
      {
          return Vector2D( std::fabs( x ), std::fabs( y ) );
      }
    //! get absolute X value.
    inline
    double absX() const
      {
          return std::fabs( x );
      }
    //! get absolute Y value.
    inline
    double absY() const
      {
          return std::fabs( y );
      }

    //! add XY values
    inline
    const
    Vector2D & add( const double & xx,
                    const double & yy )
      {
          x += xx;
          y += yy;
          return *this;
      }

    //! return new Vector that XY are reversed.
    inline
    Vector2D operator-() const
      {
          return Vector2D( -x, -y );
      }
    //! operator +=
    inline
    const
    Vector2D & operator+=( const Vector2D & v )
      {
          x += v.x;
          y += v.y;
          return *this;
      }
    //! operator -=
    inline
    const
    Vector2D & operator-=( const Vector2D & v )
      {
          x -= v.x;
          y -= v.y;
          return *this;
      }
    //! multiplication by 'scalar' operator.
    inline
    const
    Vector2D & operator*=( const double & scalar )
      {
          x *= scalar;
          y *= scalar;
          return *this;
      }
    //! division by 'scalar' operator
    inline
    const
    Vector2D & operator/=( const double & scalar )
      {
          if ( scalar != 0 )
          {
              x /= scalar;
              y /= scalar;
          }
          return *this;
      }

    //! get the squared distance from this to 'p'.
    inline
    double dist2( const Vector2D & p ) const
      {
          //return ( Vector2D( *this ) -= p ).r2();
          return ( std::pow( this->x - p.x, 2.0 )
                   + std::pow( this->y - p.y, 2.0 ) );
      }
    //! get the distance from this to 'p'.
    inline
    double dist( const Vector2D & p ) const
      {
          return std::sqrt( dist2( p ) );
      }


    //! set the length of this Vector to 'len'.
    inline
    const
    Vector2D & normalize( const double & len = 1.0 )
      {
          double mag = this->r();
          if ( mag == 0 )
          {
              return *this;
          }
          return ( (*this) *= ( len / mag ) );
      }
    //! get new Vector that the length is set to 'len'.
    inline
    Vector2D normalizedVector( const double & len = 1.0 ) const
      {
          return Vector2D( *this ).normalize( len );
      }

    //! rotate this Vector with 'deg'.
    inline const
    Vector2D & rotate( const double & deg )
      {
          double radius = this->r();
          double rotated_angle = this->th().degree();
          rotated_angle += deg;
          rotated_angle *= AngleDeg::DEG2RAD;
          x = radius * std::cos( rotated_angle );
          y = radius * std::sin( rotated_angle );
          return *this;
      }
    //! rotate this Vector with 'angle'.
    inline const
    Vector2D & rotate( const AngleDeg & angle )
      {
          return rotate( angle.degree() );
      }
    //! get new Vector that the angle is rotated by 'deg'.
    inline
    Vector2D rotatedVector( const double & deg ) const
      {
          return Vector2D( *this ).rotate( deg );
      }
    //! get new Vector that the angle is rotated by 'angle'.
    inline
    Vector2D rotatedVector( const AngleDeg & angle ) const
      {
          return Vector2D( *this ).rotate( angle.degree() );
      }

    //! get inner(dot) product with 'v'.
    inline
    double innerProduct( const Vector2D & v ) const
      {
          return this->x * v.x + this->y * v.y;
          // ==  |this| * |v| * (*this - v).th().cos()
      }
    //! get virtal outer(cross) product with 'v'.
    inline
    double outerProduct( const Vector2D & v ) const
      {
          /*---------------------*
           * assume virtual 3D environment.
           * calculate Z-coordinate of outer product in right hand orientation.
           * For the time being, Input Vector's Z-coordinate is set to ZERO.
           *---------------------*/
          // Normal 3D outer product
          //   xn = this->y * v.z - this->z * v.y;
          //   yn = this->z * v.x - this->x * v.z;
          // # zn = this->x * v.y - this->y * v.x;
          return this->x * v.y - this->y * v.x;
          // == |this| * |v| * (*this - v).th().sin()
      }

    //////////////////////////////////////////////
    // static utility

    //! get new Vector created by POLAR value.
    inline
    static
    Vector2D polar2vector( const double & mag,
                           const AngleDeg & theta )
      {
          return Vector2D( mag * theta.cos(), mag * theta.sin() );
      }

    //! get inner(dot) product for v1 and v2.
    inline
    static
    double inner_product( const Vector2D & v1,
                          const Vector2D & v2 )
      {
          return v1.innerProduct( v2 );
      }
    //! get outer(cross) product for v1 and v2.
    inline
    static
    double outer_product( const Vector2D & v1,
                          const Vector2D & v2 )
      {
          return v1.outerProduct( v2 );
      }

    //////////////////////////////////////////////
    // stream utility

    //! output XY values.
    inline
    std::ostream & print( std::ostream & o ) const
      {
          o << "(" << x << ", " << y << ")";
          return o;
      }
    //! output rounded XY values.
    inline
    std::ostream & printRound( std::ostream & o,
                               const double & step = 0.1 ) const
      {
          o << "("  << rint( x / step ) * step
            << ", " << rint( y / step ) * step << ")";
          return o;
      }

    //////////////////////////////////////////////
    // functors for comparison

    //! comparison predicate for X value.
    class XCmp
        : public std::binary_function< Vector2D, Vector2D, bool > {
    public:
        //! functional operator
        result_type operator()( const first_argument_type & lhs,
                                const second_argument_type & rhs ) const
          {
              return lhs.x < rhs.x;
          }
    };

    //! comparison predicate for Y value.
    class YCmp
        : public std::binary_function< Vector2D, Vector2D, bool > {
    public:
        //! functional operator
        result_type operator()( const first_argument_type & lhs,
                                const second_argument_type & rhs ) const
          {
              return lhs.y < rhs.y;
          }
    };

    //! comparison predicate for absolute X value.
    class AbsXCmp
        : public std::binary_function< Vector2D, Vector2D, bool > {
    public:
        //! functional operator
        result_type operator()( const first_argument_type & lhs,
                                const second_argument_type & rhs ) const
          {
              return lhs.absX() < rhs.absX();
          }
    };

    //! comparison predicate for absolute Y value.
    class AbsYCmp
        : public std::binary_function< Vector2D, Vector2D, bool > {
    public:
        //! functional operator
        result_type operator()( const first_argument_type & lhs,
                                const second_argument_type & rhs ) const
          {
              return lhs.absY() < rhs.absY();
          }
    };


    //////////////////////////////////////////////
    // functor for region

    //! template predicate for 2D region sign detection.
    template < typename REGION >
    class IsWithin
        : public std::unary_function< Vector2D, bool > {
    private:
        const REGION M_region; //!< considered region.
    public:
        //! constructor
        explicit
        IsWithin( const REGION & region )
            : M_region( region )
          { }
        //! functional operator
        result_type operator()( const argument_type & position ) const
          {
              return M_region.contains( position );
          }
    };
};

} // end of namespace


////////////////////////////////////////////////////////
// arithmetic operators

//! operator add(T, T)
inline
const
rcsc::Vector2D
operator+( const rcsc::Vector2D & lhs, const rcsc::Vector2D & rhs )
{
    return rcsc::Vector2D( lhs ) += rhs;
}

//! operator sub(T, T)
inline
const
rcsc::Vector2D
operator-( const rcsc::Vector2D & lhs, const rcsc::Vector2D & rhs )
{
    return rcsc::Vector2D( lhs ) -= rhs;
}

//! operator mult(T, U)
inline
const
rcsc::Vector2D
operator*( const rcsc::Vector2D & lhs, const double & rhs )
{
    return rcsc::Vector2D( lhs ) *= rhs;
}

//! operator div(T, U)
inline
const
rcsc::Vector2D
operator/( const rcsc::Vector2D & lhs, const double & rhs )
{
    return rcsc::Vector2D( lhs ) /= rhs;
}


////////////////////////////////////////////////////////

//! stream operator
inline
std::ostream &
operator<<( std::ostream & o, const rcsc::Vector2D & v )
{
    return v.print( o );
}


#endif
