// -*-c++-*-

/*!
  \file gzfilterstream.h
  \brief gzip filtering stream Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa Akiyama

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_LIBZ
#include <zlib.h>
#endif

#include "gzfilterstream.h"

namespace rcsc {

struct gzfilterstreambuf_impl {
#ifdef HAVE_LIBZ
    z_stream * comp_stream_; //!< compressin buffer
    z_stream * decomp_stream_; //!< decompression buffer
#endif

    gzfilterstreambuf_impl()
#ifdef HAVE_LIBZ
        : comp_stream_( NULL )
        , decomp_stream_( NULL )
#endif
      { }
};

/*-------------------------------------------------------------------*/
/*!
  Default constructor creates an internal file buffer using boost::scoped_ptr.
  This buffer is deleted automatically.
  \param strm filtered stream buffer.
  \param bufsize allocated buffer size (default: 8192)
*/
gzfilterstreambuf::gzfilterstreambuf( std::streambuf & strm,
                                      std::size_t bufsize )
    : std::streambuf()
    , M_strmbuf( strm )
    , M_output_stream( NULL )
    , M_input_stream( NULL )
    , M_buf_size( bufsize )
    , M_read_buf( NULL )
    , M_input_buf( NULL )
    , M_output_buf( NULL )
    , M_write_buf( NULL )
    , M_impl( new gzfilterstreambuf_impl )
#ifdef HAVE_LIBZ
    , M_level( Z_DEFAULT_COMPRESSION )
#else
    , M_level( -1 )
#endif
{

}

/*-------------------------------------------------------------------*/
/*!
  Destructor flush buffered data and releases all allocated buffers.
*/
gzfilterstreambuf::~gzfilterstreambuf ()
{
#ifdef HAVE_LIBZ
    writeData( Z_FINISH );
#else
    writeData();
#endif

#ifdef HAVE_LIBZ
    if ( M_impl->comp_stream_ )
    {
        deflateEnd( M_impl->comp_stream_ );
    }
    if ( M_impl->decomp_stream_ )
    {
        deflateEnd( M_impl->decomp_stream_ );
    }
#endif

    delete M_output_stream;
    M_output_stream = NULL;
    delete M_input_stream;
    M_input_stream = NULL;
    delete [] M_read_buf;
    M_read_buf = NULL;
    delete [] M_input_buf;
    M_input_buf = NULL;
    delete [] M_output_buf;
    M_output_buf = NULL;
    delete [] M_write_buf;
    M_write_buf = NULL;
#ifdef HAVE_LIBZ
    delete M_impl->comp_stream_;
    M_impl->comp_stream_ = NULL;
    delete M_impl->decomp_stream_;
    M_impl->decomp_stream_ = NULL;
#endif

    this->setg( NULL, NULL, NULL );
    this->setp( NULL, NULL );
}

/*-------------------------------------------------------------------*/
/*!
  Compression strategy is set to Z_DEFAULT_STRATEGY automatically.
  \param level new compression level(-1,0-9)
  -1 means that data is handled without modification.
  \return true if level is validated value, else false.
*/
bool
gzfilterstreambuf::setLevel( const int level )
{
    bool ret = false;
#ifdef HAVE_LIBZ
    if ( M_impl->comp_stream_ )
    {
        if ( level == Z_DEFAULT_COMPRESSION
             || ( Z_NO_COMPRESSION <= level
                  && level <= Z_BEST_COMPRESSION )
             )
        {
            // make sure there is room for the deflate to flush
            this->sync();
            deflateParams( M_impl->comp_stream_,
                           level, Z_DEFAULT_STRATEGY );
            // write data flushed by deflateParams
            this->sync();
            M_level = level;
            ret = true;
        }
    }
    // Without zlib, this class cannot handle any compression level.
#endif
    return ret;
}

/*-------------------------------------------------------------------*/
/*!
  \param flush_type zlib flush type parameter. see deflate in zlib.h.
  \return true if successfully written, else false.
*/
bool
gzfilterstreambuf::writeData( int flush_type )
{
    // size of data to write
    int size = ( pptr() - pbase() ) * sizeof( char_type );
    if ( size == 0 )
    {
        return true;
    }

    if ( M_output_stream == NULL )
    {
        M_output_stream = new std::ostream( &M_strmbuf );
    }

#ifdef HAVE_LIBZ
    if ( M_level < 0 )
    {
        M_output_stream->write( M_output_buf, size );
    }
    else
    {
        if ( M_impl->comp_stream_ == NULL )
        {
            M_impl->comp_stream_ = new z_stream;
            M_impl->comp_stream_->zalloc = Z_NULL;
            M_impl->comp_stream_->zfree = Z_NULL;
            M_impl->comp_stream_->opaque = NULL;
            M_impl->comp_stream_->avail_in = 0;
            M_impl->comp_stream_->next_in = 0;
            M_impl->comp_stream_->next_out = 0;
            M_impl->comp_stream_->avail_out = 0;
            if ( deflateInit( M_impl->comp_stream_, M_level ) != Z_OK )
            {
                //std::cerr << "error in init\n";
                return false;
            }
            if ( M_write_buf == NULL )
            {
                M_write_buf = new char[ M_buf_size ];
            }
            M_impl->comp_stream_->next_out = (Bytef*)M_write_buf;
            M_impl->comp_stream_->avail_out = M_buf_size;
        }
        M_impl->comp_stream_->next_in = (Bytef*)M_output_buf;
        M_impl->comp_stream_->avail_in = size;

        do
        {
            int bytes_out = - M_impl->comp_stream_->total_out;
            int err = deflate( M_impl->comp_stream_, flush_type );
            if ( err != Z_OK && err != Z_STREAM_END )
            {
                //std::cerr << "error deflating\n";
                //std::cerr << comp_stream_->msg << std::endl;
                return false;
            }
            bytes_out += M_impl->comp_stream_->total_out;
            M_output_stream->write( M_write_buf, bytes_out );
            M_impl->comp_stream_->next_out = (Bytef*)M_write_buf;
            M_impl->comp_stream_->avail_out = M_buf_size;
        }
        while ( M_impl->comp_stream_->avail_in != 0 );
        // we want to keep writing until all the data has been
        // consumed by the compression stream
    }
#else
    M_output_stream->write( M_output_buf, size );
#endif

    if ( flush_type != NO_FLUSH ) // == Z_NO_FLUSH
    {
        // flush the underlying stream if a flush has been used
        M_output_stream->flush();
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  \param dest buffer to record the data.
  \param dest_size reference to the buffer size variable
  \return size of the read data
*/
int
gzfilterstreambuf::readData( char * dest,
                             int & dest_size )
{
    if( M_input_stream == NULL )
    {
        M_input_stream = new std::istream( &M_strmbuf );
    }

    int ret = 0;

    if ( ! M_input_stream->good() )
    {
        M_input_stream->setstate( std::istream::failbit );
        return ret;
    }

    // get number of characters available in input buffer
    int readable_bytes = M_strmbuf.in_avail();

    if ( readable_bytes < 0 )
    {
        // no more readable data
        M_input_stream->setstate( std::istream::eofbit );
    }
    else if ( readable_bytes == 0 )
    {
        // read new data from stream
        ret = M_input_stream->read( dest, 1 ).gcount();
        // set new size
        readable_bytes = M_strmbuf.in_avail();
        if ( readable_bytes > dest_size - 1 )
        {
            // readable size is larger than buffer size.
            // adjust readable data size.
            readable_bytes = dest_size - 1;
        }
        ret += M_input_stream->read( dest + 1, readable_bytes ).gcount();
        dest_size = readable_bytes + 1;
    }
    else
    {
        if ( readable_bytes > dest_size )
        {
            // readable size is larger than buffer size.
            // adjust readable data size.
            readable_bytes = dest_size;
        }
        ret = M_input_stream->read( dest, readable_bytes ).gcount();
        dest_size = readable_bytes;
    }
    return ret;
}

/*-------------------------------------------------------------------*/
/*!
  flush current internal buffer.
  This method is called from close(), sync() and overflow().
*/
gzfilterstreambuf::int_type
gzfilterstreambuf::overflow( int_type c )
{
    // if the buffer was not already allocated nor set by user,
    // do it just now
    if( pptr() == NULL )
    {
        M_output_buf = new char_type[ M_buf_size ];
    }
    else
    {
        if( ! writeData() )
        {
            return traits_type::eof();
        }
    }
    // reset putting buffer pointer
    this->setp( M_output_buf, M_output_buf + M_buf_size );
    if( c != traits_type::eof() )
    {
        sputc( c );
    }
    return 0;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
gzfilterstreambuf::sync()
{
    if ( pptr() != NULL )
    {
        // just flush the put area
        if ( ! writeData( SYNC_FLUSH ) ) // == Z_SYNC_FLUSH
        {
            return -1;
        }
        // reset putting buffer pointer
        this->setp( M_output_buf, M_output_buf + M_buf_size );
    }
    return 0;
}

/*-------------------------------------------------------------------*/
/*!
  This method is supposed to read some bytes from the I/O device
*/
gzfilterstreambuf::int_type
gzfilterstreambuf::underflow()
{
    static int s_remained = 0; // number of bytes remaining in M_output_buffer
    static char_type s_remained_char; // remained character in M_output_buffer

    // if the buffer was not already allocated nor set by user,
    // do it just now
    if( gptr() == NULL )
    {
        M_input_buf = new char_type[ M_buf_size ];
        this->setg( M_input_buf, M_input_buf, M_input_buf );
    }

    int readn = 0;
#ifdef HAVE_LIBZ
    if ( M_level < 0 )
    {
#endif
        if ( s_remained != 0 )
        {
            M_input_buf[ 0 ] = s_remained_char;
        }

        readn = M_buf_size * sizeof( char_type ) - s_remained;
        readData( M_input_buf + s_remained, readn );
        int totalbytes = readn + s_remained;
        this->setg( M_input_buf, M_input_buf,
                    M_input_buf + totalbytes / sizeof( char_type ) );

        s_remained = totalbytes % sizeof( char_type );
        if( s_remained != 0 )
        {
            s_remained_char = M_input_buf[ totalbytes / sizeof(char_type) ];
        }
#ifdef HAVE_LIBZ
    }
    else
    {
        if ( M_read_buf == NULL )
        {
            M_read_buf = new char_type[ M_buf_size ];
        }

        if ( s_remained != 0 )
        {
            M_input_buf[ 0 ] = s_remained_char;
        }

        if ( M_impl->decomp_stream_ == NULL )
        {
            M_impl->decomp_stream_ = new z_stream;
            M_impl->decomp_stream_->zalloc = Z_NULL;
            M_impl->decomp_stream_->zfree = Z_NULL;
            M_impl->decomp_stream_->opaque = NULL;
            M_impl->decomp_stream_->avail_in = 0;
            M_impl->decomp_stream_->next_in = 0;
            M_impl->decomp_stream_->avail_out = 0;
            M_impl->decomp_stream_->next_out = 0;
            if( inflateInit( M_impl->decomp_stream_ ) != Z_OK )
            {
                return false;
            }
        }
        M_impl->decomp_stream_->next_out = (Bytef*)( M_input_buf
                                                     + s_remained );
        M_impl->decomp_stream_->avail_out = ( M_buf_size
                                              * sizeof(char_type)
                                              - s_remained );
        do
        {
            if ( M_impl->decomp_stream_->avail_in == 0 )
            {
                int bytes_read = M_buf_size;
                readData( M_read_buf, bytes_read );

                M_impl->decomp_stream_->next_in = (Bytef*)M_read_buf;
                M_impl->decomp_stream_->avail_in = bytes_read;
            }

            readn -= M_impl->decomp_stream_->total_out;

            if ( inflate( M_impl->decomp_stream_, Z_NO_FLUSH ) != Z_OK )
            {
                //cerr << decomp_stream_->msg << endl;
                return traits_type::eof();
            }
            readn += M_impl->decomp_stream_->total_out;

        }
        while( readn == 0 );
        int totalbytes = readn + s_remained;
        this->setg( M_input_buf, M_input_buf,
                    M_input_buf + totalbytes / sizeof( char_type ) );

        s_remained = totalbytes % sizeof( char_type );
        if ( s_remained != 0 )
        {
            s_remained_char = M_input_buf[ totalbytes / sizeof( char_type ) ];
        }
    }
#endif
    // if (readn == 0 && remained_ != 0)
    // error - there is not enough bytes for completing
    // the last character before the end of the stream
    // - this can mean error on the remote end
    if( readn == 0 )
    {
        return traits_type::eof();
    }

    return sgetc();
}

///////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!
  \param strmbuf another stream buffer to filter
  \param buf_size allocated size of internal buffer
*/
gzfilterstream::gzfilterstream( std::streambuf & strmbuf,
                                std::size_t buf_size )
    : std::iostream( static_cast< std::streambuf* >( 0 ) )
    , M_filter_buf( strmbuf, buf_size )
{
    this->init( &M_filter_buf );
}

/*-------------------------------------------------------------------*/
/*!
  \param strm another stream to filter
  \param buf_size allocated size of internal buffer
*/
gzfilterstream::gzfilterstream( std::iostream & strm,
                                std::size_t buf_size )
    : std::iostream( static_cast< std::streambuf* >( 0 ) )
    , M_filter_buf( *(strm.rdbuf()), buf_size )

{
    this->init( &M_filter_buf );
}

///////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!
  construct with stream buffer
*/
gzifilterstream::gzifilterstream( std::streambuf & src,
                                  std::size_t buf_size )
    : std::istream( static_cast< std::streambuf* >( 0 ) )
    , M_filter_buf( src, buf_size )
{
    this->init( &M_filter_buf );
}

/*-------------------------------------------------------------------*/
/*!
  construct with stream
*/
gzifilterstream::gzifilterstream( std::istream & src,
                                  std::size_t buf_size )
    : std::istream( static_cast< std::streambuf* >( 0 ) )
    , M_filter_buf( *(src.rdbuf()), buf_size )
{
    this->init( &M_filter_buf );
}

///////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!
  construct with stream buffer
*/
gzofilterstream::gzofilterstream( std::streambuf & dest,
                                  std::size_t buf_size )
    : std::ostream( static_cast< std::streambuf* >( 0 ) )
    , M_filter_buf( dest, buf_size )
{
    this->init( &M_filter_buf );
}

/*-------------------------------------------------------------------*/
/*!
  construct with stream
*/
gzofilterstream::gzofilterstream( std::ostream & dest,
                                  std::size_t buf_size )
    : std::ostream( static_cast< std::streambuf* >( 0 ) )
    , M_filter_buf( *(dest.rdbuf()), buf_size )

{
    this->init( &M_filter_buf );
}

} // end namespace
