// -*-c++-*-

/*!
  \file gzfilterstream.h
  \brief gzip filtering stream Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa Akiyama

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCTOOLS_RCSC_GZ_FILTER_STREAM_H
#define RCTOOLS_RCSC_GZ_FILTER_STREAM_H

#include <iostream>
#include <boost/scoped_ptr.hpp>

namespace rcsc {

struct gzfilterstreambuf_impl;

/////////////////////////////////////////////////////////////////////

//! gzip filtering stream buffer class.
class gzfilterstreambuf
    : public std::streambuf {
private:
    //! filtered stream buffer
    std::streambuf & M_strmbuf;
    //! used for writing to M_strmbuf
    std::ostream * M_output_stream;
    //! used for reading from M_strmbuf
    std::istream * M_input_stream;
    //! size of the following buffers
    std::streamsize M_buf_size;
    //! used to read compressed data from M_strmbuf
    char_type * M_read_buf;
    //! used to buffer uncompressed input to this stream
    char_type * M_input_buf;
    //! used to buffer uncompressed output from this stream
    char_type * M_output_buf;
    //! used to write compressed data to M_strmbuf
    char_type * M_write_buf;

    //! Pimpl ideom.
    boost::scoped_ptr< gzfilterstreambuf_impl > M_impl;
    /*!
      current level of compression/decompression.
      -1 means that data is handled without modification.
    */
    int M_level;

    //! not used
    gzfilterstreambuf( const gzfilterstreambuf & );
    //! not used
    gzfilterstreambuf & operator=( const gzfilterstreambuf & );

public:
    //! constructor with another stream buffer.
    explicit
    gzfilterstreambuf( std::streambuf & strm,
                       std::size_t buf_size = 8192 );

    //! destructor.
    ~gzfilterstreambuf();

    //! sets new compression level.
    bool setLevel( const int level );

protected:
    enum FlushType {
        NO_FLUSH = 0,
        PARTIAL_FLUSH  = 1, // will be removed, use SYNC_FLUSH instead
        SYNC_FLUSH = 2,
        FULL_FLUSH = 3,
        FINISH = 4
    };


    //! write data to destination.
    bool writeData( int flush_type = NO_FLUSH );

    //! read data from destination.
    int readData( char * dest,
                  int & dest_size );

    //! overrided method
    virtual
    int_type overflow( int_type c );

    //! overrided method
    virtual
    int sync();

    //! overrided method
    virtual
    int_type underflow();
};

/////////////////////////////////////////////////////////////////////

//! gzip filtering stream class.
class gzfilterstream
    : public std::iostream {
private:
    //! buffer for this steram
    gzfilterstreambuf M_filter_buf;

    //! not used
    gzfilterstream( const gzfilterstream & );
    //! not used
    gzfilterstream & operator=( const gzfilterstream & );

public:
    //! constructor with another stream buffer
    explicit
    gzfilterstream( std::streambuf & strmbuf,
                    std::size_t buf_size = 8192 );
    //! constructor with another stream
    explicit
    gzfilterstream( std::iostream & strm,
                    std::size_t buf_size = 8192 );
};

/////////////////////////////////////////////////////////////////////

//! gzip filtering input stream class.
class gzifilterstream
    : public std::istream {
private:
    //! buffer for this steram
    gzfilterstreambuf M_filter_buf;

    //! not used
    gzifilterstream( const gzifilterstream & );
    //! not used
    gzifilterstream & operator=( const gzifilterstream & );

public:
    //! constructor with another stream buffer
    explicit
    gzifilterstream( std::streambuf & src,
                     std::size_t buf_size = 8192 );
    //! constructor with another stream
    explicit
    gzifilterstream( std::istream & src,
                     std::size_t buf_size = 8192 );

};

/////////////////////////////////////////////////////////////////////

//! gzip filtering output stream class.
class gzofilterstream
    : public std::ostream {
private:
    //! buffer for this steram
    gzfilterstreambuf M_filter_buf;

    //! not used.
    gzofilterstream( const gzofilterstream & );
    //! not used.
    gzofilterstream & operator=( const gzofilterstream & );
public:
    //! constructor with another stream buffer
    explicit
    gzofilterstream( std::streambuf & dest,
                     std::size_t buf_size = 8192 );
    //! constructor with another stream
    explicit
    gzofilterstream( std::ostream & dest,
                     std::size_t buf_size = 8192 );
};

} // end namespace

#endif
